#ifndef EPT_CACHE_DEBTAGS_UPDATE_H
#define EPT_CACHE_DEBTAGS_UPDATE_H

#include <ept/forward.h>
#include <ept/path.h>
#include <string>
#include <vector>
#include <set>

#include <dirent.h>		// opendir, closedir

namespace ept {
namespace t {
namespace cache {
namespace debtags {

class VocabularyMerger;

// Nicely wrap access to directories
class Directory
{
	std::string m_path;

public:
	class const_iterator
	{
		DIR* dir;
		struct dirent* d;

	public:
		// Create an end iterator
		const_iterator() : dir(0), d(0) {}
		// Create a begin iterator
		const_iterator(DIR* dir) : dir(dir), d(0) { ++(*this); }
		// Cleanup properly
		~const_iterator() { if (dir) closedir(dir); }

		// auto_ptr style copy semantics
		const_iterator(const const_iterator& i)
		{
			dir = i.dir;
			d = i.d;
			const_iterator* wi = const_cast<const_iterator*>(&i);
			wi->dir = 0;
			wi->d = 0;
		}
		const_iterator& operator=(const const_iterator& i)
		{
			// Catch a = a
			if (&i == this) return *this;
			if (dir) closedir(dir);
			dir = i.dir;
			d = i.d;
			const_iterator* wi = const_cast<const_iterator*>(&i);
			wi->dir = 0;
			wi->d = 0;
		}

		const_iterator& operator++()
		{
			if ((d = readdir(dir)) == 0)
			{
				closedir(dir);
				dir = 0;
			}
			return *this;
		}

		struct dirent& operator*() { return *d; }
		struct dirent* operator->() { return d; }

		bool operator==(const const_iterator& iter) const
		{
			return dir == iter.dir && d == iter.d;
		}
		bool operator!=(const const_iterator& iter) const
		{
			return dir != iter.dir || d != iter.d;
		}
	};

	Directory(const std::string& path) : m_path(path) {}

	const std::string& path() const { return m_path; }
	bool valid();

	const_iterator begin();
	const_iterator end() const;
};


class SourceDir : public Directory
{
protected:
	enum FileType { SKIP, TAG, VOC, TAGGZ, VOCGZ };

	// Check if a file name is a tag file, a vocabulary file or a file to skip.
	// Please notice that it works on file names, not paths.
	FileType fileType(const std::string& name);

public:
	SourceDir(const std::string& path) : Directory(path) {}

	// Return the time of the newest file in the source directory
	time_t timestamp();

	// Return the time of the newest vocabulary file in the source directory
	time_t vocTimestamp();

	// Return the time of the newest tag file in the source directory
	time_t tagTimestamp();

	// Read the tag files in the directory and output their content to out
	template<typename OUT>
	void readTags(OUT out);

	// Read the vocabulary files in the directory and output their content to
	// out
	void readVocabularies(VocabularyMerger& out);

#if 0
	// Return the the main source directory
	static SourceDir main();

	// Return the the user's source directory
	static SourceDir user();
#endif
};

template<typename P = Path>
struct IndexManager
{
	struct Vocabulary
	{
		SourceDir mainSource;
		SourceDir userSource;
		time_t ts_main_src;
		time_t ts_user_src;
		time_t ts_main_voc;
		time_t ts_main_idx;
		time_t ts_user_voc;
		time_t ts_user_idx;

		time_t sourceTimestamp() const { return ts_main_src < ts_user_src ? ts_user_src : ts_main_src; }
		bool needsRebuild() const;
		void rebuild(const std::string& vocfname, const std::string& idxfname);
		bool rebuildIfNeeded();
		bool getUpToDateVocabulary(std::string& vocfname, std::string& idxfname);

		bool userIndexIsRedundant() const;
		bool deleteRedundantUserIndex();

		void rescan();

		Vocabulary();
	};

	template<typename C>
	struct Tagdb
	{
		typename C::Aggregator& agg;
		SourceDir mainSource;
		SourceDir userSource;
		time_t ts_pkgidx;
		time_t ts_main_src;
		time_t ts_user_src;
		time_t ts_main_tag;
		time_t ts_main_idx;
		time_t ts_user_tag;
		time_t ts_user_idx;

		time_t sourceTimestamp() const
		{
			time_t res = ts_pkgidx;
			if (ts_main_src > res) res = ts_main_src;
			if (ts_user_src > res) res = ts_user_src;
			return res;
		}
		bool needsRebuild() const;
		void rebuild(const std::string& tagfname, const std::string& idxfname);
		bool rebuildIfNeeded();
		bool getUpToDateTagdb(std::string& tagfname, std::string& idxfname);

		bool userIndexIsRedundant() const;
		bool deleteRedundantUserIndex();

		void rescan();

		Tagdb(typename C::Aggregator& agg);
	};

	template<typename C>
	struct Pkgidx
	{
		typename C::Aggregator& agg;
		time_t ts_pkgs;
		time_t ts_main_idx;
		time_t ts_user_idx;

		time_t sourceTimestamp() const { return ts_pkgs; }
		bool needsRebuild() const;
		void rebuild(const std::string& idxfname);
		bool rebuildIfNeeded();
		bool getUpToDatePkgidx(std::string& idxfname);

		bool userIndexIsRedundant() const;
		bool deleteRedundantUserIndex();

		void rescan();

		Pkgidx(typename C::Aggregator& agg);
	};

	static void obtainWorkingVocabulary(std::string& vocfname, std::string& idxfname);

	template<typename C>
	static void obtainWorkingTagdb(typename C::Aggregator& ag, std::string& tagfname, std::string& idxfname);

	template<typename C>
	static void obtainWorkingPkgidx(typename C::Aggregator& ag, std::string& idxfname);
};


}
}
}
}

// vim:set ts=4 sw=4:
#endif
