#ifndef CACHE_COMPONENT_TAGS_H
#define CACHE_COMPONENT_TAGS_H

/** @file
 * @author Enrico Zini <enrico@enricozini.org>
 * Tag vocabulary access
 */

/*
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <apt-front/cache/component/base.h>
#include <apt-front/cache/entity/tag.h>
#include <tagcoll/MMapIndex.h>

#include <string>
#include <vector>
#include <map>

namespace aptFront {
namespace cache {
namespace component {

class Tags : public Implementation<Tags>
{
public:
	class FacetIndex : public Tagcoll::MMapIndex
	{
	protected:
		// Layout of the data in the index
		struct Item {
			int offset;
			int size;
			int firsttag;
			int lasttag;
			const char name[];
		};
		inline Item* item(int id) const
		{
			if (id >= 0 && (unsigned)id < size())
				return (Item*)(m_buf + ((int*)m_buf)[id]);
			return NULL;
		}

	public:
		FacetIndex() : Tagcoll::MMapIndex() {}
		FacetIndex(const Tagcoll::MasterMMapIndex& master, size_t idx)
			: Tagcoll::MMapIndex(master, idx) {}

		/// Get the number of facets in the index
		size_t size() const { return m_size == 0 ? 0 :  *(int*)m_buf / sizeof(int); }
		/// Get the offset of the facet data in the vocabulary for this facet
		size_t offset(int id) const { Item* i = item(id); return i == NULL ? 0 : i->offset; }
		/// Get the size of the facet data in the vocabulary for this facet
		size_t size(int id) const { Item* i = item(id); return i == NULL ? 0 : i->size; }
		/// Get the id of the first tag for this facet
		int firsttag(int id) const { Item* i = item(id); return i == NULL ? -1 : i->firsttag; }
		/// Get the id of the last tag for this facet
		int lasttag(int id) const { Item* i = item(id); return i == NULL ? -1 : i->lasttag; }
		/// Get the name of this facet
		const char* name(int id) const { Item* i = item(id); return i == NULL ? "" : i->name; }
		/// Get the ID of the facet with this name
		int id(const char* name) const;
		int id(const std::string& name) const { return id(name.c_str()); }
	};
	
	class TagIndex : public Tagcoll::MMapIndex
	{
	protected:
		// Layout of the data in the index
		struct Item {
			int offset;
			int size;
			int facet;
			const char name[];
		};
		inline Item* item(int id) const
		{
			if (id >= 0 && (unsigned)id < size())
				return (Item*)(m_buf + ((int*)m_buf)[id]);
			return NULL;
		}

	public:
		TagIndex() : Tagcoll::MMapIndex() {}
		TagIndex(const Tagcoll::MasterMMapIndex& master, size_t idx)
			: Tagcoll::MMapIndex(master, idx) {}

		/// Get the number of tags in the index
		size_t size() const { return m_size == 0 ? 0 : *(int*)m_buf / sizeof(int); }
		/// Get the offset of the facet data in the vocabulary for this tag
		size_t offset(int id) const { Item* i = item(id); return i == NULL ? 0 : i->offset; }
		/// Get the size of the facet data in the vocabulary for this tag
		size_t size(int id) const { Item* i = item(id); return i == NULL ? 0 : i->size; }
		/// Get the id of the facet for this tag
		int facet(int id) const { Item* i = item(id); return i == NULL ? -1 : i->facet; }
		/// Get the name of this tag
		const char* name(int id) const { Item* i = item(id); return i == NULL ? "" : i->name; }
		/// Get the ID of the tag with this name
		int id(const char* name) const;
		int id(const std::string& name) const { return id(name.c_str()); }
	};

protected:
	// Master MMap index container
	Tagcoll::MasterMMapIndex mastermmap;

	time_t m_timestamp;

	// Mmapped vocabulary file
	std::string voc_fname;
	int voc_fd;
	size_t voc_size;
	const char* voc_buf;
	
	// Facet and tag indexes
	FacetIndex findex;
	TagIndex tindex;
	
	// Cached parsed facet and tag records
	mutable std::vector< std::map<std::string, std::string> > m_facetData;
	mutable std::vector< std::map<std::string, std::string> > m_tagData;
	// Empty parsed data to return when data is asked for IDs == -1
	std::map<std::string, std::string> emptyData;

	void parseVocBuf(std::map<std::string, std::string>& res, size_t ofs, size_t len) const;

public:
	Tags();
	~Tags();

	void init(Cache& c);

	time_t timestamp() const { return m_timestamp; }

	const FacetIndex& facetIndex() const { return findex; }
	const TagIndex& tagIndex() const { return tindex; }

	/**
	 * Check if the vocabulary contains the facet `name'
	 */
	bool hasFacet(const std::string& name) const
	{
		return findex.id(name.c_str()) != -1;
	}

	/**
	 * Check if the vocabulary contains the tag `fullname'
	 */
	bool hasTag(const std::string& fullname) const
	{
		return tindex.id(fullname.c_str()) != -1;
	}

	/**
	 * Return the facet with the given name
	 */
	entity::Facet facetByID(int id) const;

	/**
	 * Return the tag with the given full name
	 */
	entity::Tag tagByID(int id) const;

	/**
	 * Return the facet for the tag with the given ID
	 */
	entity::Facet facetByTag(int id) const { return facetByID(tindex.facet(id)); }

	/**
	 * Return the facet with the given name
	 */
	entity::Facet facetByName(const std::string& name) const { return facetByID(findex.id(name)); }

	/**
	 * Return the tag with the given full name
	 */
	entity::Tag tagByName(const std::string& fullname) const { return tagByID(tindex.id(fullname)); }

	/**
	 * Return all the facets in the vocabulary
	 */
	Tagcoll::OpSet<entity::Facet> facets() const
	{
		Tagcoll::OpSet<entity::Facet> res;
		for (size_t i = 0; i < findex.size(); i++)
			res += facetByID(i);
		return res;
	}

	/**
	 * Return all the tags in the vocabulary
	 */
	Tagcoll::OpSet<entity::Tag> tags() const
	{
		Tagcoll::OpSet<entity::Tag> res;
		for (size_t i = 0; i < tindex.size(); i++)
			res += tagByID(i);
		return res;
	}

	/**
	 * Return the tags in the given facet
	 */
	Tagcoll::OpSet<entity::Tag> tags(int facet) const
	{
		Tagcoll::OpSet<entity::Tag> res;
		for (int i = findex.firsttag(facet); i < findex.lasttag(facet); i++)
			res += tagByID(i);
		return res;
	}

	Tagcoll::OpSet<entity::Tag> tags(const std::string& facetName) const
	{
		return tags(findex.id(facetName));
	}

	Tagcoll::OpSet<entity::Tag> tags(const entity::Facet& facet) const
	{
		return tags(facet.id());
	}

#if 0
	/// Get the DerivedTagList with the Equates: expressions read from the vocabulary
	const DerivedTagList& getEquations() const throw () { return equations; }
	
	/// Get a set with all the facets present in the vocabulary that are matched by `filter'
	FacetSet facets(const FacetMatcher& filter) const throw () { return getFiltered(filter); }
#endif

#if 0
	// These functions are here just to be used by Facet and Tag.  I'm not
	// making them private because I don't want Facet and Tag to access other
	// Tags member, and other classes can't use these anyway as Facet::Data and
	// Tag::Data are protected
	const entity::Facet::Data& facetData(int idx) { return m_facets[idx]; }
	const entity::Tag::Data& tagData(int idx) { return m_tags[idx]; }
#endif

	/// Get the facet name given the facet id
	std::string facetName(int id) const { return findex.name(id); }

	/// Get the tag name given the tag id
	std::string tagName(int id) const { return tindex.name(id); }

	/// Get the tag name given the tag id
	std::string tagShortName(int id) const;

	const std::map<std::string, std::string>& facetData(int id) const;
	const std::map<std::string, std::string>& tagData(int id) const;
	
	static std::string componentName();
	static time_t computeTimestamp();
};

}
}
}

// vim:set ts=4 sw=4:
#endif
