#include <apt-front/cache/component/debtags/update.h>
#include <apt-front/cache/component/debtags/pkgidx.h>
#include <apt-front/cache/entity/package.h>
#include <apt-front/cache/entity/tag.h>
#include <apt-front/utils/vocabularymerger.h>
#include <apt-front/utils/zlibparserinput.h>
#include <apt-front/utils/paths.h>

#include <tagcoll/Filter.h>
#include <tagcoll/TextFormat.h>
#include <tagcoll/StdioParserInput.h>
#include <tagcoll/StringParserInput.h>
#include <tagcoll/InputMerger.h>

#include <apt-pkg/acquire.h>
#include <apt-pkg/acquire-item.h>
#include <apt-pkg/strutl.h>
#include <apt-pkg/error.h>
#include <apt-pkg/configuration.h>

// TODO: see if we really need to invoke pkgInitConfig or if we can delegate
// that to the rest of aptFront.  Maybe reinstate the old APTFilter
#include <apt-pkg/init.h>

// FIXME: output got ripped out, clients (and this) will need
// adjusting

#include <sys/types.h>	// chmod
#include <sys/stat.h>	// chmod
#include <dirent.h>		// opendir, closedir
#include <errno.h>

using namespace std;
using namespace Tagcoll;

namespace aptFront {
namespace cache {
namespace component {
namespace debtags {

bool source::canBeUsed() const
{
	string file;

	if (!(file = tagFile()).empty())
		if (!utils::Path::access(file, R_OK))
			return false;

	if (!(file = vocFile()).empty())
		if (!utils::Path::access(file, R_OK))
			if (!utils::Path::access(utils::Path::defaultVocabulary(), R_OK))
				return false;

	return true;
}

std::string source::tagFile() const
{
	if (uri.find("file://") == 0)
		return uri.substr(7) + "tags-current.gz";
	else if (uri == "apt://")
		return string();
	else
		return utils::Path::downloadcache() + "/" + URItoFileName(uri + "tags-current.gz");
}

std::string source::vocFile() const
{
	if (uri.find("file://") == 0)
		return uri.substr(7) + "vocabulary.gz";
	else if (uri == "apt://")
		return string();
	else
		return utils::Path::downloadcache() + "/" + URItoFileName(uri + "vocabulary.gz");
}

time_t source::timestamp() const
{
	time_t res = 0;
	time_t cand;

	string tagfile = tagFile();
	if (tagfile.empty())
		return Packages::computeTimestamp();
	else
		if ((cand = utils::Path::timestamp(tagfile)) > res)
			res = cand;

	string vocfile = vocFile();
	if (!vocfile.empty() &&
			(cand = utils::Path::timestamp(vocfile)) > res)
		res = cand;

	return res;
}

time_t source::timestamp(Cache& c) const
{
	time_t res = 0;
	time_t cand;

	string tagfile = tagFile();
	if (tagfile.empty())
		return c.packages().timestamp();
	else
		if ((cand = utils::Path::timestamp(tagfile)) > res)
			res = cand;

	string vocfile = vocFile();
	if (!vocfile.empty() &&
			(cand = utils::Path::timestamp(vocfile)) > res)
		res = cand;

	return res;
}

// TagcollFilter that removes tags not in the given Vocabulary
class VocabularyFilter : public Tagcoll::Filter<string, string>
{
protected:
	set<string> whitelist;

	virtual void consumeItemUntagged(const std::string& item);
	virtual void consumeItem(const std::string& item, const OpSet<std::string>& tags);
	virtual void consumeItemsUntagged(const OpSet<string>& items);
	virtual void consumeItems(const OpSet<string>& items, const OpSet<string>& tags);

public:
	VocabularyFilter(utils::VocabularyMerger& voc)
		: whitelist(voc.tagNames()) {}
	VocabularyFilter(utils::VocabularyMerger& voc, Consumer<string, string>& cons)
		: Filter<string, string>(cons), whitelist(voc.tagNames()) {}
};

void VocabularyFilter::consumeItemUntagged(const string& item)
{
	consumer->consume(item);
}
void VocabularyFilter::consumeItemsUntagged(const OpSet<string>& items)
{
	consumer->consume(items);
}

void VocabularyFilter::consumeItem(const string& item, const OpSet<string>& tags)
{
	OpSet<string> patched;
	for (OpSet<string>::const_iterator i = tags.begin();
			i != tags.end(); i++)
		if (whitelist.find(*i) != whitelist.end() && *i != "special::invalid-tag")
			patched += *i;

	if (patched.size())
		consumer->consume(item, patched);
	else
		consumer->consume(item);
}

// Process a set of items identically tagged, with their tags
void VocabularyFilter::consumeItems(const OpSet<string>& items, const OpSet<string>& tags)
{
	OpSet<string> patched;
	for (OpSet<string>::const_iterator i = tags.begin();
			i != tags.end(); i++)
		if (whitelist.find(*i) != whitelist.end() && *i != "special::invalid-tag")
			patched += *i;

	if (patched.size())
		consumer->consume(items, patched);
	else
		consumer->consume(items);
}


vector<source> readSources()
    // TODO: remove in favor of apt-pkg sources.list parser?
{
	FILE* in = fopen(utils::Path::debtagssources().c_str(), "rt");
	if (!in)
		return vector <source>();
		// throw FileException(errno, string("reading ") + fn_sources);

	vector<source> res;

	string line;
	int c;
	while ((c = fgetc(in)) != EOF)
	{
		if (c != '\n')
			line += c;
		else
		{
			unsigned int i = 0;

			// Skip leading spaces
			while (i < line.size() && isspace(line[i]))
				i++;

			// If it's a tag source
			if (line.substr(i, 4) == "tags")
			{
				i += 4;
				// Skip further spaces
				while (i < line.size() && isspace(line[i]))
					i++;
				res.push_back(source(source::TAGS, line.substr(i)));
			}
			
			/*
			// If it's a debtags source
			if (line.substr(i, 4) == "debtags")
			{
				i += 7;
				// Skip further spaces
				while (i < line.size() && isspace(line[i]))
					i++;
				res.push_back(source(source::DEBTAGS, line.substr(i)));
			}
			*/
			
			line = string();
		}
	}

	fclose(in);

	return res;
}

bool dbNeedsReindex(Cache& c)
{
	time_t tsdbidx = utils::Path::timestamp(utils::Path::tagdbIndex());
	if (tsdbidx == 0)
		return true;

	if (utils::Path::timestamp(utils::Path::defaultVocabulary()) > tsdbidx)
		return true;

	vector<debtags::source> sources = debtags::readSources();
	for (vector<debtags::source>::const_iterator i = sources.begin();
			i != sources.end(); i++)
		if (i->timestamp(c) > tsdbidx)
			return true;

	return false;
}

bool vocNeedsReindex(Cache& c)
{
	time_t tsidx = utils::Path::timestamp(utils::Path::vocabulary());
	if (tsidx == 0)
		return true;
	time_t tsidx1 = utils::Path::timestamp(utils::Path::vocabularyIndex());
	if (tsidx1 == 0)
		return true;
	if (tsidx1 > tsidx)
		tsidx = tsidx1;

	if (utils::Path::timestamp(utils::Path::defaultVocabulary()) > tsidx)
		return true;

	vector<debtags::source> sources = debtags::readSources();
	for (vector<debtags::source>::const_iterator i = sources.begin();
			i != sources.end(); i++)
		if (i->timestamp(c) > tsidx)
			return true;

	return false;
}

int readTags(Cache& c,
		Consumer<entity::Package, entity::Tag>& index,
		const source& source)
{
	try {
		if (source.uri == "apt://")
		{
			component::debtags::TagStringConverter& conv = c.tagstringconverter();

			bool found = false;
			component::Packages& p = c.packages();
			component::Records& r = c.records();
			for (component::Packages::iterator i = p.packagesBegin();
					i != p.packagesEnd(); ++i)
			{
				if (!i->valid())
					continue;
				if (!i->hasVersion())
					continue;
				OpSet<entity::Tag> tags;
				for (utils::Range< entity::Version > v = i->versions(); v != v.end(); ++v)
				{
					if (!v->valid())
						continue;

					// Get the tag data from the Tag: header
					string tagData = r.aptTagData(*v);
					if (tagData.empty())
						continue;
					
					// Try parsing it if it's not empty
					tags = conv.parseTagList(tagData);

					break;
				}
				// In case of empty tagset, add the special::not-yet-tagged
				// tags
				if (tags.empty())
				{
					tags += conv("special::not-yet-tagged");
					tags += conv("special::not-yet-tagged::" + i->name()[0]);
				}
				index.consume(*i, tags);
				found = true;
			}
			if (found)
				return 1;
		} else {
			string tagfile;
			if (source.uri.find("file://") == 0)
			{
				tagfile = source.uri.substr(7) + "tags-current.gz";
			} else {
				tagfile = utils::Path::downloadcache();
				tagfile += "/" + URItoFileName(source.uri + "tags-current.gz");
			}
			if (access(tagfile.c_str(), R_OK) == -1)
				return 0;

			string vocfile;
			if (source.uri.find("file://") == 0)
			{
				vocfile = source.uri.substr(7) + "vocabulary.gz";
			} else {
				vocfile = utils::Path::downloadcache();
				vocfile += "/" + URItoFileName(source.uri + "vocabulary.gz");
			}
			if (access(vocfile.c_str(), R_OK) == -1)
			{
				// Import the tag file

				// Read compressed data
				utils::ZlibParserInput in(tagfile);

				// Read the collection
				TextFormat<entity::Package, entity::Tag>::parse(
						c.packagestringconverter(),
						c.tagstringconverter(),
						in, index);
			} else {
				// Import the tag file, filtered using the vocabulary file

				// Convert strings into entities
				ConversionFilter<string, string, entity::Package, entity::Tag> conv(
						c.packagestringconverter(),
						c.tagstringconverter(),
						index);
				
				// Filter out tags not in this source's vocabulary
				utils::ZlibParserInput vocin(vocfile);
				utils::VocabularyMerger filterVoc;
				filterVoc.read(vocin);
				VocabularyFilter filter(filterVoc, conv);

				// Read the compressed collection
				utils::ZlibParserInput in(tagfile);
				TrivialConverter<string, string> tconv;
				TextFormat<string, string>::parse(tconv, tconv, in, filter);
			}

			return 1;
		}
	} catch (Exception& e) {
		// fprintf(stderr, "%s: %.*s.  Ignoring source %.*s\n", e.type(), PFSTR(e.desc()), PFSTR(i->uri));
	}
	return 0;
}

int readBasicVocabularies(utils::VocabularyMerger& systemvoc)
{
	try {
		StdioParserInput mainInput(utils::Path::defaultVocabulary());
		systemvoc.read(mainInput);
		return 1;
	} catch (Exception& e) {
		// fprintf(stderr, "%s: %.*s.  Ignoring source %.*s\n", e.type(), PFSTR(e.desc()), PFSTR(i->uri));
	}
	
#if 0
	DIR* dir = opendir(path_tagvoc_d);
	if (!dir)
		throw SystemException(errno, string("reading directory ") + path_tagvoc_d);

	while (struct dirent* d = readdir(dir))
	{
		if (d->d_name[0] != '.' &&
				d->d_name[strlen(d->d_name) - 1] != '~')
		{
			string fn = string(path_tagvoc_d) + '/' + d->d_name;
			if (access(fn.c_str(), R_OK) == 0)
			{
				StdioParserInput patchInput(fn);
				systemvoc.read(patchInput);
			}
		}
	}
	closedir(dir);
#endif

	return 0;
}

int readVocabulary(utils::VocabularyMerger& merger, const source& source)
{
	try {
		string localVoc;
		if (source.uri.find("file://") == 0)
		{
			localVoc = source.uri.substr(7) + "vocabulary.gz";
		} else {
			localVoc = utils::Path::downloadcache();
			localVoc += "/" + URItoFileName(source.uri + "vocabulary.gz");
		}
		if (access(localVoc.c_str(), R_OK) != -1)
		{
			utils::ZlibParserInput in(localVoc);
			merger.read(in);
			return 1;
		}
	} catch (Exception& e) {
		// fprintf(stderr, "%s: %.*s.  Ignoring source %.*s\n", e.type(), PFSTR(e.desc()), PFSTR(i->uri));
	}
	return 0;
}

}
}
}
}

using namespace aptFront;
using namespace aptFront::cache;

// Item class for index files
class AcqTagfile: public pkgAcquire::Item
{
protected:
	pkgAcquire::ItemDesc Desc;
	string RealURI;

public:

	// Specialized action members
	virtual void Done(string Message,unsigned long Size,string Md5Hash,
			pkgAcquire::MethodConfig *Cnf);
	virtual string DescURI() {return RealURI + ".gz";};

	AcqTagfile(pkgAcquire *Owner,string URI,string URIDesc,
			string ShortDesct);
};

// AcqTagfile::AcqTagfile - Constructor                                     
// ---------------------------------------------------------------------
/* The package file is added to the queue and a second class is
   instantiated to fetch the revision file */
AcqTagfile::AcqTagfile(pkgAcquire *Owner,
                         string URI,string URIDesc,string ShortDesc) :
	Item(Owner), RealURI(URI)
{
	//fprintf(stderr, "AcqTagfile::AcqTagfile ('%.*s') called\n", PFSTR(URI));

	DestFile = utils::Path::downloadcache() + "/partial/"; //_config->FindDir("Dir::State::lists") + "partial/";
	DestFile += URItoFileName(URI);
	//fprintf(stderr, "DestFile: %.*s\n", PFSTR(DestFile));

	// Create the item
	Desc.URI = URI;
	Desc.Description = URIDesc;
	Desc.Owner = this;
	Desc.ShortDesc = ShortDesc;

	QueueURI(Desc);
}
                                                                        
// AcqTagfile::Done - Finished a fetch                                 
// ---------------------------------------------------------------------
/* This goes through a number of states.. On the initial fetch the
   method could possibly return an alternate filename which points
   to the uncompressed version of the file. If this is so the file
   is copied into the partial directory. In all other cases the file
   is decompressed with a gzip uri. */
void AcqTagfile::Done(string Message,unsigned long Size,string MD5,
                       pkgAcquire::MethodConfig *Cfg)
{
	Item::Done(Message,Size,MD5,Cfg);
	//fprintf(stderr, "AcqTagfile::Done (%.*s) called.\n", PFSTR(Message));

    /* if (MD5.empty())
    {
       MD5Summation sum;
       FileFd Fd(DestFile, FileFd::ReadOnly);
       sum.AddFD(Fd.Fd(), Fd.Size());
       Fd.Close();
       MD5 = (string)sum.Result();
    } */

	// Done, move it into position
	//string FinalFile = _config->FindDir("Dir::State::lists") + "tags/";
	string FinalFile = utils::Path::downloadcache();
	FinalFile += "/" + URItoFileName(RealURI);
	//fprintf(stderr, "FinalFile: %.*s\n", PFSTR(FinalFile));
	// TODO: replace this by a properly checked rename
	//Rename(DestFile,FinalFile);
	if (access(DestFile.c_str(), R_OK) != -1)
	{
//		fprintf(stderr, "%.*s -> %.*s\n",
//				PFSTR(DestFile), PFSTR(FinalFile));
		if (rename(DestFile.c_str(), FinalFile.c_str()) == -1)
			throw FileException(errno, "renaming " + DestFile + " to " + FinalFile);
		chmod(FinalFile.c_str(),0644);
	} else {
		// fprintf(stderr, "Skipping rename of %.*s\n", PFSTR(DestFile));
    }

	/* We restore the original name to DestFile so that the clean operation
	   will work OK */
	//DestFile = _config->FindDir("Dir::State::lists") + "partial/";
	//DestFile += URItoFileName(RealURI);
	//fprintf(stderr, "DestFile: %.*s\n", PFSTR(DestFile));

	// Remove the compressed version.
	//if (erase == true)
		//unlink(DestFile.c_str());
	return;

/*
   erase = false;
   Complete = true;

   // Handle the unzipd case
   string FileName = LookupTag(Message,"Alt-Filename");
   if (FileName.empty() == false)
   {
      // The files timestamp matches
      if (StringToBool(LookupTag(Message,"Alt-IMS-Hit"),false) == true)
         return;

      decompression = true;
      Local = true;
      DestFile += ".decomp";
      Desc.URI = "copy:" + FileName;
      QueueURI(Desc);
      Mode = "copy";
      return;
   }
   FileName = LookupTag(Message,"Filename");
   if (FileName.empty() == true)
   {
      Status = StatError;
      ErrorText = "Method gave a blank filename";
   }

   // The files timestamp matches
   if (StringToBool(LookupTag(Message,"IMS-Hit"),false) == true)
      return;

   if (FileName == DestFile)
      erase = true;
   else
      Local = true;

   decompression = true;
   DestFile += ".decomp";
   Desc.URI = "gzip:" + FileName;
   QueueURI(Desc);
   Mode = "gzip";
*/
}

class basicAcquireStatus : public pkgAcquireStatus
{
public:
	virtual bool MediaChange(string /*Media*/,string /*Drive*/) { return true; }
	virtual void IMSHit(pkgAcquire::ItemDesc &/*Itm*/)
	{
		fprintf(stderr, "IMSHit\n");
	}
	virtual void Fetch(pkgAcquire::ItemDesc &/*Itm*/)
	{
		//fprintf(stderr, "Fetch\n");
	}
	virtual void Done(pkgAcquire::ItemDesc &/*Itm*/)
	{
		//fprintf(stderr, "Done\n");
	}
	virtual void Fail(pkgAcquire::ItemDesc &Itm)
	{
		fprintf(stderr, "Failed downloading from %.*s\n", PFSTR(Itm.URI));
		_error->DumpErrors();
	}
	/*
	virtual bool Pulse(pkgAcquire *Owner)
	{
		fprintf(stderr, "PULSE\n");
		return pkgAcquireStatus::Pulse(Owner);
	}
	virtual void Start()
	{
		fprintf(stderr, "START\n");
		pkgAcquireStatus::Start();
	}
	virtual void Stop()
	{
		fprintf(stderr, "STOP\n");
		pkgAcquireStatus::Stop();
	}
	*/
};

namespace aptFront {
namespace cache {
namespace component {
namespace debtags {

void fetchNewData()
{
	basicAcquireStatus status;
	fetchNewData(&status);
}

void fetchNewData(pkgAcquireStatus* status)
{
	if (!_config->Exists("Dir::State"))
	{
		// Need to read the libapt-pkg configuration...
		pkgInitConfig(*_config);
	}
	
	vector<source> sources = readSources();

	pkgAcquire fetcher(status);

	vector<AcqTagfile*> acquirers;
	// TODO: deallocate acquirers (or is pkgAcquire doing it?)
	for (vector<source>::const_iterator i = sources.begin();
			i != sources.end(); i++)
		switch (i->type)
		{
			case source::TAGS:
				if (i->uri != "apt://")
				{
					acquirers.push_back(new AcqTagfile (&fetcher, i->uri + "tags-current.gz", i->uri + "tags-current.gz", "Tag database"));
					acquirers.push_back(new AcqTagfile (&fetcher, i->uri + "vocabulary.gz", i->uri + "vocabulary.gz", "Tag vocabulary"));
				}
				break;
			/*
			case source::DEBTAGS:
				acquirers.push_back(new AcqTagfile (&fetcher, i->uri + "tags-current.gz", "", ""));
				acquirers.push_back(new AcqTagfile (&fetcher, i->uri + "vocabulary.gz", "", ""));
				break;
				*/
		}

	if (fetcher.Run() == pkgAcquire::Failed)
	{
		_error->DumpErrors();
		// FIXME: need a better message as soon as I understand what's going on
		throw ConsistencyCheckException("Acquirer failed");
	}

	for (pkgAcquire::ItemIterator it = fetcher.ItemsBegin (); it != fetcher.ItemsEnd (); it ++)
	{
		if ((*it)->Status == pkgAcquire::Item::StatDone)
			continue;
		(*it)->Finished();
		// TODO: warn user about failures
		// TODO: I would like to: how?
	}
}

}
}
}
}

// vim:set ts=4 sw=4:
