/*
 * Tag vocabulary access
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <apt-front/utils/vocabularymerger.h>
#include <apt-front/utils/debdbparser.h>

#include <tagcoll/TDBFile.h>
#include <errno.h>

using namespace std;
using namespace Tagcoll;
using namespace aptFront;
using namespace utils;

static void writeDebStyleField(FILE* out, const string& name, const string& val) throw ()
{
	fprintf(out, "%.*s: ", PFSTR(name));

	// Properly escape newlines
	bool was_nl = false;
	for (string::const_iterator s = val.begin(); s != val.end(); s++)
		if (was_nl)
			// \n\n -> \n .\n
			if (*s == '\n')
			{
				fputc(' ', out);
				fputc('.', out);
				fputc(*s, out);
			}
			// \n([^ \t]) -> \n \1
			else if (*s != ' ' && *s != '\t')
			{
				fputc(' ', out);
				fputc(*s, out);
				was_nl = false;
			}
			// \n[ \t] goes unchanged
			else
			{
				fputc(*s, out);
				was_nl = false;
			}
		else
			if (*s == '\n')
			{
				fputc(*s, out);
				was_nl = true;
			}
			else
				fputc(*s, out);

	fputc('\n', out);
}

VocabularyMerger::TagData& VocabularyMerger::FacetData::obtainTag(const std::string& name)
{
	std::map<std::string, TagData>::iterator i = tags.find(name);
	if (i == tags.end())
	{
		// Create the tag if it's missing
		pair<std::map<std::string, TagData>::iterator, bool> res = tags.insert(make_pair<std::string, TagData>(name, TagData()));
		i = res.first;
		i->second.name = name;
	}
	return i->second;
}

VocabularyMerger::FacetData& VocabularyMerger::obtainFacet(const std::string& name)
{
	std::map<std::string, FacetData>::iterator i = facets.find(name);
	if (i == facets.end())
	{
		// Create the facet if it's missing
		pair<std::map<std::string, FacetData>::iterator, bool> res = facets.insert(make_pair<std::string, FacetData>(name, FacetData()));
		i = res.first;
		i->second.name = name;
	}
	return i->second;
}

VocabularyMerger::TagData& VocabularyMerger::obtainTag(const std::string& fullname)
{
	unsigned int p = fullname.find("::");
	if (p == string::npos)
	{
		FacetData& facet = obtainFacet("legacy");
		return facet.obtainTag(fullname);
	} else {
		FacetData& facet = obtainFacet(fullname.substr(0, p));
		return facet.obtainTag(fullname.substr(p + 2));
	}
}


void VocabularyMerger::read(ParserInput& input)
{
	DebDBParser parser(input);
	DebDBParser::Record record;

	while (parser.nextRecord(record))
	{
		DebDBParser::Record::const_iterator fi = record.find("Facet");
		DebDBParser::Record::const_iterator ti = record.find("Tag");
		if (fi != record.end())
		{
			// Get the facet record
			FacetData& facet = obtainFacet(fi->second);
			//fprintf(stderr, "Read facet@%d %.*s\n", parser.lineNumber(), PFSTR(facet.name));
			assert(facet.name == fi->second);

			// Merge the data
			for (DebDBParser::Record::const_iterator i = record.begin();
					i != record.end(); i++)
				if (i->first != "Facet")
					facet[i->first] = i->second;
		}
		else if (ti != record.end())
		{
			// Get the tag record
			TagData& tag = obtainTag(ti->second);
			//fprintf(stderr, "Read tag@%d %.*s\n", parser.lineNumber(), PFSTR(tag.name));
			//assert(tag.name == ti->second);

			// Merge the data
			for (DebDBParser::Record::const_iterator i = record.begin();
					i != record.end(); i++)
				if (i->first != "Tag")
					tag[i->first] = i->second;
		}
		else
		{
			fprintf(stderr, "%.*s:%d: Skipping record without Tag or Facet field\n",
					PFSTR(input.fileName()), input.lineNumber());
		}
	}
}

bool VocabularyMerger::hasTag(const std::string& fullname) const
{
	unsigned int p = fullname.find("::");
	std::string facetName;
	std::string tagName;
	if (p == string::npos)
	{
		facetName = "legacy";
		tagName = fullname;
	} else {
		facetName = fullname.substr(0, p);
		tagName = fullname.substr(p + 2);
	}

	std::map<std::string, FacetData>::const_iterator i = facets.find(facetName);
	if (i == facets.end())
		return false;
	return i->second.tags.find(tagName) != i->second.tags.end();
}

void VocabularyMerger::write(const std::string& fname)
{
	FILE* out = fopen(fname.c_str(), "wt");
	if (!out)
		throw FileException(errno, "creating file " + fname);
	write(out);
	fclose(out);
}

void VocabularyMerger::write(FILE* out)
{
	long start_ofs = ftell(out);

	//fprintf(stderr, "Write\n");
	for (std::map<std::string, FacetData>::iterator f = facets.begin(); f != facets.end(); f++)
	{
		//fprintf(stderr, "Writing facet %.*s\n", PFSTR(f->first));
		f->second.ofs = ftell(out) - start_ofs;
		writeDebStyleField(out, "Facet", f->first);
		for (std::map<std::string, std::string>::const_iterator j = f->second.begin();
				j != f->second.end(); j++)
			writeDebStyleField(out, j->first, j->second);
		fputc('\n', out);
		f->second.len = ftell(out) - f->second.ofs;

		for (std::map<std::string, TagData>::iterator t = f->second.tags.begin();
				t != f->second.tags.end(); t++)
		{
			//fprintf(stderr, "Writing tag %.*s\n", PFSTR(t->first));
			t->second.ofs = ftell(out) - start_ofs;
			writeDebStyleField(out, "Tag", f->first + "::" + t->first);
			for (std::map<std::string, std::string>::const_iterator j = t->second.begin();
					j != t->second.end(); j++)
				writeDebStyleField(out, j->first, j->second);
			fputc('\n', out);
			t->second.len = ftell(out) - t->second.ofs;
		}
	}
}

struct IndexData
{
	long ofs;
	int len;
	IndexData(long ofs, int len) : ofs(ofs), len(len) {}
};

void VocabularyMerger::writeIndex(const std::string& fname, long base_ofs)
{
	TDBFile db(fname);
	db.open(TDB_CLEAR_IF_FIRST, O_RDWR | O_CREAT, 0666);

	OpSet<string> facetList;
	for (std::map<std::string, FacetData>::const_iterator f = facets.begin(); f != facets.end(); f++)
	{
		db.set<IndexData>("F"+f->first, IndexData(f->second.ofs + base_ofs, f->second.len));

		OpSet<string> tags;
		for (std::map<std::string, TagData>::const_iterator t = f->second.tags.begin();
				t != f->second.tags.end(); t++)
		{
			db.set<IndexData>("T"+f->first + "::" + t->first, IndexData(t->second.ofs + base_ofs, t->second.len));
			tags += t->first;
		}

		db.setStringSet("t"+f->first, tags);
		facetList += (f->first);
	}
	db.setStringSet("f", facetList);
}



#if 0
void Vocabulary::read() throw (FileException, ParserException)
{
	StdioParserInput mainInput(Debtags::Environment::get().path_vocabulary());
	read(mainInput);
}

void Vocabulary::readVocabularies() throw (SystemException, FileException, ParserException)
{
	StdioParserInput mainInput(fn_dist_vocab);
	read(mainInput);
	
	DIR* dir = opendir(path_tagvoc_d);
	if (!dir)
		throw SystemException(errno, string("reading directory ") + path_tagvoc_d);

	while (struct dirent* d = readdir(dir))
	{
		if (d->d_name[0] != '.' &&
				d->d_name[strlen(d->d_name) - 1] != '~')
		{
			string fn = string(path_tagvoc_d) + '/' + d->d_name;
			if (access(fn.c_str(), R_OK) == 0)
			{
				StdioParserInput patchInput(fn);
				read(patchInput);
			}
		}
	}
	closedir(dir);
}

#endif

// vim:set ts=4 sw=4:
