/*
 * Tag vocabulary access
 *
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <apt-front/cache/component/tags.h>
#include <apt-front/utils/debdbparser.h>
#include <apt-front/utils/paths.h>

#include <tagcoll/StdioParserInput.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>

#include <errno.h>

using namespace aptFront::cache;
using namespace component;
using namespace std;
using namespace Tagcoll;

//#define TRACE
#ifdef TRACE
#include <iostream>
static bool trace = false;
#endif


Tags::Tags()
{
	init(utils::Path::vocabulary());
}

Tags::Tags(const std::string& filename)
{
	init(filename);
}

void Tags::init(const std::string& filename)
{
	StdioParserInput in(filename);
	utils::DebDBParser parser(in);
	utils::DebDBParser::Record record;

	while (parser.nextRecord(record))
	{
		utils::DebDBParser::Record::const_iterator fi = record.find("Facet");
		utils::DebDBParser::Record::const_iterator ti = record.find("Tag");
		if (fi != record.end())
		{
#ifdef TRACE
			if (trace) cerr << "Found facet: " << fi->second << endl;
#endif
			utils::DebDBParser::Record::const_iterator desc = record.find("Description");
			if (desc != record.end())
			{
				unsigned int send = desc->second.find('\n');
				if (send == string::npos)
					m_facets.push_back(entity::Facet::Data(fi->second, desc->second, string()));
				else
					m_facets.push_back(entity::Facet::Data(
								fi->second,
								desc->second.substr(0, desc->second.find('\n')),
								desc->second.substr(send + 1)));
			}
			else
				m_facets.push_back(entity::Facet::Data(fi->second));
#ifdef TRACE
			if (trace) {
				cerr << "  added id: " << m_facets.size() - 1 << endl;
				cerr << "  added name: " << m_facets[m_facets.size() - 1].name << endl;
				cerr << "  added sdes: " << m_facets[m_facets.size() - 1].shortDesc << endl;
				cerr << "  added ldes: " << m_facets[m_facets.size() - 1].longDesc << endl;
			}
#endif
			m_facetinfo.insert(make_pair(fi->second, FacetInfo(m_facets.size() - 1, m_tags.size())));
		}
		else if (ti != record.end())
		{
#ifdef TRACE
			if (trace) cerr << "Found tag: " << ti->second << endl;
#endif
			// Get the facet ID for this tag
			unsigned int seppos = ti->second.find("::");

			// We are not interested in legacy facetless tags
			if (seppos == string::npos)
				continue;
			// And there should be something after the ::
			if (seppos + 2 >= ti->second.size())
				continue;

			std::string name(ti->second.substr(seppos + 2));
			std::map<std::string, FacetInfo>::iterator facet(
					m_facetinfo.find(ti->second.substr(0, seppos)));
			
			// Skip all tags defined before their facet
			if (facet == m_facetinfo.end())
				continue;
			
			utils::DebDBParser::Record::const_iterator desc = record.find("Description");
			if (desc != record.end())
			{
				unsigned int send = desc->second.find('\n');
				if (send == string::npos)
					m_tags.push_back(entity::Tag::Data(facet->second.id, name, desc->second, string()));
				else
					m_tags.push_back(entity::Tag::Data(
								facet->second.id,
								name,
								desc->second.substr(0, send),
								desc->second.substr(send + 1)));
			}
			else
				m_tags.push_back(entity::Tag::Data(facet->second.id, name));
#ifdef TRACE
			if (trace) {
				cerr << "  added id: " << m_tags.size() - 1 << endl;
				cerr << "  added fcet: " << m_tags[m_tags.size() - 1].facet << endl;
				cerr << "  added name: " << m_tags[m_tags.size() - 1].name << endl;
				cerr << "  added sdes: " << m_tags[m_tags.size() - 1].shortDesc << endl;
				cerr << "  added ldes: " << m_tags[m_tags.size() - 1].longDesc << endl;
			}
#endif
			m_taginfo[ti->second] = m_tags.size() - 1;
			++facet->second.tagCount;
		}
		else
		{
			fprintf(stderr, "%.*s:%d: Skipping record without Tag or Facet field\n",
					PFSTR(in.fileName()), in.lineNumber());
		}
	}
}

entity::Facet Tags::facetByName(const std::string& name) const
{
	std::map<std::string, FacetInfo>::const_iterator i(m_facetinfo.find(name));
	if (i == m_facetinfo.end())
		return entity::Facet();
	else
		return entity::Facet(ownerCache(), i->second.id);
}

entity::Tag Tags::tagByName(const std::string& fullname) const
{
	std::map<std::string, unsigned int>::const_iterator i(m_taginfo.find(fullname));
	if (i == m_taginfo.end())
		return entity::Tag();
	else
		return entity::Tag(ownerCache(), i->second);
}

OpSet<entity::Facet> Tags::facets() const
{
	OpSet<entity::Facet> res;

	for (unsigned int i = 0; i < m_facets.size(); i++)
		res += entity::Facet(ownerCache(), i);

	return res;
}

OpSet<entity::Tag> Tags::tags() const
{
	OpSet<entity::Tag> res;

	for (unsigned int i = 0; i < m_tags.size(); ++i)
		res += entity::Tag(ownerCache(), i);

	return res;
}

OpSet<entity::Tag> Tags::tags(const std::string& facetName) const
{
	std::map<std::string, FacetInfo>::const_iterator facet(m_facetinfo.find(facetName));
	if (facet == m_facetinfo.end())
		return OpSet<entity::Tag>();

	OpSet<entity::Tag> res;
	for (unsigned int i = 0; i < facet->second.tagCount; ++i)
		res += entity::Tag(ownerCache(), facet->second.firstTag + i);
	return res;
}


#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>

namespace tut {

struct cache_component_tags_shar {
    cache_component_tags_shar() {
        aptInit();
        c.open( Cache::OpenDefault | Cache::OpenTags | Cache::OpenReadOnly );
    }
    Cache c;
};
TESTGRP(cache_component_tags);

template<> template<>
void to::test< 1 >()
{
    c.tags(); // this will throw if the open above didn't work
}

template<> template<>
void to::test< 2 >()
{
    ensure( c.tags().hasFacet( "works-with" ) );
    ensure( !c.tags().hasFacet( "blah" ) );
}

template<> template<>
void to::test< 3 >()
{
    ensure( c.tags().hasTag( "works-with::people" ) );
    ensure( !c.tags().hasTag( "works-with::midgets" ) );
}

template<> template<>
void to::test< 4 >()
{
    entity::Tag people = c.tags().tagByName( "works-with::people" ),
                midgets = c.tags().tagByName( "works-with::midgets" ),
                blahg = c.tags().tagByName( "works-with::blahg" ),
                text = c.tags().tagByName( "works-with::text" ),
                people2 = c.tags().tagByName( "works-with::people" );
    ensure( people != midgets );
    ensure( people != text );
    ensure( people != blahg );
    ensure( midgets == blahg );
    ensure( midgets == midgets );
    ensure( people == people2 );
    ensure( people == people );
}

template<> template<>
void to::test< 5 >()
{
    entity::Tag a = c.tags().tagByName( "works-with::people" ),
                b = c.tags().tagByName( "works-with::midgets" );
    OpSet< entity::Tag > s = c.tags().tags(),
                         f = c.tags().tags( "works-with" ),
                         n = c.tags().tags( "nonsense" );
    ensure( s.contains( a ) );
    ensure( f.contains( a ) );
    ensure( s.contains( f ) );
    ensure( !s.contains( b ) );
    ensure( !f.contains( b ) );
    ensure( n.empty() );
}

template<> template<>
void to::test< 6 >()
{
	entity::Facet f = c.tags().facetByName( "works-with" );
    entity::Tag t = c.tags().tagByName( "works-with::people" );
	ensure_equals(f.name(), "works-with");
	ensure_equals(t.name(), "people");
	ensure_equals(t.fullname(), "works-with::people");
}

template<> template<>
void to::test< 7 >()
{
    entity::Facet f = c.tags().facetByName( "works-with" );
    OpSet< entity::Tag > x = c.tags().tags( "works-with" );
    ensure( x == f.tags() );
}

template<> template<>
void to::test< 8 >()
{
    entity::Facet f = c.tags().facetByName( "does-not-work-with" );
    int x = 1;
    try {
        f.tags();
        x = 2;
    } catch (...) {
        x = 3;
    }
    ensure_equals( x, 3 );
}

template<> template<>
void to::test< 9 >()
{
    entity::Facet f = c.tags().facetByName( "legacy" );
    ensure_equals(f.shortDescription(), "");
    ensure_equals(f.longDescription(), "");
    //ensure_equals(f.shortDescription( "weehee" ), "weehee");
}

template<> template<>
void to::test< 10 >()
{
	// ensure that one-character tag names are parsed correctly
	ensure( c.tags().hasTag( "langdevel::c" ) );
}

}

#endif

// vim:set ts=4 sw=4:
