/**
 * @file cache/component/debtags/serializer.h
 * @author Enrico Zini (enrico) <enrico@enricozini.org>
 */

#include <apt-front/cache/component/debtags/serializer.h>
#include <apt-front/cache/component/tags.h>
#include <apt-front/cache/entity/package.h>
#include <apt-front/cache/entity/tag.h>
#include <apt-front/cache/cache.h>

using namespace aptFront::cache;
using namespace component;

namespace Tagcoll {

/**
 * Convert between strings and Packages
 */

std::string Converter<entity::Package, std::string>::operator()(const entity::Package& item)
{
	return item.name();
}

OpSet<std::string> Converter<entity::Package, std::string>::operator()(const OpSet<entity::Package>& items)
{
	OpSet<std::string> res;
	for (OpSet<entity::Package>::const_iterator i = items.begin();
			i != items.end(); i++)
		if (i->valid())
			res += (*this)(*i);
	return res;
}

entity::Package Converter<std::string, entity::Package>::operator()(const std::string& item)
{
	return Global::get(cache).packages().packageByName(item);
}

OpSet<entity::Package> Converter<std::string, entity::Package>::operator()(const OpSet<std::string>& items)
{
	OpSet<entity::Package> res;
	for (OpSet<std::string>::const_iterator i = items.begin();
			i != items.end(); i++)
	{
		entity::Package p = (*this)(*i);
		if (p.valid())
			res += p;
	}
	return res;
}

/**
 * Convert between strings and Facets
 */

std::string Converter<entity::Facet, std::string>::operator()(const entity::Facet& item)
{
	return item.name();
}

OpSet<std::string> Converter<entity::Facet, std::string>::operator()(const OpSet<entity::Facet>& items)
{
	OpSet<std::string> res;
	for (OpSet<entity::Facet>::const_iterator i = items.begin();
			i != items.end(); i++)
		if (i->valid())
			res += (*this)(*i);
	return res;
}

entity::Facet Converter<std::string, entity::Facet>::operator()(const std::string& item)
{
	return Global::get(cache).tags().facetByName(item);
}

OpSet<entity::Facet> Converter<std::string, entity::Facet>::operator()(const OpSet<std::string>& items)
{
	OpSet<entity::Facet> res;
	for (OpSet<std::string>::const_iterator i = items.begin();
			i != items.end(); i++)
	{
		entity::Facet p = (*this)(*i);
		if (p.valid())
			res += p;
	}
	return res;
}

/**
 * Convert between strings and Tags
 */

std::string Converter<entity::Tag, std::string>::operator()(const entity::Tag& item)
{
	return item.fullname();
}

OpSet<std::string> Converter<entity::Tag, std::string>::operator()(const OpSet<entity::Tag>& items)
{
	OpSet<std::string> res;
	for (OpSet<entity::Tag>::const_iterator i = items.begin();
			i != items.end(); i++)
		if (i->valid())
			res += (*this)(*i);
	return res;
}

entity::Tag Converter<std::string, entity::Tag>::operator()(const std::string& item)
{
	return Global::get(cache).tags().tagByName(item);
}

OpSet<entity::Tag> Converter<std::string, entity::Tag>::operator()(const OpSet<std::string>& items)
{
	OpSet<entity::Tag> res;
	for (OpSet<std::string>::const_iterator i = items.begin();
			i != items.end(); i++)
	{
		entity::Tag p = (*this)(*i);
		if (p.valid())
			res += p;
	}
	return res;
}

}


#ifdef INSTANTIATING_TEMPLATES
template class Converter<std::string, entity::Package>;
template class Converter<entity::Package, std::string>;
#endif


#ifdef COMPILE_TESTSUITE
#include <apt-front/cache/component/debtags/update.h>
#include "test-utils.h"

namespace tut {

struct cache_component_debtags_serializer_shar {
    cache_component_debtags_serializer_shar () {
        aptInit ();
        ok = true;
        debtags::generateIndexes();
        c.open( Cache::OpenDefault |
                Cache::OpenReadOnly | Cache::OpenDebtags );
    }
    void check() {
        if (ok) return;
        ok = true;
        throw warning( "debtags init failed, cancelling" );
    }
    ~cache_component_debtags_serializer_shar() {
        check();
    }
    Cache c;
    bool ok;
};

TESTGRP( cache_component_debtags_serializer );

using namespace Tagcoll;

template<> template<>
void to::test<1> ()
{
    check();

	Converter<entity::Package, std::string> tostring;
	Converter<std::string, entity::Package> topackage;
    topackage.setOwnerCache( &c );

	ensure(topackage("Slartibartsfart") == entity::Package());

    /* Get the 'debtags' package */
    entity::Package p = c.packages().packageByName( "debtags" );
    ensure(p.valid());

	/* Get the 'debtags' package using the serializer */
	entity::Package p1 = topackage("debtags");
	ensure(p1.valid());

	/* They must be the same */
	ensure(p == p1);

	ensure_equals(tostring(p), "debtags");
	ensure_equals(tostring(p1), "debtags");
	ensure_equals(tostring(p), tostring(p1));

	/* If there is an invalid package to serialize, it should be discarded */
	{
		Tagcoll::OpSet<entity::Package> pkgs;
		pkgs += c.packages().packageByName( "debtags" );
		pkgs += c.packages().packageByName( "tagcoll" );
		pkgs += entity::Package();

		ensure_equals (pkgs.size(), 3);
		ensure_equals (tostring(pkgs).size(), 2);
		ensure (tostring(pkgs).contains("debtags"));
		ensure (tostring(pkgs).contains("tagcoll"));
	}

	/* If there is an invalid package to serialize, it should be discarded */
	{
		Tagcoll::OpSet<std::string> pkgs;
		pkgs += "debtags";
		pkgs += "tagcoll";
		pkgs += "Slartibartsfart";

		ensure_equals (pkgs.size(), 3);
		ensure_equals (topackage(pkgs).size(), 2);
		ensure (topackage(pkgs).contains(topackage("debtags")));
		ensure (topackage(pkgs).contains(topackage("tagcoll")));
		ensure (!topackage(pkgs).contains(entity::Package()));
	}
}

}
#endif
// vim:set ts=4 sw=4:
