/*
 * greeter_iface.c
 * LTSP display manager.
 * Manages spawning a session to a server.
 *
 * Copyright 2007, Scott Balneaves <sbalneav@ltsp.org>
 *

 * Author: Scott Balneaves <sbalneav@ltsp.org>

 * 2007, Scott Balneaves <sbalneav@ltsp.org>
 *       Oliver Grawert <ogra@canonical.com>
 *       Vagrant Cascadian <vagrant@freegeek.org>
 * 2008, Scott Balneaves <sbalneav@ltsp.org>
 *       Ryan Niebur <RyanRyan52@gmail.com>
 *       John Ellson <john.ellson@comcast.net>
 *       Gideon Romm <gadi@ltsp.org>
 *       Warren Togami <wtogami@redhat.com>
 *       Vagrant Cascadian <vagrant@freegeek.org>
 *       Francis Giraldeau <francis.giraldeau@revolutionlinux.com>
 *       Toshio Kuratomi

 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, you can find it on the World Wide
 * Web at http://www.gnu.org/copyleft/gpl.html, or write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
 * MA 02110-1301, USA.

 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <sys/wait.h>
#include <glib.h>
#include <glib/gi18n.h>

#include "ldm.h"

int
get_userid(void)
{
    gchar *prompt;
    gchar *isguest;
    gsize l, t;
    gboolean empty;

    empty = TRUE;

    while(empty) {
        prompt = g_strconcat("prompt <b>", _("Username"), "</b>\n", NULL);
        if (g_io_channel_write_chars(ldm.greeterw, prompt, -1, NULL, NULL) !=
            G_IO_STATUS_NORMAL) {
            return 1;
        }
        if (g_io_channel_flush(ldm.greeterw, NULL) != G_IO_STATUS_NORMAL) {
            return 1;
        }
        if (g_io_channel_read_line(ldm.greeterr, &isguest, &l, &t, NULL) !=
            G_IO_STATUS_NORMAL) {
            return 1;
        }
        /* figure out if user is a guest */
        if (ldm.allowguest && !strcmp(isguest, "@GUEST@\n")) {
            ldm.autologin = TRUE;
            empty = FALSE;
        } else {
            ldm.autologin = FALSE;
            if (g_io_channel_read_line(ldm.greeterr, &ldm.username, &l, &t, NULL)
                != G_IO_STATUS_NORMAL) {
                return 1;
            }
            ldm.username[t] = '\0';
            g_strstrip(ldm.username);
            empty = g_regex_match_simple("^\\s*$", ldm.username, 0, 0);
        }
        g_free(isguest);
        g_free(prompt);
    }
    return 0;
}

int
get_passwd(void)
{
    gchar *prompt;
    gchar *command = "passwd\n";
    gsize l, t;

    prompt = g_strconcat("prompt <b>", _("Password"), "</b>\n", NULL);
    if (g_io_channel_write_chars(ldm.greeterw, prompt, -1, NULL, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_write_chars(ldm.greeterw, command, -1, NULL, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_flush(ldm.greeterw, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_read_line(ldm.greeterr, &ldm.password, &l, &t, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    ldm.password[t] = '\0';
    g_free(prompt);
    set_message(_("<b>Verifying password.  Please wait.</b>"));
    return 0;
}

int
set_message(gchar *message)
{
    gchar *prompt;

    if (ldm.autologin) {
        return 0;
    }

    prompt = g_strconcat("msg ", message, "\n", NULL);
    if (g_io_channel_write_chars(ldm.greeterw, prompt, -1, NULL, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_flush(ldm.greeterw, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    g_free(prompt);
    return 0;
}

int
get_host(void)
{
    char *cmd = "hostname\n";
    gsize l, t;

    if (g_io_channel_write_chars(ldm.greeterw, cmd, -1, NULL, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_flush(ldm.greeterw, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_read_line(ldm.greeterr, &ldm.server, &l, &t, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    ldm.server[t] = '\0';
    return 0;
}

int
get_language(void)
{
    char *cmd = "language\n";
    gchar *lang;
    gsize l, t;

    if (g_io_channel_write_chars(ldm.greeterw, cmd, -1, NULL, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_flush(ldm.greeterw, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_read_line(ldm.greeterr, &lang, &l, &t, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    lang[t] = '\0';
    if (g_ascii_strncasecmp(lang, "None", 4)) {
        if (ldm.lang) {
            g_free(ldm.lang);
        }
        ldm.lang = lang;
    }
    return 0;
}

int
get_session(void)
{
    char *cmd = "session\n";
    gchar *session;
    gsize l, t;

    if (g_io_channel_write_chars(ldm.greeterw, cmd, -1, NULL, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_flush(ldm.greeterw, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    if (g_io_channel_read_line(ldm.greeterr, &session, &l, &t, NULL) !=
            G_IO_STATUS_NORMAL) {
        return 1;
    }
    session[t] = '\0';
    if (g_ascii_strncasecmp(session, "None", 4)) {
        if (ldm.session) {
            g_free(ldm.session);
        }
        ldm.session = session;
    }
    return 0;
}

void
close_greeter(void)
{
    gint rfd, wfd;
    gboolean failed = FALSE;
    if (!ldm.greeterpid) {
        return;
    }
    if (g_io_channel_write_chars(ldm.greeterw, "quit\n", -1, NULL, NULL) == G_IO_STATUS_NORMAL) {
        if(g_io_channel_flush(ldm.greeterw, NULL) != G_IO_STATUS_NORMAL) {
            failed = TRUE;
        }
    } else {
        failed = TRUE;
    }
    if(failed == TRUE) {
        logerr("Sending greeter the quit command failed");
        if (kill(ldm.greeterpid, SIGTERM) < 0) {
            logerr("Sending SIGTERM to greeter failed");
            kill(ldm.greeterpid, SIGKILL);
        }
    }
    ldm_wait(ldm.greeterpid);
    rfd = g_io_channel_unix_get_fd(ldm.greeterr);
    wfd = g_io_channel_unix_get_fd(ldm.greeterw);
    g_io_channel_shutdown(ldm.greeterr, TRUE, NULL);
    g_io_channel_shutdown(ldm.greeterw, TRUE, NULL);
    close(rfd);
    close(wfd);
    ldm.greeterpid = 0;
}
