
from qt import *
from kdecore import *
from kdeui import *
import os
from displayconfigabstraction import *

# Running as the root user or not?
isroot = os.getuid()==0

############################################################################
class ResizeSlider(QVGroupBox):
    """ An abstracted QSlider in a nice box to change the resolution of a screen """
    def __init__(self,parent):
        # Screen size group
        QVGroupBox.__init__(self,QString("groupBox8"),parent)
        self._buildGUI()
    
    def _buildGUI(self):
        self.setTitle("Screen Size")
        self.setInsideSpacing(KDialog.spacingHint())
        self.setInsideMargin(KDialog.marginHint())
        
        hbox3 = QHBox(self)
        hbox3.setSpacing(KDialog.spacingHint())
        label = QLabel(hbox3,"textLabel2_4")
        label.setText("Lower")
        self.screensizeslider = QSlider(hbox3,"slider1")
        self.screensizeslider.setMinValue(0)
        self.screensizeslider.setMaxValue(4)
        self.screensizeslider.setPageStep(1)
        self.screensizeslider.setOrientation(QSlider.Horizontal)
        self.screensizeslider.setTickmarks(QSlider.Below)
        self.connect(self.screensizeslider,SIGNAL("valueChanged(int)"),self.slotResolutionChange)
        label = QLabel(hbox3)
        label.setText("Higher")

        self.resolutionlabel = QLabel(self)
        self.resolutionlabel.setText("640x400")

    def setScreen(self, screen):
        self.screen = screen
        self.screensizeslider.setMaxValue(len(screen.getAvailableResolutions())-1)
        self.screensizeslider.setValue(screen.getResolutionIndex())
        self.setResolutionIndex(screen.getResolutionIndex())
        
    def slotResolutionChange(self,i):
        """ Pass signal from slider through to App """
        try:
            self.setResolutionIndex(i)
        except AttributeError:
            self.setResolutionLabel("640x400")
        print "Hmm ... "
        self.emit(PYSIGNAL("resolutionChange(int)"),(i,))

    def setMaxValue(self,value):
        return self.screensizeslider.setMaxValue(value)
        
    def setMinValue(self,value):
        return self.screensizeslider.setMinValue(value)
        
    def setValue(self,value):
        return self.screensizeslider.setValue(value)
        
    def value(self):
        return self.screensizeslider.value()
        
    def setResolutionLabel(self,text):
        self.resolutionlabel.setText(text) 
    
    def setResolutionIndex(self,i):
        self.setResolutionLabel( "%i x %i" % self.screen.getAvailableResolutions()[i] )

############################################################################
class MonitorPreview(QWidget):
    """ A ResizableMonitor is an Image in a grid which has resizable edges, 
        fixed-size corners and is thus expandable. """
        
    def __init__(self, parent=None, imagedir="", name=None, modal=0, fl=0):
        QWidget.__init__(self,parent)
        
        self.ROT_NORMAL = "normal"
        self.ROT_LEFT = "left"
        self.ROT_RIGHT = "right"
        self.ROT_UPSIDEDOWN = "upsidedown"
        
        self.ROT_DISABLED = False

        self.rotation = self.ROT_NORMAL
        self.last_rotation = self.rotation
        
        self.mirror_ver = 0
        self.mirror_hor = 0

        self.baseW = 230
        self.baseH = 210
        
        self.imagedir = imagedir + "monitor_resizable/"
        self._loadImages()
        
    def _loadImages(self):
        """ Load all images, they form a grid with 4 rows and 3 coloms, the first three
         rows form the screen, with corners (1,3,7,9), edges (2,4,6,8) the center
         image, a screenshot that can be rotated and mirrored. The fourth row only contains
         one image, the center one (11) with the foot. """
         
        imagedir = self.imagedir
        
        self.image1 = QPixmap(imagedir+"res-01.png")
        self.image2 = QPixmap(imagedir+"res-02.png")
        self.image3 = QPixmap(imagedir+"res-03.png")
        self.image4 = QPixmap(imagedir+"res-04.png")
        # Image 5 is the center image and will get loaded later on.
        self.image6 = QPixmap(imagedir+"res-06.png")
        self.image7 = QPixmap(imagedir+"res-07.png")
        self.image8 = QPixmap(imagedir+"res-08.png")
        self.image9 = QPixmap(imagedir+"res-09.png")
        self.image11 = QPixmap(imagedir+"res-11.png")

        # All images loaded, let's complete the puzzle.
        self.Form2Layout = QGridLayout(self,2,1,11,6,"Form2Layout")

        self.groupBox2 = QGroupBox(self,"groupBox2")
        self.groupBox2.setEnabled(1)
        self.groupBox2.setColumnLayout(0,Qt.Vertical)
        self.groupBox2.layout().setSpacing(0)
        self.groupBox2.layout().setMargin(11)
        groupBox2Layout = QGridLayout(self.groupBox2.layout())
        groupBox2Layout.setAlignment(Qt.AlignTop)

        self.res_1 = QLabel(self.groupBox2,"res_1")
        self.res_1.setSizePolicy(QSizePolicy(QSizePolicy.Fixed,QSizePolicy.Fixed,0,0, \
                self.res_1.sizePolicy().hasHeightForWidth()))
        self.res_1.setPixmap(self.image1)
        self.res_1.setScaledContents(1)

        groupBox2Layout.addWidget(self.res_1,0,0)

        self.res_2 = QLabel(self.groupBox2,"res_2")
        self.res_2.setSizePolicy(QSizePolicy(QSizePolicy.Expanding,QSizePolicy.Fixed,0,0, \
                self.res_2.sizePolicy().hasHeightForWidth()))
        self.res_2.setPixmap(self.image2)
        self.res_2.setScaledContents(1)

        groupBox2Layout.addMultiCellWidget(self.res_2,0,0,1,2)

        self.res3 = QLabel(self.groupBox2,"res3")
        self.res3.setSizePolicy(QSizePolicy(QSizePolicy.Fixed,QSizePolicy.Fixed,0,0, \
                self.res3.sizePolicy().hasHeightForWidth()))
        self.res3.setPixmap(self.image3)
        self.res3.setScaledContents(1)

        groupBox2Layout.addWidget(self.res3,0,3)

        self.res_4 = QLabel(self.groupBox2,"res_4")
        self.res_4.setSizePolicy(QSizePolicy(QSizePolicy.Fixed,QSizePolicy.Expanding,0,0, \
                self.res_4.sizePolicy().hasHeightForWidth()))
        self.res_4.setPixmap(self.image4)
        self.res_4.setScaledContents(1)

        groupBox2Layout.addWidget(self.res_4,1,0)
        
        self.res_5 = QLabel(self.groupBox2,"res_5")
        self.res_5.setSizePolicy(QSizePolicy(QSizePolicy.Expanding,QSizePolicy.Expanding,0,0, \
                self.res_5.sizePolicy().hasHeightForWidth()))
        self._loadCenterImage()
        self.res_5.setScaledContents(1)
        
        groupBox2Layout.addMultiCellWidget(self.res_5,1,1,1,2)

        self.res_6 = QLabel(self.groupBox2,"res_6")
        self.res_6.setSizePolicy(QSizePolicy(QSizePolicy.Fixed,QSizePolicy.Expanding,0,0, \
                self.res_6.sizePolicy().hasHeightForWidth()))
        self.res_6.setPixmap(self.image6)
        self.res_6.setScaledContents(1)

        groupBox2Layout.addWidget(self.res_6,1,3)

        self.res_7 = QLabel(self.groupBox2,"res_7")
        self.res_7.setSizePolicy(QSizePolicy(QSizePolicy.Fixed,QSizePolicy.Fixed,0,0, \
                self.res_7.sizePolicy().hasHeightForWidth()))
        self.res_7.setPixmap(self.image7)
        self.res_7.setScaledContents(1)

        groupBox2Layout.addWidget(self.res_7,2,0)

        self.res_8 = QLabel(self.groupBox2,"res_8")
        self.res_8.setSizePolicy(QSizePolicy(QSizePolicy.Expanding,QSizePolicy.Fixed,0,0, \
                self.res_8.sizePolicy().hasHeightForWidth()))
        self.res_8.setPixmap(self.image8)
        self.res_8.setScaledContents(1)

        groupBox2Layout.addMultiCellWidget(self.res_8,2,2,1,2)

        self.res_9 = QLabel(self.groupBox2,"res_9")
        self.res_9.setSizePolicy(QSizePolicy(QSizePolicy.Fixed,QSizePolicy.Fixed,0,0, \
                self.res_9.sizePolicy().hasHeightForWidth()))
        self.res_9.setPixmap(self.image9)
        self.res_9.setScaledContents(1)

        groupBox2Layout.addWidget(self.res_9,2,3)

        self.res_11 = QLabel(self.groupBox2,"res_11")
        self.res_11.setSizePolicy(QSizePolicy(QSizePolicy.Expanding,QSizePolicy.Fixed,0,0, \
                self.res_11.sizePolicy().hasHeightForWidth()))
        self.res_11.setPixmap(self.image11)
        self.res_11.setScaledContents(1)
        self.res_11.setAlignment(QLabel.AlignCenter)

        groupBox2Layout.addMultiCellWidget(self.res_11,3,3,1,2)

        self.Form2Layout.addWidget(self.groupBox2,0,0)
        self.Form2Layout.addWidget(self.groupBox2,0,0)

        self.resize(QSize(689,683).expandedTo(self.minimumSizeHint()))
        self.clearWState(Qt.WState_Polished)
    
    def _loadCenterImage(self):
        """ 
        The original size of centerimage.png is 400x300, it is resized to
        FIXED_HEIGHT, which is the actual size of the centerimage in the 
        preview screen. FIXED_HEIGHT is also responsible for the absolute
        size of this widget since width is computed from the screen ratio
        (width/height), the total size of the widget is screensize + size
        of surrounding images (see _changeResolution)."""

        self.FIXED_HEIGHT = 200
        self.centerimage = QPixmap(self.imagedir+'centerimage.png')
        x,y = (self.centerimage.width(), self.centerimage.height())
        ratio = float(x) / float(y)
        x = self.FIXED_HEIGHT*ratio
        y = self.FIXED_HEIGHT
        self.orig_centerimage = self.centerimage
        
    def setMinimalResolution(self, size):
        self.min_w, self.min_h = size
    
    def setAvailableResolutions(self,resolutions):
        self.resolutions = resolutions
         
    def _pixmapResize(self,image,x,y):
        """ Resize an QPixmap to size = (x,y) """
        o_x = image.width()
        o_y = image.height()
        scale_x = float(x) / float(o_x)
        scale_y = float(y) / float(o_y)
        matrix = QWMatrix()
        matrix.scale(scale_x,scale_y)
        return image.xForm(matrix)
        
    def _clipPixmap(self,size):
        """ 
        To show the resolution we cut the logo off the middle of the centerimage
        and have it upscaled automatically again so content on the screen gets 
        bigger without increasing the size of the actual widget. Reference point 
        is the biggest resolution that's not very wide (i.e. width > 2*height).
        """
        # Find reasonable reference point.
        resolutions = []
        for res in self.resolutions:
            if res[0] < 2*res[1]:
                resolutions.append(res)
        resolutions.sort()
        biggest = resolutions[-1]
        
        if size == biggest or size not in resolutions:
            return self.orig_centerimage
        
        # Compute clipping points in the center of the image.
        # We decrease the difference between the resolutions a little, because otherwise our
        # logo would become to big for small resolutions.
        o_width = self.orig_centerimage.width()
        w_clipping = ((o_width*(float(biggest[0])/float(size[0])))-o_width) / 1.5
       
        o_height = self.orig_centerimage.height()
        h_clipping = ((o_height*(float(biggest[1])/float(size[1])))-o_height) / 1.5
        
        x = w_clipping/2 
        y = h_clipping/2 
        w = o_width-w_clipping
        h = o_height-h_clipping
        
        # Create QWidget from our image where we can clip our new center off.
        pixlabel = QLabel(None,"plaatje")
        pixlabel.setPixmap(self.orig_centerimage)
        pixlabel.setLineWidth(0)
        pixlabel.setMargin(0)
        pixlabel.setScaledContents(0)
        pixlabel.setFixedSize(o_width,o_height)
        
        # The dirty work ...
        return QPixmap.grabWidget(pixlabel,x,y,w,h)
        
    def changeResolution(self, size):
        """ 
        We use the size (which is the actual resolution) to compute the size of the 
        centerimage (fixed height), width is dependant on the ratio of "size" parameter.
        The actual size of the widget is the size of the centerimage plus some pixels for
        the frame and surrouning images.
        The overall size of the preview thus only reflects changes in width / height ratio,
        Changes in resolution are reflected in the size of the logo in the middle of the 
        preview screen which gets clipped and scaled. """
        w,h = size
        ratio = float(w) / float(h)
        w, h = (self.FIXED_HEIGHT*ratio, self.FIXED_HEIGHT)

        if self._changePreview(self._pixmapResize(self._clipPixmap(size),w,h)):
            w,h = h,w

        # Widgetsize is the size of the centerimage + size of surrounding images.
        self.setFixedSize(w+100,h+110)

    def setRotation(self, rotation):
        self.rotation = rotation

    def getRotation(self):
        return self.rotation
     
    def disableRotation(self, disable):
        self.ROT_DISABLED = disable
        self.setRotation(self.rotation)
        
    def setMirrorHorizontally(self, enable):
        self.mirror_hor = enable
        
    def setMirrorVertically(self, enable):
        self.mirror_ver = enable
        
    def _changePreview(self,image=None):
        """ We use the size (which is the actual resolution to compute the size of the 
            centerimage (fixed height), width is dependant on the ratio of "size" parameter.
        """
        
        rotation = self.rotation

        if self.ROT_DISABLED:
            self.res_5.setPixmap(self.centerimage)
            return
        matrix = QWMatrix()
        rot = 0
        # Portrait is passed back to changeResolution because the 
        # widget size must be turned in case we're lying on left or 
        # right edge.
        portrait = False
        if rotation == self.ROT_UPSIDEDOWN:
            rot = 180
        elif rotation == self.ROT_LEFT:
            rot = 90
            portrait = True
        elif rotation == self.ROT_RIGHT:
            rot = 270
            portrait = True
        matrix.rotate(rot)
        
        if self.mirror_ver:
            matrix.scale(1,-1)
        if self.mirror_hor:
            matrix.scale(-1,1)

        # Finally, change the Pixmap
        self.res_5.setPixmap(image.xForm(matrix))
        return portrait

############################################################################
class DualMonitorPreview(QLabel):
    """ This is the Widget to use elsewhere. It consists of a canvas and an
        arbitrary number of gizmos on the canvas. The gizmos can be dragged and 
        dropped around. Painting is double-buffered so flickering should not occur.
    """
    def __init__(self,parent,size,imagedir):
        self.imagedir = imagedir + "dualhead/"
        self.snap_distance = 15
        self.snapping = True
        QLabel.__init__(self,parent)
        self.setLineWidth(3)
        self.setMargin(3)
        self.dd_is_enabled = False
        w,h = size
        self.resize(w,h)
        self.mycanvas = Canvas(w,h,self.imagedir)
        self.setPixmap(self.mycanvas)
        
    def getCurrentLabel(self):
        return self.mycanvas.currentGizmo().getLabel()
        
    def enableDD(self): 
        """ Switch on moving of the target, used when clicked on it """
        self.dd_is_enabled = True
        
    def enableSnapping(self,enable=True):
        self.snapping = enable
        
    def disableSnapping(self,disable=True):
        self.enableSnapping(not disable)
        
    def setCurrentScreen(self,screen):
        self.current_screen = screen
        
    def getCurrentScreen(self):
        return self.current_screen
        
    def disableDD(self): 
        """ Switch off moving of the target, used when clicked outside of it """
        self.dd_is_enabled = False
        
    def mousePressEvent(self,event):
        """ Check if the click has hit one of our gizmos and set this one to the 
            as current gizmo, so it's the one that we're moving while dragging.
            This is basically starting to receive mousemovements in order to drag."""
        mouse_position = event.pos()
        cur_giz = self.mycanvas.currentGizmo()
        
        monitor = QRect(cur_giz.getPosition(),cur_giz.getSize())
        
        # First handle active gizmo since it stays on top.
        if monitor.contains(mouse_position):
            self.enableDD()
            self.old_x = event.pos().x()
            self.old_y = event.pos().y()
            self.emit(PYSIGNAL("pressed()"), (event.pos(),) )
        # If the active gizmo is not hit, check the rest.
        else:
            for giz in self.mycanvas.gizmos:
                monitor = QRect(giz.getPosition(),QSize(giz.width(),giz.height()))
                if giz is not cur_giz and monitor.contains(mouse_position):
                    self.mycanvas.setCurrentGizmo(giz)
                    self.emit(PYSIGNAL("currentMonitorChanged()"),())
                    for giz_hi in self.mycanvas.gizmos:
                        if giz_hi is not giz:
                            giz_hi.setHiglight(False)
                    self.enableDD()
                self.old_x = event.pos().x()
                self.old_y = event.pos().y()
                self.emit(PYSIGNAL("pressed()"), (event.pos(),) )
        self.mycanvas.paintMe()
        self.paintEvent() 
        
    def mouseReleaseEvent(self,event):
        """ Mouse is no more pressed, disable dragging, so this is basically dropping
            of the gizmo (if one had been hit). """
        mv_x = self.old_x + event.pos().x()
        mv_y = self.old_y + event.pos().y()
        # Snapping
        if self.snapping:
            cur_giz = self.mycanvas.currentGizmo()
            dist = self.snap_distance
            x = y = -1
            for giz in self.mycanvas.gizmos:
                if giz is not cur_giz:
                    # Snapping?
                    if   (cur_giz.x_lines[0]-dist) < giz.x_lines[0] < (cur_giz.x_lines[0]+dist):
                        # Left edges snap
                        x = giz.x_lines[0]
                    elif (cur_giz.x_lines[1]-dist) < giz.x_lines[0] < (cur_giz.x_lines[1]+dist):
                        # Current right snaps other's left edge
                        x = giz.x_lines[0]-cur_giz.width()
                    elif (cur_giz.x_lines[0]-dist) < giz.x_lines[1] < (cur_giz.x_lines[0]+dist):
                        # Current left snaps other's right edge
                        x = giz.x_lines[1]
                    elif (cur_giz.x_lines[1]-dist) < giz.x_lines[1] < (cur_giz.x_lines[1]+dist):
                        # Right edges snap
                        x = giz.x_lines[1]-cur_giz.width()
                        
                    if   (cur_giz.y_lines[0]-dist) < giz.y_lines[0] < (cur_giz.y_lines[0]+dist):
                        # Top edges snap
                        y = giz.y_lines[0]
                    elif (cur_giz.y_lines[1]-dist) < giz.y_lines[0] < (cur_giz.y_lines[1]+dist):
                        # Current bottom snaps other's top edge
                        y = giz.y_lines[0]-cur_giz.height()
                    elif (cur_giz.y_lines[0]-dist) < giz.y_lines[1] < (cur_giz.y_lines[0]+dist):
                        # Current top snaps other's bottom edge
                        y = giz.y_lines[1]
                    elif (cur_giz.y_lines[1]-dist) < giz.y_lines[1] < (cur_giz.y_lines[1]+dist):
                        # Bottom edges snap
                        y = giz.y_lines[1]-cur_giz.height()
                        
            orig_x = self.mycanvas.currentGizmo().getPosition().x()
            orig_y = self.mycanvas.currentGizmo().getPosition().y()
            
            if x != -1 == y:
                self.mycanvas.currentGizmo().setPosition(QPoint(x,orig_y))
            if x == -1 != y:
                self.mycanvas.currentGizmo().setPosition(QPoint(orig_x,y))
            if x != -1 and y != -1:
                self.mycanvas.currentGizmo().setPosition(QPoint(x,y))
        
        nx,ny = cur_giz.getPosition().x(),cur_giz.getPosition().y()
        self.emit(PYSIGNAL("released()"), (nx,ny,) )
        self.disableDD()
        self.mycanvas.paintMe()
        self.paintEvent()
        
    def mouseMoveEvent(self,event):
        """ First compute the difference between the last and the current point,
            set the current gizmo to the new position. then paint a new pixmap 
            with all gizmos at the new position and trigger a paintEvent in order 
            to blit it to the widget. """        
        if not self.dd_is_enabled:
            return
        # Compute difference between old and new position.
        cur_x = event.pos().x()
        cur_y = event.pos().y()
        mv_point = QPoint((self.old_x - cur_x),(self.old_y - cur_y))
        self.old_x = cur_x
        self.old_y = cur_y
        # Move dragged gizmo to the new position
        cur_pos = self.mycanvas.currentGizmo().getPosition()
        
        new_position = (cur_pos - mv_point)
        # Make sure the gizmo stays inside our canvas.
        new_rect = QRect(new_position,self.mycanvas.currentGizmo().getSize())
        if self.mycanvas.rect().contains(new_rect):
            self.mycanvas.currentGizmo().setPosition(new_position)
        # Paint everything offscreen
        p = QPainter()
        p.begin(self.mycanvas)
        self.mycanvas.paintMe()
        p.end()
        # Trigger paintEvent to copy the Canvas onto the visible widget.
        self.paintEvent()
        # Trigger whatever the "outside world wants to do.
        self.emit(PYSIGNAL("moved()"), () )
        
    def paintEvent(self,event=None):
        bitBlt(self,0,0,self.mycanvas)

############################################################################
class Canvas(QPixmap):
    """ A Canvas is a Pixmap we're painting all the gizmos on """
    def __init__(self,w=500,h=500,imagedir="."):
        QPixmap.__init__(self,w,h)
        self.frame_rect = QRect(0,0,w-1,h-1)
        self.imagedir = imagedir
        self.resize(w,h)
        self.gizmos = []
        self.gizmos.append(MovingGizmo("Monitor 1","monitor_1.png","monitor_1_hi.png",QPoint(20,50),self.imagedir))
        self.gizmos.append(MovingGizmo("Monitor 2","monitor_2.png","monitor_2_hi.png",QPoint(180,50),self.imagedir))
        
        self.gizmos[0].rescale(.666)
        #self.gizmos[1].rescale(.75)
        
        # Check if all our gizmos are in the visible area.
        for giz in self.gizmos:
            if not self.rect().contains(giz.getRect()):
                print "Not all gizmos are in visible area"
                sys.exit(1)
        self.setCurrentGizmo(self.gizmos[0])
        self.p = QPainter(self)
        self.paintMe()
        
    def currentGizmo(self):
        """ ... is the gizmo that's been dragged right now, or the last one that 
            has been dragged """
        return self.current_gizmo
            
    def setCurrentGizmo(self,gizmo):
        """ This happens when a gizmo is clicked, it becomes "active" """
        gizmo.setHiglight()
        self.current_gizmo = gizmo
        
    def paintMe(self):
        """ Do everything necessary to paint the full canvas:
            fill with background color and paint all the gizmos, the current one 
            is the last one to be painted, finally paint a black frame around it."""
        bgcolor = QColor(200,200,200)
        self.fill(bgcolor) # grey
        for giz in self.gizmos:
            if giz is not self.currentGizmo():
                self.p.drawPixmap(giz.getPosition(),giz.pixmap)
        self.p.drawPixmap(self.currentGizmo().getPosition(),self.currentGizmo().pixmap)
        self.p.setPen(QPen(QColor(100,100,100),1))
        self.p.drawRoundRect(self.frame_rect,2,2)
        
############################################################################
class MovingGizmo:
    """ A gizmo is a pixmap that 
            * has a current position on the Canvas, 
            * can be highlighed
            * can be rescaled """
    
    def __init__(self,label,filename,filename_hi=False,initial_pos=QPoint(0,0),imagedir="."):
        """ filename is the unhighlighted pixmap, filename_hi is the pixmap that's 
            shown when the gizmo is highlighed and being dragged and dropped. """
        self.label = label
        self.imagedir = imagedir
        self.filename = filename
        self.pixmap_unhi_orig = QPixmap(self.imagedir+filename)
        self.pixmap_unhi = self.pixmap_unhi_orig
        if not filename_hi:
            self.pixmap_hi = self.pixmap_unhi
        else:
            self.pixmap_hi_orig = QPixmap(self.imagedir+filename_hi)
            self.pixmap_hi = self.pixmap_hi_orig
        self.pixmap = self.pixmap_unhi
        self.highlight = False
        self.size = QSize(self.width(),self.height())
        self.rectangle = QRect(initial_pos, self.getSize())
        self.setPosition(initial_pos)
        
    def setPixmap(self):
        """ Sets the pixmap dependant on if it's highlighed or not. """
        if self.highlight:
            self.pixmap = self.pixmap_hi
        else:
            self.pixmap = self.pixmap_unhi
        
    def setHiglight(self,enable=True):
        self.highlight = enable
        self.setPixmap()
    
    def getLabel(self):
        return self.label
    
    def setPosition(self,position):
        self.x_lines = [position.x(),(position.x()+self.width())]
        self.y_lines = [position.y(),(position.y()+self.height())]
        self.position = position
        
    def getSize(self):
        return self.size
        
    def getPosition(self):
        return self.position
    
    def getRect(self):
        return QRect(self.position,self.getSize())
    
    def width(self):
        return self.pixmap.width()
    
    def height(self):
        return self.pixmap.height()
    
    def resize(self,factor):
        """ Resize the pixmap by a certain factor, which is a tuple. We always use
            _orig as source since resizing pixmap multiple times will produce ugly 
            results."""
        matrix = QWMatrix()
        matrix.scale(factor[0],factor[1])
        self.pixmap_hi = self.pixmap_hi_orig.xForm(matrix)
        self.pixmap_unhi = self.pixmap_unhi_orig.xForm(matrix)
        self.setPixmap()
        self.size = QSize(self.width(),self.height())
        
    def rescale(self,scale):
        """ Helper function, wraps resize for convenience. """
        self.resize((scale,scale))
        self.setPosition(self.position)


############################################################################
class GfxCardWidget(QVGroupBox):
    def __init__(self, parent, xsetup, gfxcard, gfxcarddialog, monitordialog):
        global imagedir
        QVGroupBox.__init__(self,parent)

        self.xsetup = xsetup
        self.gfxcard = gfxcard
        self.gfxcarddialog = gfxcarddialog
        self.monitordialog = monitordialog
        self._buildGUI()
        self._syncGUI()
        
    def _buildGUI(self):
        # Create the GUI    
        
        gridwidget = QWidget(self)
        grid = QGridLayout(gridwidget,2+2*len(self.gfxcard.getScreens()))
        grid.setSpacing(KDialog.spacingHint())
        grid.setColStretch(0,0)
        grid.setColStretch(1,0)
        grid.setColStretch(2,1)
        grid.setColStretch(3,0)
        
        gfxcardpic = QLabel(gridwidget)
        gfxcardpic.setPixmap(UserIcon('hi32-gfxcard'))
        grid.addMultiCellWidget(gfxcardpic,0,1,0,0)
        
        label = QLabel(gridwidget)
        label.setText("Graphics card:")
        grid.addWidget(label,0,1)
        
        self.gfxcardlabel = QLabel(gridwidget)
        grid.addWidget(self.gfxcardlabel,0,2)
        
        label = QLabel(gridwidget)
        label.setText("Driver:")
        grid.addWidget(label,1,1)
        
        self.driverlabel = QLabel(gridwidget)
        grid.addWidget(self.driverlabel,1,2)
        
        gfxbutton = QPushButton(gridwidget)
        gfxbutton.setText("Configure...")
        self.connect(gfxbutton,SIGNAL("clicked()"),self.slotGfxCardConfigureClicked)
        grid.addWidget(gfxbutton,0,3)
        gfxbutton.setEnabled(isroot)
        
        # Add all of the screens
        row = 2
        count = 1
        self.monitorlabels = []
        self.monitorbuttons = []
        for screen in self.gfxcard.getScreens():
            frame = QFrame(gridwidget)
            frame.setFrameShape(QFrame.HLine)
            frame.setFrameShadow(QFrame.Sunken)
            grid.addMultiCellWidget(frame,row,row,0,3)
            row += 1
            
            monitorpic = QLabel(gridwidget)
            monitorpic.setPixmap(UserIcon('hi32-display'))
            grid.addWidget(monitorpic,row,0)
    
            label = QLabel(gridwidget)
            if len(self.gfxcard.getScreens())==1:
                label.setText("Monitor:")
            else:
                label.setText("Monitor #%i:" % count)
            grid.addWidget(label,row,1)
            
            self.monitorlabels.append(QLabel(gridwidget))
            grid.addWidget(self.monitorlabels[-1],row,2)
    
            monitorbutton = QPushButton(gridwidget)
            self.monitorbuttons.append(monitorbutton)
            monitorbutton.setText("Configure...")
            self.connect(monitorbutton,SIGNAL("clicked()"),self.slotMonitorConfigureClicked)
            grid.addWidget(monitorbutton,row,3)
            monitorbutton.setEnabled(isroot)
            row += 1
            count += 1
            
    def _syncGUI(self):        
        if self.gfxcard.getGfxCardModel() is not None:
            self.setTitle(self.gfxcard.getGfxCardModel().getName())
            self.gfxcardlabel.setText(self.gfxcard.getGfxCardModel().getName())
            
            if self.gfxcard.isProprietaryDriver():
                self.driverlabel.setText(self.gfxcard.getGfxCardModel().getProprietaryDriver())
            else:
                self.driverlabel.setText(self.gfxcard.getGfxCardModel().getDriver())
        else:
            self.setTitle("<Unknown>")
            self.gfxcardlabel.setText("<Unknown>")
            self.driverlabel.setText("<none>")
        
        # Sync the monitors.
        for i in range(len(self.gfxcard.getScreens())):
            screen = self.gfxcard.getScreens()[i]
            if screen.getMonitorModel() is None:
                self.monitorlabels[i].setText("<unknown>")
            else:
                self.monitorlabels[i].setText(screen.getMonitorModel().getName())
                
    def slotGfxCardConfigureClicked(self):
        (new_card_model, new_proprietary_driver) = self.gfxcarddialog.do(self.gfxcard.getGfxCardModel(), \
            self.gfxcard.isProprietaryDriver(), self.gfxcard.getDetectedGfxCardModel())
            
        if new_card_model is self.gfxcard.getGfxCardModel() and new_proprietary_driver==self.gfxcard.isProprietaryDriver():
            return
        self.gfxcard.setGfxCardModel(new_card_model)
        self.gfxcard.setProprietaryDriver(new_proprietary_driver)
        self._syncGUI()
        self.emit(PYSIGNAL("xconfigChanged"), () )
        
    def slotMonitorConfigureClicked(self):
        screen_index = self.monitorbuttons.index(self.sender())
        screen_obj = self.gfxcard.getScreens()[screen_index]
        new_monitor_model = self.monitordialog.do(screen_obj.getMonitorModel())
        screen_obj.setMonitorModel(new_monitor_model)
        self._syncGUI()
        self.emit(PYSIGNAL("xconfigChanged"), () )
        
############################################################################
class XFreeScreenWidget(QVGroupBox):
    def __init__(self, parent, xconfig, screen, gfxcarddb, monitordb, monitormodedb, gfxcarddialog, monitordialog):
        global imagedir
        QVGroupBox.__init__(self,parent)
        
        self.modified = False
        
        self.xconfig = xconfig
        self.screen = screen
        self.gfxcarddialog = gfxcarddialog
        self.monitordialog = monitordialog
        self.monitormodedb = monitormodedb
        self.selectedcard = None
        self.selectedmonitor = None
        
        # Find the device object in the xconfig that this screen config is referring to.
        #devicename = self.screen.device
        devicename = self.screen.getDevice()
        print "Screen.device:", devicename
        for device in self.xconfig.getSections('device'):
            print "DevID:",device.identifier
            if device.identifier==devicename:
                self.device = device
                break
        else:
            print "*** Couldn't find device!!!"
            #FIXME something smarter here.
            
        self.proprietarydriver = False
        # Match the gfx card in the xconfig to a Card object in the Gfxcard DB.
        if self.device.boardname in gfxcarddb.db:
            self.selectedcard = gfxcarddb.db[self.device.boardname]
            self.proprietarydriver = self.selectedcard.getProprietaryDriver()==self.device.driver
        else:
            self.selectedcard = gfxcarddb.db[self.device.driver]
            
        # Find the monitor object in xconfig that the screen config refers to.
        monitorname = self.screen.getMonitor()
        for monitor in self.xconfig.getSections('monitor'):
            if monitor.identifier==monitorname:
                self.monitor = monitor
                break
        else:
            print "*** Couldn't find monitor!!!"
            #FIXME something smarter here.

        modelname = getattr(self.monitor,"modelname",None)
        if modelname in monitordb.db:
            self.selectedmonitor = monitordb.db[modelname]
        else:
            # Create a monitor object for the monitor in the xconfig.
            # It is probably a Plug N Play monitor and as such doesn't 
            # appear in our monitor DB.
            self.selectedmonitor = MonitorModel()
            if modelname is None: modelname = "(unknown)"
            self.selectedmonitor.setName(modelname)
            self.selectedmonitor.setManufacturer(getattr(self.monitor,'vendor',None))
            #def setEisaId(self,eisaid): self.eisaid = eisaid
            #def setDpms(self,on): self.dpms = on
            self.selectedmonitor.setHorizontalSync(','.join(self.monitor.getRow('horizsync')))
            self.selectedmonitor.setVerticalSync(','.join(self.monitor.getRow('vertrefresh')))

        self.originalmonitor = self.selectedmonitor
        self.originalcard = self.selectedcard
        self.originalproprietarydriver = self.proprietarydriver
        
        self._buildGUI()
        self._syncGUI()
        
    def isChanged(self):
        return (self.originalmonitor is not self.selectedmonitor) \
            or (self.originalcard is not self.selectedcard) \
            or (self.originalproprietarydriver!=self.proprietarydriver)

    def reset(self):
        self.selectedmonitor = self.originalmonitor
        self.selectedcard = self.originalcard
        self.proprietarydriver = self.originalproprietarydriver
        self._syncGUI()
    
    def _buildGUI(self):
        # Create the GUI    
        self.setTitle(self.screen.getIdentifier())
        
        gridwidget = QWidget(self)
        grid = QGridLayout(gridwidget,4)
        grid.setSpacing(KDialog.spacingHint())
        grid.setColStretch(0,0)
        grid.setColStretch(1,0)
        grid.setColStretch(2,1)
        grid.setColStretch(3,0)
        
        gfxcardpic = QLabel(gridwidget)
        gfxcardpic.setPixmap(UserIcon('hi32-gfxcard'))
        grid.addMultiCellWidget(gfxcardpic,0,1,0,0)
        
        label = QLabel(gridwidget)
        label.setText("Graphics card:")
        grid.addWidget(label,0,1)
        
        self.gfxcardlabel = QLabel(gridwidget)
        grid.addWidget(self.gfxcardlabel,0,2)
        
        label = QLabel(gridwidget)
        label.setText("Driver:")
        grid.addWidget(label,1,1)
        
        self.driverlabel = QLabel(gridwidget)
        grid.addWidget(self.driverlabel,1,2)
        
        gfxbutton = QPushButton(gridwidget)
        gfxbutton.setText("Configure...")
        self.connect(gfxbutton,SIGNAL("clicked()"),self.slotGfxCardConfigureClicked)
        grid.addWidget(gfxbutton,0,3)
        gfxbutton.setEnabled(isroot)
        
        frame = QFrame(gridwidget)
        frame.setFrameShape(QFrame.HLine)
        frame.setFrameShadow(QFrame.Sunken)
        grid.addMultiCellWidget(frame,2,2,0,3)

        monitorpic = QLabel(gridwidget)
        monitorpic.setPixmap(UserIcon('hi32-display'))
        grid.addWidget(monitorpic,3,0)

        label = QLabel(gridwidget)
        label.setText("Monitor:")
        grid.addWidget(label,3,1)
        
        self.monitorlabel = QLabel(gridwidget)
        grid.addWidget(self.monitorlabel,3,2)

        monitorbutton = QPushButton(gridwidget)
        monitorbutton.setText("Configure...")
        self.connect(monitorbutton,SIGNAL("clicked()"),self.slotMonitorConfigureClicked)
        grid.addWidget(monitorbutton,3,3)
        monitorbutton.setEnabled(isroot)
        
    def _syncGUI(self):
        self.gfxcardlabel.setText(self.selectedcard.getName())
        if self.proprietarydriver:
            self.driverlabel.setText(self.selectedcard.getProprietaryDriver())
        else:
            self.driverlabel.setText(self.selectedcard.getDriver())

        # Build a reasonable human readable name for the monitor model/type.
        if self.selectedmonitor.getManufacturer() is not None:
            name = self.selectedmonitor.getManufacturer()
        else:
            name = ""
        if self.selectedmonitor.getName() is not None:
            if name!="":
                name += ", "
            name += self.selectedmonitor.getName()
        self.monitorlabel.setText(name)
        
    def slotGfxCardConfigureClicked(self):
        (newselectedcard, newproprietarydriver) = self.gfxcarddialog.do(self.selectedcard, self.proprietarydriver)
        if newselectedcard is self.selectedcard and newproprietarydriver==self.proprietarydriver:
            return
        self.selectedcard = newselectedcard
        self.proprietarydriver = newproprietarydriver
        self._syncGUI()

        # Create a new Device section in the Xorg config file.
        newdevice = self.xconfig.makeSection(None,'device')
        newdevice.identifier = self.device.identifier
        if getattr(self.device,'busid',None) is not None:
            newdevice.busid = self.device.busid
        newdevice.boardname = self.selectedcard.getName()
        if self.selectedcard.getVendor() is not None:
            newdevice.vendorname = self.selectedcard.getVendor()
        
        if self.proprietarydriver and self.selectedcard.getProprietaryDriver() is not None:
            driver = self.selectedcard.getProprietaryDriver()
        else:
            driver = self.selectedcard.getDriver()
        newdevice.driver = driver
        
        if driver=='fglrx':
            # Default options for the ATI proprietary driver.
            self._insertRawLinesIntoConfig(newdevice,self.ati_fglrx_lines)
        
        # $raw_X->set_devices($card, @{$card->{cards} || []});
        # $raw_X->get_ServerLayout->{Xinerama} = { commented => !$card->{Xinerama}, Option => 1 }
        #if defined $card->{Xinerama};
        module = self.xconfig.getSections('module')[0]
        module.removeModule('GLcore')
        module.removeModule('glx')
        module.removeModule("/usr/X11R6/lib/modules/extensions/libglx.so")
        if driver=='nvidia':
            # This loads the NVIDIA GLX extension module.
            # IT IS IMPORTANT TO KEEP NAME AS FULL PATH TO libglx.so ELSE
            # IT WILL LOAD XFree86 glx module and the server will crash.
            module.addModule("/usr/X11R6/lib/modules/extensions/libglx.so")
            # FIXME lib64
        elif self.selectedcard.getProprietaryDriver()!='fglrx':
            module.addModule('glx')
            module.addModule('GLcore')

        module.removeModule("/usr/X11R6/lib/modules/extensions/libglx.a")
        if driver!='nvidia':
            module.addModule("/usr/X11R6/lib/modules/extensions/libglx.a")
            
        # DRI
        module.removeModule('dri')
        if self.selectedcard.getDriGlx():
            module.addModule('dri')

        module.removeModule('v4l')
        if not (self.selectedcard.getDriGlx() and self.selectedcard.getDriver()=='r128'):
            module.addModule('v4l')
            
        self._insertRawLinesIntoConfig(newdevice,self.selectedcard.getLines())
        
        # Find the device object that matches the device name in the screen config.
        devicename = self.screenconfig.device
        i = 0
        devicesections = self.xconfig.getSections('device')
        matchingdevice = None
        for device in devicesections:
            if device.identifier==devicename:
                matchingdevice = device
        self.xconfig.remove(matchingdevice)

        self.xconfig.append(newdevice)
        self.device = newdevice
        
        self.emit(PYSIGNAL("xconfigChanged"), () )

    def slotMonitorConfigureClicked(self):
        newmonitor = self.monitordialog.do(self.selectedmonitor)
        if newmonitor is self.selectedmonitor:
            return
        self.selectedmonitor = newmonitor
        self.modified = True
        self._syncGUI()

        self._setMonitor(self.xconfig,self.selectedmonitor,self.monitormodedb.getAvailableModes(self.selectedmonitor))
        self.emit(PYSIGNAL("xconfigChanged"), () )

    def isModified(self): return self.modified
    def getBadFbRestore(self):
        return self.selectedcard.getBadFbRestore(self.proprietarydriver)
    def _insertRawLinesIntoConfig(self,section,lines):
        reader = csv.reader(lines,delimiter=' ')
        for row in reader:
            if len(row)>=2:
                if row[0].lower()=="option":
                    option = section.makeLine(None,row)
                    section.append(option)

    def _setMonitor(self,xconfig,monitorobj,modearray):
        monitorname = self.screenconfig.monitor
        monitorsection = self.xconfig.makeSection(None,'monitor')
        monitorsection.identifier = monitorname
        monitorsection.vendorname = monitorobj.getManufacturer()
        monitorsection.modelname = monitorobj.getName()
    
        hsyncline = monitorsection.makeLine(None,['HorizSync',monitorobj.getHorizontalSync()])
        monitorsection.append(hsyncline)

        vsyncline = monitorsection.makeLine(None,['VertRefresh',monitorobj.getVerticalSync()])
        monitorsection.append(vsyncline)
    
        for modesize in modearray:
            for mode in modesize:
                modelinesection = monitorsection.makeSection(None,'mode')
                modelinesection.identifier = mode.name
                modelinesection.dotclock = int(math.floor(mode.clock*1000.0))
                
                htimings = modelinesection.makeLine(None,['htimings',
                                            mode.hdisp,mode.hsyncstart,mode.hsyncend,mode.htotal])
                modelinesection.append(htimings)
                
                vtimings = modelinesection.makeLine(None,['vtimings',
                                            mode.vdisp,mode.vsyncstart,mode.vsyncend,mode.vtotal])
                modelinesection.append(vtimings)
                
                modelinesection.flags = mode.flags
                monitorsection.append(modelinesection)

        # Remove the existing monitor config object.
        monitorsections = xconfig.getSections('monitor')
        matchingmonitor = None
        for monitor in monitorsections:
            if monitor.identifier==monitorname:
                matchingmonitor = monitor
        xconfig.remove(matchingmonitor)
        
        xconfig.append(monitorsection)

    ati_fglrx_lines = """# === disable PnP Monitor  ===
#Option                              "NoDDC"
# === disable/enable XAA/DRI ===
Option "no_accel"                   "no"
Option "no_dri"                     "no"
# === FireGL DDX driver module specific settings ===
# === Screen Management ===
Option "DesktopSetup"               "0x00000000" 

Option "MonitorLayout"              "AUTO, AUTO"
Option "IgnoreEDID"                 "off"
Option "HSync2"                     "unspecified" 
Option "VRefresh2"                  "unspecified" 
Option "ScreenOverlap"              "0" 
# === TV-out Management ===
Option "NoTV"                       "yes"     
Option "TVStandard"                 "NTSC-M"     
Option "TVHSizeAdj"                 "0"     
Option "TVVSizeAdj"                 "0"     
Option "TVHPosAdj"                  "0"     
Option "TVVPosAdj"                  "0"     
Option "TVHStartAdj"                "0"     
Option "TVColorAdj"                 "0"     
Option "GammaCorrectionI"           "0x00000000"
Option "GammaCorrectionII"          "0x00000000"
# === OpenGL specific profiles/settings ===
Option "Capabilities"               "0x00000000"
# === Video Overlay for the Xv extension ===
Option "VideoOverlay"               "on"
# === OpenGL Overlay ===
# Note: When OpenGL Overlay is enabled, Video Overlay
#       will be disabled automatically
Option "OpenGLOverlay"              "off"
Option "CenterMode"                 "off"
# === QBS Support ===
Option "Stereo"                     "off"
Option "StereoSyncEnable"           "1"
# === Misc Options ===
Option "UseFastTLS"                 "0"
Option "BlockSignalsOnLock"         "on"
Option "UseInternalAGPGART"         "no"
Option "ForceGenericCPU"            "no"
# === FSAA ===
Option "FSAAScale"                  "1"
Option "FSAADisableGamma"           "no"
Option "FSAACustomizeMSPos"         "no"
Option "FSAAMSPosX0"                "0.000000"
Option "FSAAMSPosY0"                "0.000000"
Option "FSAAMSPosX1"                "0.000000"
Option "FSAAMSPosY1"                "0.000000"
Option "FSAAMSPosX2"                "0.000000"
Option "FSAAMSPosY2"                "0.000000"
Option "FSAAMSPosX3"                "0.000000"
Option "FSAAMSPosY3"                "0.000000"
Option "FSAAMSPosX4"                "0.000000"
Option "FSAAMSPosY4"                "0.000000"
Option "FSAAMSPosX5"                "0.000000"
Option "FSAAMSPosY5"                "0.000000"
"""

##############################################################################
##class XFreeScreenWidget(QVGroupBox):
##    def __init__(self, parent, xconfig, screen, gfxcarddb, monitordb, monitormodedb, gfxcarddialog, monitordialog):
##        global imagedir
##        QVGroupBox.__init__(self,parent)
##        
##        self.modified = False
##        
##        self.xconfig = xconfig
##        self.screen = screen
##        self.gfxcarddialog = gfxcarddialog
##        self.monitordialog = monitordialog
##        self.monitormodedb = monitormodedb
##        self.selectedcard = None
##        self.selectedmonitor = None
##        
##        # Find the device object in the xconfig that this screen config is referring to.
##        #devicename = self.screen.device
##        devicename = self.screen.getDevice()
##        print "Screen.device:", devicename
##        for device in self.xconfig.getSections('device'):
##            print "DevID:",device.identifier
##            if device.identifier==devicename:
##                self.device = device
##                break
##        else:
##            print "*** Couldn't find device!!!"
##            #FIXME something smarter here.
##            
##        self.proprietarydriver = False
##        # Match the gfx card in the xconfig to a Card object in the Gfxcard DB.
##        if self.device.boardname in gfxcarddb.db:
##            self.selectedcard = gfxcarddb.db[self.device.boardname]
##            self.proprietarydriver = self.selectedcard.getProprietaryDriver()==self.device.driver
##        else:
##            self.selectedcard = gfxcarddb.db[self.device.driver]
##            
##        # Find the monitor object in xconfig that the screen config refers to.
##        monitorname = self.screen.getMonitor()
##        for monitor in self.xconfig.getSections('monitor'):
##            if monitor.identifier==monitorname:
##                self.monitor = monitor
##                break
##        else:
##            print "*** Couldn't find monitor!!!"
##            #FIXME something smarter here.
##
##        modelname = getattr(self.monitor,"modelname",None)
##        if modelname in monitordb.db:
##            self.selectedmonitor = monitordb.db[modelname]
##        else:
##            # Create a monitor object for the monitor in the xconfig.
##            # It is probably a Plug N Play monitor and as such doesn't 
##            # appear in our monitor DB.
##            self.selectedmonitor = MonitorModel()
##            if modelname is None: modelname = "(unknown)"
##            self.selectedmonitor.setName(modelname)
##            self.selectedmonitor.setManufacturer(getattr(self.monitor,'vendor',None))
##            #def setEisaId(self,eisaid): self.eisaid = eisaid
##            #def setDpms(self,on): self.dpms = on
##            self.selectedmonitor.setHorizontalSync(','.join(self.monitor.getRow('horizsync')))
##            self.selectedmonitor.setVerticalSync(','.join(self.monitor.getRow('vertrefresh')))
##
##        self.originalmonitor = self.selectedmonitor
##        self.originalcard = self.selectedcard
##        self.originalproprietarydriver = self.proprietarydriver
##        
##        self._buildGUI()
##        self._syncGUI()
##        
##    def isChanged(self):
##        return (self.originalmonitor is not self.selectedmonitor) \
##            or (self.originalcard is not self.selectedcard) \
##            or (self.originalproprietarydriver!=self.proprietarydriver)
##
##    def reset(self):
##        self.selectedmonitor = self.originalmonitor
##        self.selectedcard = self.originalcard
##        self.proprietarydriver = self.originalproprietarydriver
##        self._syncGUI()
##    
##    def _buildGUI(self):
##        # Create the GUI    
##        self.setTitle(self.screen.getIdentifier())
##        
##        gridwidget = QWidget(self)
##        grid = QGridLayout(gridwidget,4)
##        grid.setSpacing(KDialog.spacingHint())
##        grid.setColStretch(0,0)
##        grid.setColStretch(1,0)
##        grid.setColStretch(2,1)
##        grid.setColStretch(3,0)
##        
##        gfxcardpic = QLabel(gridwidget)
##        gfxcardpic.setPixmap(UserIcon('hi32-gfxcard'))
##        grid.addMultiCellWidget(gfxcardpic,0,1,0,0)
##        
##        label = QLabel(gridwidget)
##        label.setText("Graphics card:")
##        grid.addWidget(label,0,1)
##        
##        self.gfxcardlabel = QLabel(gridwidget)
##        grid.addWidget(self.gfxcardlabel,0,2)
##        
##        label = QLabel(gridwidget)
##        label.setText("Driver:")
##        grid.addWidget(label,1,1)
##        
##        self.driverlabel = QLabel(gridwidget)
##        grid.addWidget(self.driverlabel,1,2)
##        
##        gfxbutton = QPushButton(gridwidget)
##        gfxbutton.setText("Configure...")
##        self.connect(gfxbutton,SIGNAL("clicked()"),self.slotGfxCardConfigureClicked)
##        grid.addWidget(gfxbutton,0,3)
##        gfxbutton.setEnabled(isroot)
##        
##        frame = QFrame(gridwidget)
##        frame.setFrameShape(QFrame.HLine)
##        frame.setFrameShadow(QFrame.Sunken)
##        grid.addMultiCellWidget(frame,2,2,0,3)
##
##        monitorpic = QLabel(gridwidget)
##        monitorpic.setPixmap(UserIcon('hi32-display'))
##        grid.addWidget(monitorpic,3,0)
##
##        label = QLabel(gridwidget)
##        label.setText("Monitor:")
##        grid.addWidget(label,3,1)
##        
##        self.monitorlabel = QLabel(gridwidget)
##        grid.addWidget(self.monitorlabel,3,2)
##
##        monitorbutton = QPushButton(gridwidget)
##        monitorbutton.setText("Configure...")
##        self.connect(monitorbutton,SIGNAL("clicked()"),self.slotMonitorConfigureClicked)
##        grid.addWidget(monitorbutton,3,3)
##        monitorbutton.setEnabled(isroot)
##        
##    def _syncGUI(self):
##        self.gfxcardlabel.setText(self.selectedcard.getName())
##        if self.proprietarydriver:
##            self.driverlabel.setText(self.selectedcard.getProprietaryDriver())
##        else:
##            self.driverlabel.setText(self.selectedcard.getDriver())
##
##        # Build a reasonable human readable name for the monitor model/type.
##        if self.selectedmonitor.getManufacturer() is not None:
##            name = self.selectedmonitor.getManufacturer()
##        else:
##            name = ""
##        if self.selectedmonitor.getName() is not None:
##            if name!="":
##                name += ", "
##            name += self.selectedmonitor.getName()
##        self.monitorlabel.setText(name)
##        
##    def slotGfxCardConfigureClicked(self):
##        (newselectedcard, newproprietarydriver) = self.gfxcarddialog.do(self.selectedcard, self.proprietarydriver)
##        if newselectedcard is self.selectedcard and newproprietarydriver==self.proprietarydriver:
##            return
##        self.selectedcard = newselectedcard
##        self.proprietarydriver = newproprietarydriver
##        self._syncGUI()
##
##        # Create a new Device section in the Xorg config file.
##        newdevice = self.xconfig.makeSection(None,'device')
##        newdevice.identifier = self.device.identifier
##        if getattr(self.device,'busid',None) is not None:
##            newdevice.busid = self.device.busid
##        newdevice.boardname = self.selectedcard.getName()
##        if self.selectedcard.getVendor() is not None:
##            newdevice.vendorname = self.selectedcard.getVendor()
##        
##        if self.proprietarydriver and self.selectedcard.getProprietaryDriver() is not None:
##            driver = self.selectedcard.getProprietaryDriver()
##        else:
##            driver = self.selectedcard.getDriver()
##        newdevice.driver = driver
##        
##        if driver=='fglrx':
##            # Default options for the ATI proprietary driver.
##            self._insertRawLinesIntoConfig(newdevice,self.ati_fglrx_lines)
##        
##        # $raw_X->set_devices($card, @{$card->{cards} || []});
##        # $raw_X->get_ServerLayout->{Xinerama} = { commented => !$card->{Xinerama}, Option => 1 }
##        #if defined $card->{Xinerama};
##        module = self.xconfig.getSections('module')[0]
##        module.removeModule('GLcore')
##        module.removeModule('glx')
##        module.removeModule("/usr/X11R6/lib/modules/extensions/libglx.so")
##        if driver=='nvidia':
##            # This loads the NVIDIA GLX extension module.
##            # IT IS IMPORTANT TO KEEP NAME AS FULL PATH TO libglx.so ELSE
##            # IT WILL LOAD XFree86 glx module and the server will crash.
##            module.addModule("/usr/X11R6/lib/modules/extensions/libglx.so")
##            # FIXME lib64
##        elif self.selectedcard.getProprietaryDriver()!='fglrx':
##            module.addModule('glx')
##            module.addModule('GLcore')
##
##        module.removeModule("/usr/X11R6/lib/modules/extensions/libglx.a")
##        if driver!='nvidia':
##            module.addModule("/usr/X11R6/lib/modules/extensions/libglx.a")
##            
##        # DRI
##        module.removeModule('dri')
##        if self.selectedcard.getDriGlx():
##            module.addModule('dri')
##
##        module.removeModule('v4l')
##        if not (self.selectedcard.getDriGlx() and self.selectedcard.getDriver()=='r128'):
##            module.addModule('v4l')
##            
##        self._insertRawLinesIntoConfig(newdevice,self.selectedcard.getLines())
##        
##        # Find the device object that matches the device name in the screen config.
##        devicename = self.screenconfig.device
##        i = 0
##        devicesections = self.xconfig.getSections('device')
##        matchingdevice = None
##        for device in devicesections:
##            if device.identifier==devicename:
##                matchingdevice = device
##        self.xconfig.remove(matchingdevice)
##
##        self.xconfig.append(newdevice)
##        self.device = newdevice
##        
##        self.emit(PYSIGNAL("xconfigChanged"), () )
##
##    def slotMonitorConfigureClicked(self):
##        newmonitor = self.monitordialog.do(self.selectedmonitor)
##        if newmonitor is self.selectedmonitor:
##            return
##        self.selectedmonitor = newmonitor
##        self.modified = True
##        self._syncGUI()
##
##        self._setMonitor(self.xconfig,self.selectedmonitor,self.monitormodedb.getAvailableModes(self.selectedmonitor))
##        self.emit(PYSIGNAL("xconfigChanged"), () )
##
##    def isModified(self): return self.modified
##    def getBadFbRestore(self):
##        return self.selectedcard.getBadFbRestore(self.proprietarydriver)
##    def _insertRawLinesIntoConfig(self,section,lines):
##        reader = csv.reader(lines,delimiter=' ')
##        for row in reader:
##            if len(row)>=2:
##                if row[0].lower()=="option":
##                    option = section.makeLine(None,row)
##                    section.append(option)
##
##    def _setMonitor(self,xconfig,monitorobj,modearray):
##        monitorname = self.screenconfig.monitor
##        monitorsection = self.xconfig.makeSection(None,'monitor')
##        monitorsection.identifier = monitorname
##        monitorsection.vendorname = monitorobj.getManufacturer()
##        monitorsection.modelname = monitorobj.getName()
##    
##        hsyncline = monitorsection.makeLine(None,['HorizSync',monitorobj.getHorizontalSync()])
##        monitorsection.append(hsyncline)
##
##        vsyncline = monitorsection.makeLine(None,['VertRefresh',monitorobj.getVerticalSync()])
##        monitorsection.append(vsyncline)
##    
##        for modesize in modearray:
##            for mode in modesize:
##                modelinesection = monitorsection.makeSection(None,'mode')
##                modelinesection.identifier = mode.name
##                modelinesection.dotclock = int(math.floor(mode.clock*1000.0))
##                
##                htimings = modelinesection.makeLine(None,['htimings',
##                                            mode.hdisp,mode.hsyncstart,mode.hsyncend,mode.htotal])
##                modelinesection.append(htimings)
##                
##                vtimings = modelinesection.makeLine(None,['vtimings',
##                                            mode.vdisp,mode.vsyncstart,mode.vsyncend,mode.vtotal])
##                modelinesection.append(vtimings)
##                
##                modelinesection.flags = mode.flags
##                monitorsection.append(modelinesection)
##
##        # Remove the existing monitor config object.
##        monitorsections = xconfig.getSections('monitor')
##        matchingmonitor = None
##        for monitor in monitorsections:
##            if monitor.identifier==monitorname:
##                matchingmonitor = monitor
##        xconfig.remove(matchingmonitor)
##        
##        xconfig.append(monitorsection)
##
##    ati_fglrx_lines = """# === disable PnP Monitor  ===
###Option                              "NoDDC"
### === disable/enable XAA/DRI ===
##Option "no_accel"                   "no"
##Option "no_dri"                     "no"
### === FireGL DDX driver module specific settings ===
### === Screen Management ===
##Option "DesktopSetup"               "0x00000000" 
##
##Option "MonitorLayout"              "AUTO, AUTO"
##Option "IgnoreEDID"                 "off"
##Option "HSync2"                     "unspecified" 
##Option "VRefresh2"                  "unspecified" 
##Option "ScreenOverlap"              "0" 
### === TV-out Management ===
##Option "NoTV"                       "yes"     
##Option "TVStandard"                 "NTSC-M"     
##Option "TVHSizeAdj"                 "0"     
##Option "TVVSizeAdj"                 "0"     
##Option "TVHPosAdj"                  "0"     
##Option "TVVPosAdj"                  "0"     
##Option "TVHStartAdj"                "0"     
##Option "TVColorAdj"                 "0"     
##Option "GammaCorrectionI"           "0x00000000"
##Option "GammaCorrectionII"          "0x00000000"
### === OpenGL specific profiles/settings ===
##Option "Capabilities"               "0x00000000"
### === Video Overlay for the Xv extension ===
##Option "VideoOverlay"               "on"
### === OpenGL Overlay ===
### Note: When OpenGL Overlay is enabled, Video Overlay
###       will be disabled automatically
##Option "OpenGLOverlay"              "off"
##Option "CenterMode"                 "off"
### === QBS Support ===
##Option "Stereo"                     "off"
##Option "StereoSyncEnable"           "1"
### === Misc Options ===
##Option "UseFastTLS"                 "0"
##Option "BlockSignalsOnLock"         "on"
##Option "UseInternalAGPGART"         "no"
##Option "ForceGenericCPU"            "no"
### === FSAA ===
##Option "FSAAScale"                  "1"
##Option "FSAADisableGamma"           "no"
##Option "FSAACustomizeMSPos"         "no"
##Option "FSAAMSPosX0"                "0.000000"
##Option "FSAAMSPosY0"                "0.000000"
##Option "FSAAMSPosX1"                "0.000000"
##Option "FSAAMSPosY1"                "0.000000"
##Option "FSAAMSPosX2"                "0.000000"
##Option "FSAAMSPosY2"                "0.000000"
##Option "FSAAMSPosX3"                "0.000000"
##Option "FSAAMSPosY3"                "0.000000"
##Option "FSAAMSPosX4"                "0.000000"
##Option "FSAAMSPosY4"                "0.000000"
##Option "FSAAMSPosX5"                "0.000000"
##Option "FSAAMSPosY5"                "0.000000"
##"""
##
