#!/usr/bin/env python

import os
import string
import select
import math
import csv
import xf86misc
import xorgconfig
import ScanPCI

"""Classes for dealing with X.org configuration in a sane way.

The object model used here is fairly simple. An XSetup object represents
the complete configuration of the server. The XSetup object contains one or
more GfxCard objects. One for each graphics card present in the machine.
Each GfxCard has one or more Screen objects with each Screen representing
one 'output' on the graphics card.

Each GfxCard object is also associated with a GfxCardModel object which
describes the model of graphics card.

Each Screen object is associated with a MonitorModel object which
describes the model of monitor attached.

"""

############################################################################
class XSetup(object):
    """Represents the current configuration of the X.org X11 server.
    
    
    """
    # Map positions
    ABOVE = 0
    UNDER = 1
    LEFTOF = 2
    RIGHTOF = 3
    
    def __init__(self,xorg_config_filename='/etc/X11/xorg.conf'):
        self.screens = []
        self.gfxcards = []
        
        xdevice_id_to_gfxcard = {} # Lookup table, useful later on.
        pci_device_to_gfxcard = {}
        
        self.xorg_config = xorgconfig.readConfig(xorg_config_filename)
        self.x_live_info = xf86misc.XF86Server()
        
        pci_bus = ScanPCI.PCIBus()
        pci_bus.detect()
        
        # First thing. Scan the PCI bus and find out how many Gfx cards we have.
        found_list = self._detectGfxCards(pci_bus)

        # Prepare some useful data structures.
        
        # Maps screen section names to xorg screens section objects.
        xorg_screen_name_dict = {} 
        xorg_screen_sections = self.xorg_config.getSections("Screen")
        for screen in xorg_screen_sections:
            xorg_screen_name_dict[screen.identifier] = screen
        
        # Maps device sections names to xorg device sections
        xorg_device_name_dict = {}
        xorg_device_sections = self.xorg_config.getSections("Device")
        for device in xorg_device_sections:
            xorg_device_name_dict[device.identifier] = device
        
        # Do some probing of the xorg config to find out what kind of
        # configuration this is (eg single head, dual head, etc)
        pass
        
        # Find all of the active xorg device and screen sections.
        # Active meaning that is being used in the xorg configuration.
        active_device_sections = []
        active_screen_sections = []
        
        server_layouts = self.xorg_config.getSections("ServerLayout")
        for layout in server_layouts:
            for screen in layout.screen:
                screen_name = self._getFirstXorgScreenName(screen)
                if screen_name in xorg_screen_name_dict:
                    screen_section = xorg_screen_name_dict[screen_name]
                    active_screen_sections.append(screen_section)
                    print screen_section.device
                    if screen_section.device in xorg_device_name_dict:
                        active_device_sections.append(xorg_device_name_dict[screen_section.device])
        
        # Work out the list of inactive screen sections.
        inactive_screen_sections = []
        for screen in self.xorg_config.getSections("Screen"):
            if screen.identifier not in active_screen_sections:
                inactive_screen_sections.append(screen)
        
        # Work out the list of inactive device sections.
        inactive_device_sections = []
        for device in self.xorg_config.getSections("Device"):
            if device.identifier not in active_device_sections:
                inactive_device_sections.append(device)

        # Match xorg device sections with detected devices by using the PCI ids.
        for x_device in active_device_sections[:]: # Iterate over copy.
            if x_device.busid is not None:
                bus_id = self._canonicalPCIBusID(x_device.busid)
                for pci_device in found_list[:]:
                    if pci_device[0]==bus_id:
                    
                        new_gfx_card = GfxCard( x_config=self.xorg_config, \
                            pci_id=bus_id, \
                            x_device=x_device, \
                            detected_model=pci_device[2], \
                            proprietary_driver=(pci_device[2].getProprietaryDriver()==x_device.driver))

                        self.gfxcards.append( new_gfx_card )
                        xdevice_id_to_gfxcard[x_device.identifier] = new_gfx_card
                        pci_device_to_gfxcard[pci_device[1]] = new_gfx_card
                        active_device_sections.remove(x_device)
                        found_list.remove(pci_device)
        
        # Match based on driver.
        for x_device in active_device_sections[:]:
            if x_device.driver is not None:
                for pci_device in found_list[:]:
                    if pci_device[1] is not None:
                        # Also set some unknown device, we don't want it to crash here...
                        if pci_device[1].getDriver()==x_device.driver or pci_device[2].getProprietaryDriver()==x_device.driver:
                            
                            new_gfx_card = GfxCard( x_config=self.xorg_config, \
                                pci_id=pci_device[0], \
                                x_device=x_device, \
                                detected_model=pci_device[2], \
                                proprietary_driver=(pci_device[2].getProprietaryDriver()==x_device.driver))
                            
                            self.gfxcards.append(new_gfx_card)
                            xdevice_id_to_gfxcard[x_device.identifier] = new_gfx_card
                            pci_device_to_gfxcard[pci_device[1]] = new_gfx_card
                            active_device_sections.remove(x_device)
                            found_list.remove(pci_device)

        # Most of the xorg devices and pci devices should be matched up.
        # Use a little bit of very basic deduction to match left overs.
        if len(active_device_sections)==1 and len(found_list)==1:

            new_gfx_card = GfxCard( pci_id=bus_id, \
                x_config=self.xorg_config, \
                x_device=active_device_sections[0])
            
            self.gfxcards.append(new_gfx_card)
            xdevice_id_to_gfxcard[x_device.identifier] = new_gfx_card
            pci_device_to_gfxcard[found_list[0][1]] = new_gfx_card
            active_device_sections.remove(active_device_sections[0])
        else:
            # Ok, this stuff isn't matching up so well. The detected gfx cards
            # are probably more trustworthy than what is in xorg config.
            for pci_device in found_list:
            
                new_gfx_card = GfxCard( x_config=self.xorg_config, \
                    pci_id=pci_device[0], \
                    detected_model=pci_device[2])
                    
                self.gfxcards.append(new_gfx_card)
                pci_device_to_gfxcard[pci_device[1]] = new_gfx_card
    
        # Start filling in the GfxCard objects with xorg screen objects.
        i = 0
        x_live_screens = self.x_live_info.getScreens()
        for x_screen in active_screen_sections[:]:
            try:
                gfxcard = xdevice_id_to_gfxcard[x_screen.device]
                screen_name = "Screen %i" % (i+1,)
                print screen_name
                (monitor_section,monitor_model)= self._matchMonitor(self.xorg_config, x_screen.monitor)

                gfxcard._addScreen( Screen(x_config_screen=x_screen, identifier=screen_name, \
                                        screen_index=i, x_live_screen=x_live_screens[i],\
                                        x_config_monitor=monitor_section, monitor_model=monitor_model ))
                i += 1
            except KeyError:
                print "*** Couldn't find device with id:",x_screen.device

        # Detect ATI dualhead cards and create extra Screen objects if needed.
        for pci_gfxcard in pci_device_to_gfxcard.keys():
            for pci_device in pci_bus.devices:
                if pci_gfxcard is not pci_device:
                    if pci_device.text.find("Secondary")!=-1:
                        if pci_device.pci_bus==pci_gfxcard.pci_bus and pci_device.pci_device==pci_gfxcard.pci_device:
                            gfxcard = pci_device_to_gfxcard[pci_gfxcard]
                            screens = gfxcard.getScreens()
                            if len(screens)<2:
                                gfxcard._addScreen( Screen(identifier=(screens[0].getIdentifier()+" dual")) )
        
        if self.getScreen(0) is None: print "Hmm, no screen ..."
        else: print "Wow ... ", self.getScreen(0)
        
    def _matchMonitor(self,x_org_config,monitor_name):
        monitor_model_db = GetMonitorModelDB()
    
        for monitor_section in x_org_config.getSections("Monitor"):
            if monitor_section.identifier==monitor_name:
                model_name = monitor_section.modelname
                if monitor_model_db.getMonitorByName(model_name) is not None:
                    monitor_model = monitor_model_db.getMonitorByName(model_name)
                else:
                    # Create a monitor object for the monitor in the xconfig.
                    # It is probably a Plug N Play monitor and as such doesn't 
                    # appear in our monitor DB.
                    monitor_model = MonitorModel()
                    if model_name is None:
                        model_name = "(unknown)"
                    monitor_model.setName(model_name)
                    monitor_model.setManufacturer(monitor_section.vendorname)
                    #def setEisaId(self,eisaid): self.eisaid = eisaid
                    monitor_model.setHorizontalSync(','.join(monitor_section.getRow('horizsync')))
                    monitor_model.setVerticalSync(','.join(monitor_section.getRow('vertrefresh')))
                    monitor_model.setDpms("dpms" in monitor_section.option)
                    
                return (monitor_section,monitor_model)
        
        #print screen_list
        self.screens = self.gfxcard.getScreens()
        if len(self.screens) ==0:
            print "No screens in XSetup.__init__, that is probably bad"
        else:
            print "No handling for more than one screen yet."
        return (None,None)
    
    def _getFirstXorgScreenName(self,x_screen_entry):
        if len(x_screen_entry._row)==2:
            return x_screen_entry._row[1]
        elif len(x_screen_entry._row)>2 and x_screen_entry._row[1].isdigit():
            return x_screen_entry._row[2]
        else:
            return x_screen_entry._row[1]
    
    def _detectGfxCards(self,pci_bus):
        """Scans the PCI bus for graphics cards.
        
        Returns a list of (PCI_ID, PCIDevice, GfxCard) tuples."""
        self.gfx_card_db = GetGfxCardModelDB()
        
        # Look for a gfxcard.
        found_list = []
        for pci_device in pci_bus.devices:
            print "PCI device", pci_device
            if pci_device.module is not None:
                if pci_device.isGfxCard():
                    pci_id = "PCI:%i:%i:%i" % (pci_device.pci_bus, pci_device.pci_device, pci_device.pci_function)
                    try:
                        cardname = pci_device.getModule()
                        model = self.gfx_card_db.getGfxCardModelByName(cardname)
                    except KeyError:
                        # Exact matching is not enough, try fuzzy detection.
                        model = self._detectGfxCardFuzzy(cardname)
                    found_list.append( (pci_id,pci_device,model) )
        if not len(found_list): 
            self._detectGfxCardFromLspci()
        return found_list

    def _detectGfxCardFuzzy(self, cardname):
        """This does a fuzzy detect, in case exact matching didn't work out 
        It does work for example for ATI Radeon Cards.
        It matches for example "ATI Radeon (fglrx) with ** ATI Radeon (generic) [radeon]
        which should help us out with a lot of more generic chipsets.
        
        Maybe it's not enough to split on (, but it's a good start.
        """
        cardnamepart = cardname.split("(")[0].strip()
        #gfx_card_db = GetGfxCardModelDB()
        print "Fuzzy: ",cardname
        # Matching on less than 6 chars seems too fishy.
        if len(cardnamepart) > 5:
            for card in self.gfx_card_db.getAllGfxCardModelNames():
                if card.find(cardnamepart) > -1:
                    print "Our winner: " + card
                    return self.gfx_card_db.getGfxCardModelByName(card)
        # Bummer, not even fuzzy detection worked. 
        print "Fuzzy Failed."
        return None
        
    def _detectGfxCardFromLspci(self):
        """ This one's use if all other detection fails. Some very new graphics cards can't be 
            matched via /proc/bus/devices, but have information in lspci. These ones should be 
            taken care of here. """
        lspci_pipe = os.popen("/bin/lspci")
        for line in lspci_pipe.readlines():
            if line.split()[1].startswith("VGA"):
                # First part of the string from lspci seems to be the manufacturer, so ...
                mycard = line.split(":")[3].split()[0].strip().lower()
                print "Found VGA: ", mycard
                #TODO: This is not yet used, use it as GfxCard.
                for card in self.gfx_card_db.getAllGfxCardModelNames():
                    if card.find(mycard) > -1:
                        print "hmm ... ", card 
                        return self.gfx_card_db.getGfxCardModelByName(card)

    def _canonicalPCIBusID(self,bus_id):
        try:
            parts = bus_id.split(":")
            if parts[0].lower()!="pci":
                return None
            bus = int(parts[1])
            device = int(parts[2])
            function = int(parts[3])
            return "PCI:%i:%i:%i" % (bus,device,function)
        except IndexError:
            return None
        except ValueError:
            return None
        
    def isXOrgConfigChanged(self):
        """Check if the xorg.config needs to updated
        
        Returns True if the xorg.config file needs to updated to reflect new changes.
        """
        changed = False
        for gfxcard in self.gfxcards:
            changed = changed or gfxcard.isXOrgConfigChanged()
        for screen in self.getAllScreens():
            changed = changed or screen.isXOrgConfigChanged()
        return changed
        
    def writeXOrgConfig(self,filename):
        self.xorg_config.writeConfig(filename)
       
    def getAllScreens(self):
        """Returns screen sections of current setup."""
        screens = []
        for card in self.gfxcards:
            for screen in card.getScreens():
                if screen.inUse():
                    screens.append(screen)
        return screens
        
    def getScreen(self,screenindex):
        return self.getAllScreens()[screenindex]
        
    def getGfxCards(self):
        return self.gfxcards[:] # No messin' with the gfx card list.

    # Dualhead ----------
    def setUseDualhead(self,bool):
        """ Tell the XSetup to use second head """
        self.dualhead = bool
        
    def getUseDualhead(self):
        """ Tells if the current setup is using the second head """
        return self.dualhead
        
    def maySetDualhead(self):
        """ Tells whether the current permission context can enable dualhead features """
        # FIXME: if xconfig is dualhead enabled or user is root
        return True
    
    def setDualheadOrientation(self,orientation):
        """ Sets orientation of monitor to one of 
            XSetup.ABOVE, XSetup.UNDER, XSetup.LEFTOF, XSetup.RIGHTOF
        """
        self.orientation = orientation
        
    def getDualheadOrientation(self):
        """ Returns the current orientation, one of 
            XSetup.ABOVE, XSetup.UNDER, XSetup.LEFTOF, XSetup.RIGHTOF
        """
        return self.orientation
        
    def setDualheadOffset(self,offset):
        """ Sets the offset of screen[1] relative to screen[0]. The absolute position is 
            also dependent on the current orientation 
            FIXME: Need return method to give absolute position from both settings.
        """
        x_offset,y_offset = offset
        self.x_offset = x_offset
        self.y_offset = y_offset
    
    def getDualheadOffset(self):
        return (x_offset,y_offset)
    
    def isHWAccelerated(self):
        # FIXME:
        # if twinview-alike and screen[0].res = screen[1].res
        # else: if primary screen
        return True

    # Internal ----------
    def _addScreen(self,screen):
        self.screens.append(screen)
        
    def _addGfxCard(self,gfxcard):
        self.gfxcards.append(gfxcard)
        
    def __str__(self):
        string  = "XSetup:\n"
        i = 1
        for gfxcard in self.gfxcards:
            string += "    Gfxcard %i: %s\n" % (i,str(gfxcard))
            i += 1
        return string

############################################################################
class GfxCard(object):
    """Represents a graphics card that is present in this computer."""
    
    def __init__(self, x_config, pci_id=None, x_device=None, detected_model=None, proprietary_driver=False):
        self.x_config = x_config
        self.x_device = x_device
        
        if x_device is not None:
            if x_device.boardname is not None:
                self.gfxcard_model = GetGfxCardModelDB().getGfxCardModelByName(x_device.boardname)
            else:
                self.gfxcard_model = detected_model
                
        else:
            self.gfxcard_model = detected_model
        
        self.pci_id = pci_id
        self.screens = []
        self.detected_model = detected_model
        self.original_gfxcard_model = detected_model
        self.proprietary_driver = proprietary_driver 
        self.original_proprietary_driver = self.proprietary_driver
        
    def getScreens(self):
        return self.screens[:]

    def getGfxCardModel(self):
        return self.gfxcard_model
        
    def setGfxCardModel(self,gfx_card_model):
        self.gfxcard_model = gfx_card_model
        self.__updateXConfig(self.gfxcard_model, self.proprietary_driver)
        
    def isProprietaryDriver(self):
        return self.proprietary_driver
        
    def setProprietaryDriver(self,proprietary_driver):
        self.proprietary_driver = proprietary_driver
        self.__updateXConfig(self.gfxcard_model, self.proprietary_driver)
        
    def getDetectedGfxCardModel(self):
        return self.detected_model

    def isXOrgConfigChanged(self):
        return self.original_gfxcard_model is not self.gfxcard_model or \
            self.original_proprietary_driver != self.proprietary_driver
    
    def _addScreen(self,screen):
        self.screens.append(screen)
        
    def __str__(self):
        screen_string = ",".join([str(s) for s in self.screens])
        return "GfxCard: {model:"+str(self.gfxcard_model)+",screens:"+screen_string+"}"
    
    def __updateXConfig(self,model,proprietary_driver):
        
        identifier = self.x_device.identifier
        busid = self.x_device.busid
        
        self.x_device.clear()
        
        # Create a new Device section in the Xorg config file.
        self.x_device.identifier = identifier
        self.x_device.boardname = model.getName()
        self.x_device.busid = self.pci_id
        
        if model.getVendor() is not None:
            self.x_device.vendorname = model.getVendor()
        
        if proprietary_driver and model.getProprietaryDriver() is not None:
            driver = model.getProprietaryDriver()
        else:
            driver = model.getDriver()
        self.x_device.driver = driver
        
        if driver=='fglrx':
            # Default options for the ATI proprietary driver.
            self._insertRawLinesIntoConfig(self.x_device,self.ati_fglrx_lines)
        
        # $raw_X->set_devices($card, @{$card->{cards} || []});
        # $raw_X->get_ServerLayout->{Xinerama} = { commented => !$card->{Xinerama}, Option => 1 }
        #if defined $card->{Xinerama};
        module = self.x_config.getSections('module')[0]
        module.removeModule('GLcore')
        module.removeModule('glx')
        module.removeModule("/usr/X11R6/lib/modules/extensions/libglx.so")
        if driver=='nvidia':
            # This loads the NVIDIA GLX extension module.
            # IT IS IMPORTANT TO KEEP NAME AS FULL PATH TO libglx.so ELSE
            # IT WILL LOAD XFree86 glx module and the server will crash.
            module.addModule("/usr/X11R6/lib/modules/extensions/libglx.so")
            # FIXME lib64
        elif model.getProprietaryDriver()!='fglrx':
            module.addModule('glx')
            module.addModule('GLcore')

        module.removeModule("/usr/X11R6/lib/modules/extensions/libglx.a")
        if driver!='nvidia':
            module.addModule("/usr/X11R6/lib/modules/extensions/libglx.a")
            
        # DRI
        module.removeModule('dri')
        if model.getDriGlx():
            module.addModule('dri')

        module.removeModule('v4l')
        if not (model.getDriGlx() and model.getDriver()=='r128'):
            module.addModule('v4l')
            
        self._insertRawLinesIntoConfig(self.x_device, model.getLines())
        
    def _insertRawLinesIntoConfig(self,section,lines):
        reader = csv.reader(lines,delimiter=' ')
        for row in reader:
            if len(row)>=2:
                if row[0].lower()=="option":
                    option = section.option.makeLine(None,row[1:])
                    section.option.append(option)

    ati_fglrx_lines = """# === disable PnP Monitor  ===
#Option                              "NoDDC"
# === disable/enable XAA/DRI ===
Option "no_accel"                   "no"
Option "no_dri"                     "no"
# === FireGL DDX driver module specific settings ===
# === Screen Management ===
Option "DesktopSetup"               "0x00000000" 

Option "MonitorLayout"              "AUTO, AUTO"
Option "IgnoreEDID"                 "off"
Option "HSync2"                     "unspecified" 
Option "VRefresh2"                  "unspecified" 
Option "ScreenOverlap"              "0" 
# === TV-out Management ===
Option "NoTV"                       "yes"     
Option "TVStandard"                 "NTSC-M"     
Option "TVHSizeAdj"                 "0"     
Option "TVVSizeAdj"                 "0"     
Option "TVHPosAdj"                  "0"     
Option "TVVPosAdj"                  "0"     
Option "TVHStartAdj"                "0"     
Option "TVColorAdj"                 "0"     
Option "GammaCorrectionI"           "0x00000000"
Option "GammaCorrectionII"          "0x00000000"
# === OpenGL specific profiles/settings ===
Option "Capabilities"               "0x00000000"
# === Video Overlay for the Xv extension ===
Option "VideoOverlay"               "on"
# === OpenGL Overlay ===
# Note: When OpenGL Overlay is enabled, Video Overlay
#       will be disabled automatically
Option "OpenGLOverlay"              "off"
Option "CenterMode"                 "off"
# === QBS Support ===
Option "Stereo"                     "off"
Option "StereoSyncEnable"           "1"
# === Misc Options ===
Option "UseFastTLS"                 "0"
Option "BlockSignalsOnLock"         "on"
Option "UseInternalAGPGART"         "no"
Option "ForceGenericCPU"            "no"
# === FSAA ===
Option "FSAAScale"                  "1"
Option "FSAADisableGamma"           "no"
Option "FSAACustomizeMSPos"         "no"
Option "FSAAMSPosX0"                "0.000000"
Option "FSAAMSPosY0"                "0.000000"
Option "FSAAMSPosX1"                "0.000000"
Option "FSAAMSPosY1"                "0.000000"
Option "FSAAMSPosX2"                "0.000000"
Option "FSAAMSPosY2"                "0.000000"
Option "FSAAMSPosX3"                "0.000000"
Option "FSAAMSPosY3"                "0.000000"
Option "FSAAMSPosX4"                "0.000000"
Option "FSAAMSPosY4"                "0.000000"
Option "FSAAMSPosX5"                "0.000000"
Option "FSAAMSPosY5"                "0.000000"
"""
    
############################################################################
class Screen(object):
    """Represents a single output/screen/monitor on a graphics card.
    
    Changes to the screen resolution, refresh rate, rotation and reflection
    settings are not made active until the method applyResolutionSettings() is
    called. After calling applyResolutionSettings(), changes can be backed out
    of with the revertResolutionSettings() method. If you, should I say the user,
    is satisfied with the new settings then call the acceptResolutionSettings()
    method.
    
    Gamma correction settings take effect immediately, and don't take part in the
    apply, revert and accept mechanism above.
    """
    
    standard_sizes = [
        (640,480),(800,600),(1024,768),
        (1152,864),(1280,768),(1280,800),(1280,1024),
        (1440,900),(1400,1050),(1600,1024),
        (1600,1200),(1680,1050),(1600,1200),
        (1920,1200),(1920,1440),(2048,1536),
        (2560,1600)]
        
    RR_Rotate_0 = xf86misc.XF86Screen.RR_Rotate_0
    RR_Rotate_90 = xf86misc.XF86Screen.RR_Rotate_90
    RR_Rotate_180 = xf86misc.XF86Screen.RR_Rotate_180
    RR_Rotate_270 = xf86misc.XF86Screen.RR_Rotate_270
    RR_Reflect_X = xf86misc.XF86Screen.RR_Reflect_X
    RR_Reflect_Y = xf86misc.XF86Screen.RR_Reflect_Y

    def __init__(self, identifier, screen_index=-1, x_config_screen=None, x_live_screen=None, \
                        x_config_monitor=None, monitor_model=None):
        """Create a Screen object.
        
        This method is private to this module.
        """
        self.identifier = identifier
        
        self.x_config_screen = x_config_screen
        self.screen_index = screen_index
        self.x_live_screen = x_live_screen
        self.x_config_monitor = x_config_monitor
        self.monitor_model = monitor_model
        self.original_monitor_model = monitor_model
        
        if self.x_live_screen is not None:
        
            # Cookup some sensible screen sizes.
            self.available_sizes = []
            widths = {}
            for size in self.x_live_screen.getAvailableSizes():
                (pw,ph,wm,hm) = size
                if (pw,ph) in Screen.standard_sizes:
                    self.available_sizes.append( (pw,ph) )
                    widths[pw] = ph
            for size in self.x_live_screen.getAvailableSizes():
                (pw,ph,wm,hm) = size
                if pw>=640 and ph>=480 and pw not in widths:
                    self.available_sizes.append( (pw,ph) )
                    widths[pw] = ph
            self.available_sizes.sort()
            
            (cw,ch,x,x) = self.x_live_screen.getSize()
            i = 0
            print "available", self.available_sizes
            for size in self.available_sizes:
                print size
                if (cw,ch)==size:
                    self.currentsizeindex = self.originalsizeindex = i
                    print "Current::", i
                    break
                i += 1
            
            self.currentrefreshrate = self.originalrefreshrate = self.x_live_screen.getRefreshRate()
            self.currentrotation = self.originalrotation = self.x_live_screen.getRotation() & (
                Screen.RR_Rotate_0 | Screen.RR_Rotate_90 | Screen.RR_Rotate_180 | Screen.RR_Rotate_270)
            self.currentreflection = self.originalreflection = self.x_live_screen.getRotation() & (
                                                            Screen.RR_Reflect_X | Screen.RR_Reflect_Y)
    
            # Read the current gamma settings directly from X.
            (self.redgamma, self.greengamma, self.bluegamma) = self.x_live_screen.getGamma()
            # Round the values off to 2 decimal places.
            (self.redgamma, self.greengamma, self.bluegamma)  = (
                            round(self.redgamma,2), round(self.greengamma,2) , round(self.bluegamma,2) ) 
            
            self.allgamma = self.redgamma
            self.settingall = self.redgamma==self.greengamma==self.bluegamma
    
            (self.originalredgamma, self.originalgreengamma, self.originalbluegamma) = (
                                                        self.redgamma, self.greengamma, self.bluegamma)
            self.originalallgamma = self.allgamma
            self.originalsettingall = self.settingall
    
    def getIdentifier(self):
        return self.identifier
        
    def inUse(self):
        """Returns True if this screen is currently being used by the X server.
        """
        return self.x_config_screen is not None
    
    def getMonitorModel(self):
        """
        
        Returns a MonitorModel object or None.
        """
        return self.monitor_model
    
    def setMonitorModel(self,monitor_model):
        """
        
        """
        self.monitor_model = monitor_model
        
        # FIXME maybe we don't have a X config monitor section.
        
        # Empty the monitor section and fill it in again.
        monitor_identifier = self.x_config_monitor.identifier
        self.x_config_monitor.clear()
        self.x_config_monitor.identifier = monitor_identifier

        self.x_config_monitor.vendorname = monitor_model.getManufacturer()
        self.x_config_monitor.modelname = monitor_model.getName()
    
        hsyncline = self.x_config_monitor.makeLine(None,['HorizSync',monitor_model.getHorizontalSync()])
        self.x_config_monitor.append(hsyncline)

        vsyncline = self.x_config_monitor.makeLine(None,['VertRefresh',monitor_model.getVerticalSync()])
        self.x_config_monitor.append(vsyncline)
    
        # Add a bunch of standard mode lines.
        mode_array = GetMonitorModeDB().getAvailableModes(monitor_model)
        for mode_size in mode_array:
            for mode in mode_size:
                modeline_section = self.x_config_monitor.makeSection(mode.name,'mode')
                modeline_section.dotclock = int(math.floor(mode.clock*1000.0))
                
                htimings = modeline_section.makeLine(None,['htimings',
                                mode.hdisp, mode.hsyncstart, mode.hsyncend, mode.htotal])
                modeline_section.append(htimings)
                
                vtimings = modeline_section.makeLine(None,['vtimings',
                                mode.vdisp, mode.vsyncstart, mode.vsyncend, mode.vtotal])
                modeline_section.append(vtimings)
                
                modeline_section.flags = mode.flags
                self.x_config_monitor.append(modeline_section)
        
    def isXOrgConfigChanged(self):
        return self.original_monitor_model is not self.monitor_model
        
    def getRedGamma(self):
        """Get the current red gamma value.
        """
        return self.redgamma
        
    def setRedGamma(self,value):
        """Set gamma correction value for red
        
        This method takes effect immediately on the X server.
        
        Keyword arguments:
        value -- gamma correction value (float)
        """
        self.redgamma = value
        self.x_live_screen.setGamma( (self.redgamma,self.greengamma,self.bluegamma) )
        
    def getGreenGamma(self):
        """Get the current green gamma value
        """
        return self.greengamma
        
    def setGreenGamma(self,value):
        """Set gamma correction value for green
        
        This method takes effect immediately on the X server.
        
        Keyword arguments:
        value -- gamma correction value (float)
        """
        self.greengamma = value
        self.x_live_screen.setGamma( (self.redgamma,self.greengamma,self.bluegamma) )
        
    def getBlueGamma(self):
        """Get the current blue gamma value
        """
        return self.bluegamma
        
    def setBlueGamma(self,value):
        """Set gamma correction value for blue
        
        Keyword arguments:
        value -- gamma correction value (float)
        
        This method takes effect immediately on the X server.
        """
        self.bluegamma = value
        self.x_live_screen.setGamma( (self.redgamma,self.greengamma,self.bluegamma) )
        
    def getAllGamma(self):
        """Get the gamma correction value for all colours.
        
        Returns a float.
        
        See isGammaEqual()
        """
        return self.allgamma
        
    def setAllGamma(self,value):
        """Set the gamma correction value for all colours.
        
        Keyword arguments:
        value -- gamma correction value (float)
        """
        self.allgamma = value
        self.x_live_screen.setGamma( (self.allgamma,self.allgamma,self.allgamma) )
    
    def isGammaEqual(self):
        """Test whether each colour is using the same gamma correction value.
        
        Returns True if the gamma value is the same for all colours
        """
        return self.getRedGamma()==self.getGreenGamma()==self.getBlueGamma()

    # FIXME relevant?
    def resetGammaSettings(self):
        (self.redgamma, self.greengamma, self.bluegamma) = (
                self.originalredgamma, self.originalgreengamma, self.originalbluegamma)
        self.allgamma = self.originalallgamma
        self.settingall = self.originalsettingall

    # FIXME relevant?
    def isGammaSettingsChanged(self):
        return self.redgamma != self.originalredgamma or \
            self.greengamma != self.originalgreengamma or \
            self.bluegamma != self.originalbluegamma or \
            self.allgamma != self.originalallgamma or \
            self.settingall != self.originalsettingall
    
    # FIXME dead code?
    def getScreenIndex(self):
        return self.screen_index
        
    # Size and resolution
    def getResolutionIndex(self):
        """Get the current resolution of this screen.
        
        Returns an index into the list of available resolutions. See
        getAvailableResolutions().
        """
        return self.currentsizeindex
        
    def setResolutionIndex(self,index):
        """Set the resolution for this screen.
        
        This method does not take effect immediately, only after applyResolutionSetttings()
        has been called.
        
        Keyword arguments:
        index -- The index of the resolution to use. See getAvailableResolutions().
        """
        self.currentsizeindex = index
        
    def getAvailableResolutions(self):
        """Get the list of available resolutions.
        
        Returns a list of screen (width,height) tuples. width and height are in
        pixels.
        """
        return self.available_sizes[:]

    # Rotation
    def getRotation(self):
        """Get the current rotation settings for this screen.
        
        Returns one of Screen.RR_Rotate_0, Screen.RR_Rotate_90,
        Screen.RR_Rotate_180 or Screen.RR_Rotate_270
        """
        return self.currentrotation
        
    def setRotation(self,rotation):
        """Set the rotation for this screen
        
        This method does not take effect immediately, only after
        applyResolutionSetttings() has been called. See getAvailableRotations()
        for how to find out which rotations are supported.
        
        Keyword arguments:
        rotation -- One of Screen.RR_Rotate_0, Screen.RR_Rotate_90,
                    Screen.RR_Rotate_180 or Screen.RR_Rotate_270
        """
        self.currentrotation = rotation
        
    def getAvailableRotations(self):
        """Get the supported rotations for this screen.
        
        Returns a bitmask of support rotations for this screen. The returned
        integer is the bitwise OR of one or more of the constants here below.
        * Screen.RR_Rotate_0
        * Screen.RR_Rotate_90
        * Screen.RR_Rotate_180
        * Screen.RR_Rotate_270
        """
        return self.x_live_screen.getAvailableRotations() & \
            (self.RR_Rotate_0 | self.RR_Rotate_90 | self.RR_Rotate_180 | self.RR_Rotate_270)
        
    # Reflection
    def getReflection(self):
        """Get the current reflection settings for this screen.
        
        Returns the reflection settings as a bit string. Use Screen.RR_Reflect_X
        and Screen.RR_Reflect_Y as bitmasks to determine which reflections are
        in use.
        """
        return self.currentreflection
        
    def setReflection(self,reflection):
        """Set the reflection settings for this screen.

        This method does not take effect immediately, only after
        applyResolutionSetttings() has been called. See getAvailableReflections()
        for how to find out which rotations are supported.
        
        Keyword arguments:
        reflection -- Bit string (Python integer) of desired reflections.
                      Bitwise OR Screen.RR_Reflect_X and Screen.RR_Reflect_Y
                      to construct the string.
        """
        self.currentreflection = reflection

    def getAvailableReflections(self):
        """Get the supported reflections for this screen.
        
        Returns a bit string (Python integer) of supported reflections. Use
        Screen.RR_Reflect_X and Screen..RR_Reflect_Y as bitmasks to determine
        which reflections are available.
        """        
        return self.x_live_screen.getAvailableRotations() & (self.RR_Reflect_X | self.RR_Reflect_Y)
        
    # Refresh rates
    def getRefreshRateIndex(self):
        """Get the current refresh rate index for this screen.
        
        Returns an index into the list of refresh rates. See getAvailableRefreshRates().
        """
        rates = self.getAvailableRefreshRates()
        i = 0
        for r in rates:
            if r>=self.currentrefreshrate:
                return i
            i += 1
        return len(rates)-1
        
    def setRefreshRateIndex(self,index):
        """Set the refresh rate for this screen.
        
        Keyword arguments:
        index -- Index into the list of refresh rates. See getAvailableRefreshRates().
        """
        self.currentrefreshrate = self.getAvailableRefreshRates()[index]

    def getAvailableRefreshRates(self):
        """Get the list of available refresh rates
        
        Get the list of available refresh rates for the currently selected
        resolution. See setResolutionIndex().
        
        Returns a list of integers in Hz.
        """
        isize = self.available_sizes[self.currentsizeindex]
        j = 0
        for size in self.x_live_screen.getAvailableSizes():
            (sw,sh,wm,hm) = size
            if isize==(sw,sh):
                rates = self.x_live_screen.getAvailableRefreshRates(j)
                rates.sort()
                return rates
            j += 1
        return [] # FIXME this is bad and should not happen.
    
    # Applying changes.
    def applyResolutionSettings(self):
        """Apply any tending resolution changes on the X server

        
        """
        if self.isChanged():
            # Work out what the correct index is for randr.
            (width,height) = self.available_sizes[self.currentsizeindex]
            sizeindex = 0
            for size in self.x_live_screen.getAvailableSizes():
                (pw,ph,wm,hm) = size
                if pw==width and ph==height:
                    break
                sizeindex += 1  
            
            rc = self.x_live_screen.setScreenConfigAndRate(sizeindex, \
                self.currentrotation | self.currentreflection, self.currentrefreshrate)
            #self.originalsizeindex = sizeindex
            
            # FIXME this can fail if the config on the server has been updated.

    def revertResolutionSettings(self):
        """Revert the last resolution change on the X server
        
        """
        # Work out what the correct index is for randr.
        (width,height) = self.availablesizes[self.originalsizeindex]
        sizeindex = 0
        for size in self.x_live_screen.getAvailableResolutions():
            (pw,ph,wm,hm) = size
            if pw==width and ph==height:
                break
            sizeindex += 1  
            
        self.x_live_screen.setScreenConfigAndRate(sizeindex, \
            self.originalrotation | self.originalreflection, self.originalrefreshrate)
        # FIXME this can fail if the config on the server has been updated.
        
    def isChanged(self):
        return self.isResolutionSettingsChanged()
        
    def acceptResolutionSettings(self):
        """Accept the last resolution change
        """
        self.originalsizeindex = self.currentsizeindex
        self.originalrefreshrate = self.currentrefreshrate
        self.originalrotation = self.currentrotation
        self.originalreflection = self.currentreflection

        (self.originalredgamma, self.originalgreengamma, self.originalbluegamma) = (
                                        self.redgamma, self.greengamma, self.bluegamma)
        self.originalallgamma = self.allgamma
        self.originalsettingall = self.settingall
            
    # FIXME needed?
    #def resetResolutionSettings(self):
    #    self.currentsizeindex = self.originalsizeindex
    #    self.currentrefreshrate = self.originalrefreshrate
    #    self.currentrotation = self.originalrotation
    #    self.currentreflection = self.originalreflection
        
    def isResolutionSettingsChanged(self):
        return self.currentsizeindex != self.originalsizeindex or \
            self.currentrefreshrate != self.originalrefreshrate or \
            self.currentrotation != self.originalrotation or \
            self.currentreflection != self.originalreflection
    
    def __str__(self):
        string = str(self.getIdentifier()) + " {"
        if self.inUse():
            string += "Size: "
            string += str(self.getAvailableResolutions()[self.getResolutionIndex()])
            string += " "
        else:
            string += "Not in use, "
        if self.monitor_model is not None:
            string += "Monitor:" + str(self.monitor_model)
        string += "}"
        return string

############################################################################
class GfxCardModel(object):
    """Describes the properties of a particular model of graphics card.
    
    """
    def __init__(self,name):
        self.name = name
        self.vendor = None
        self.server = None
        self.driver = None
        self.proprietarydriver = None
        self.lines = []
        self.seeobj = None
        self.noclockprobe = None
        self.unsupported = None
        self.driglx = None
        self.utahglx = None
        self.driglxexperimental = None        
        self.utahglxexperimental = None
        self.badfbrestore = None
        self.badfbrestoreexf3 = None
        self.multihead = None
        self.fbtvout = None
        
    def getName(self): return self.name

    def setVendor(self,vendor): self.vendor = vendor
    def getVendor(self): return self._get(self.vendor,"getVendor",None)
    def setServer(self,server): self.server = server
    def getServer(self): return self._get(self.server,"getServer",None)
    def setDriver(self,driver): self.driver = driver
    def getDriver(self): return self._get(self.driver,"getDriver",None)
    def setProprietaryDriver(self,proprietarydriver): self.proprietarydriver = proprietarydriver
    def getProprietaryDriver(self): return self._get(self.proprietarydriver,"getProprietaryDriver",None)
    def addLine(self,line): self.lines.append(line)
    def getLines(self):
        if (len(self.lines)==0) and (self.seeobj is not None):
            return self.seeobj.getLines()
        else:
            return self.lines[:] # Copy
    
    def setNoClockProbe(self,noprobe): self.noclockprobe = noprobe
    def getNoClockProbe(self): return self._get(self.noclockprobe,"getNoClockProbe",False)
    def setUnsupported(self,unsupported): self.unsupported = unsupported
    def getUnsupported(self): return self._get(self.unsupported,"getUnsupported",False)
    def setDriGlx(self,on): self.driglx = on
    def getDriGlx(self): return self._get(self.driglx,"getDriGlx",False)
    def setUtahGlx(self,on): self.utahglx = on
    def getUtahGlx(self): return self._get(self.utahglx,"getUtahGlx",False)
    def setDriGlxExperimental(self,on): self.driglxexperimental = on
    def getDriGlxExperimental(self): return self._get(self.driglxexperimental,"getDriGlxExperimental",False)
    def setUtahGlxExperimental(self,on): self.utahglxexperimental = on
    def getUtahGlxExperimental(self): return self._get(self.utahglxexperimental,"getUtahGlxExperimental",False)
    def setBadFbRestore(self,on): self.badfbrestore = on
    def getBadFbRestore(self,proprietary=False):
        if proprietary:
            driver = self.getProprietaryDriver()
        else:
            driver = self.getDriver()
        if driver in ['i810','fbdev','nvidia','vmware']:
            return True
        if self.badfbrestore is not None:
            return self.badfbrestore
        if self.seeobj is not None:
            return self.seeobj.getBadFbRestore(proprietary)
        return False
    def setBadFbRestoreXF3(self,on): self.badfbrestoreexf3 = on
    def getBadFbRestoreXF3(self): return self._get(self.badfbrestoreexf3,"getBadFbRestoreXF3",False)
    def setMultiHead(self,n): self.multihead = n
    def getMultiHead(self): return self._get(self.multihead,"getMultiHead",1)
    def setFbTvOut(self,on): self.fbtvout = on
    def getFbTvOut(self): return self._get(self.fbtvout,"getFbTvOut",False)
    def setSee(self,seeobj): self.seeobj = seeobj
    
    # If the seeobj is set, then all attributes that are not filled in for this
    # instance are inheritted from seeobj.
    def _get(self,attr,meth,default):
        if attr is not None:
            return attr
        if self.seeobj is not None:
            return getattr(self.seeobj,meth)()
        else:
            return default
            
    def __str__(self):
        return self.getName()
        
############################################################################
gfxcard_model_db_instance = None # Singleton.

def GetGfxCardModelDB():
    """Returns a GfxCardModelDB instance.
    """
    global gfxcard_model_db_instance
    # Lazy instantiation.
    if gfxcard_model_db_instance is None:
        gfxcard_model_db_instance = GfxCardModelDB()
    return gfxcard_model_db_instance
        
############################################################################
class GfxCardModelDB(object):
    def __init__(self,filename=None):
        # List of gfx card databases, if order of preference.
        possible_filenames =  ['/usr/share/ldetect-lst/Cards+',
                                '/usr/share/hwdata/Cards',
                                '/usr/X11R6/lib/X11/Cards']
        filename = None
        for file in possible_filenames:
            if os.path.isfile(file):
                print "Found Cards database",file
                filename = file
                break
        else:
            print "Ooops, no Cards database found ..."
            
        # The card DB. A dict mapping card names to card objects.
        self.db = {}
        # The keys in this dict will be vendor names, values are dicts mapping card names to objects.
        self.vendordb = {} 
        self.driverdb = {}
        
        self.drivers = ["ati", "glint", "vga", "fbdev"]
        #FIXME
        #if arch() .startswith("sparc"):
        #    self.drivers.extend(["sunbw2", "suncg14", "suncg3", "suncg6", "sunffb", "sunleo", "suntcx"])
        #else:
        self.drivers.extend(["apm", "ark", "chips", "cirrus", "cyrix", "glide", "i128", "i740",
            "i810", "imstt", "mga", "neomagic", "newport", "nv", "rendition", "r128", "radeon",
            "vesa", "s3", "s3virge", "savage", "siliconmotion", "sis", "tdfx", "tga", "trident",
            "tseng", "vmware"])
        self._checkProprietaryDrivers()
        self._loadDrivers(self.drivers)
        self._loadDB(filename)

    def getGfxCardModelByName(self,name):
        return self.db.get(name)
        
    def getGfxCardModelByDriverName(self,driver_name):
        return self.driverdb.get(driver_name)
        
    def getAllGfxCardModelNames(self):
        return self.db.keys()
        
    def _checkProprietaryDrivers(self):
        # Check for the NVidia driver.
        # FIXME x86_64 => 'lib64'
        if os.path.exists("/usr/X11R6/lib/modules/drivers/nvidia_drv.o") and \
                os.path.islink("/usr/X11R6/lib/modules/extensions/libglx.so"):
            self.drivers.append("nvidia")
        
        # Check for the ATI driver
        if os.path.exists("/usr/X11R6/lib/modules/dri/fglrx_dri.so") and \
            os.path.exists("/usr/X11R6/lib/modules/drivers/fglrx_drv.o"):
            self.drivers.append("fglrx")
            
        # FIXME MATROX_HAL?
        
    def _loadDrivers(self,drivers):
        # Insert the Driver entries.
        for drivername in drivers:
            cardobj = GfxCardModel(drivername)
            cardobj.setDriver(drivername)
            self.db[drivername] = cardobj
            self.driverdb[drivername] = cardobj
        
    def _loadDB(self,filename):
        vendors = ['3Dlabs', 'AOpen', 'ASUS', 'ATI', 'Ark Logic', 'Avance Logic',
            'Cardex', 'Chaintech', 'Chips & Technologies', 'Cirrus Logic', 'Compaq',
            'Creative Labs', 'Dell', 'Diamond', 'Digital', 'ET', 'Elsa', 'Genoa',
            'Guillemot', 'Hercules', 'Intel', 'Leadtek', 'Matrox', 'Miro', 'NVIDIA',
            'NeoMagic', 'Number Nine', 'Oak', 'Orchid', 'RIVA', 'Rendition Verite',
            'S3', 'Silicon Motion', 'STB', 'SiS', 'Sun', 'Toshiba', 'Trident',
            'VideoLogic']

        cardobj = None
        # FIXME the file might be compressed.
        fhandle = open(filename,'r')
        for line in fhandle.readlines():
            line = line.strip()
            if len(line)!=0:
                if line.startswith("#")==False:
                    if line.startswith("NAME"):
                        cardobj = GfxCardModel(line[4:].strip())
                        cardname = cardobj.getName()
                        self.db[cardname] = cardobj
                        
                        # Try to extract a vendor name from the card's name.
                        for vendor in vendors:
                            if vendor in cardname:
                                cardobj.setVendor(vendor)
                                if vendor not in self.vendordb:
                                    self.vendordb[vendor] = {}
                                self.vendordb[vendor][cardname] = cardobj
                                break
                        else:
                            if "Other" not in self.vendordb:
                                self.vendordb["Other"] = {}
                            self.vendordb["Other"][cardname] = cardobj
                        
                    elif line.startswith("SERVER"):
                        cardobj.setServer(line[6:].strip())
                    elif line.startswith("DRIVER2"):
                        driver = line[7:].strip()
                        if driver in self.drivers:
                            cardobj.setProprietaryDriver(driver)
                    elif line.startswith("DRIVER"):
                        cardobj.setDriver(line[6:].strip())
                    elif line.startswith("LINE"):
                        cardobj.addLine(line[4:].strip())
                    elif line.startswith("SEE"):
                        try:
                            cardobj.setSee(self.db[line[3:].strip()])
                        except KeyError:
                            pass
                    elif line.startswith("NOCLOCKPROBE"):
                        cardobj.setNoClockProbe(True)
                    elif line.startswith("UNSUPPORTED"):
                        cardobj.setUnsupported(True)
                    elif line.startswith("DRI_GLX"):
                        cardobj.setDriGlx(True)
                    elif line.startswith("UTAH_GLX"):
                        cardobj.setUtahGlx(True)
                    elif line.startswith("DRI_GLX_EXPERIMENTAL"):
                        cardobj.setDriGlxExperimental(True)
                    elif line.startswith("UTAH_GLX_EXPERIMENTAL"):
                        cardobj.setUtahGlxExperimental(True)
                    elif line.startswith("BAD_FB_RESTORE"):
                        cardobj.setBadFbRestore(True)
                    elif line.startswith("BAD_FB_RESTORE_XF3"):
                        cardobj.setBadFbRestoreXF3(True)
                    elif line.startswith("MULTI_HEAD"):
                        cardobj.setMultiHead(int(line[10:].strip()))
                    elif line.startswith("FB_TVOUT"):
                        cardobj.setFbTvOut(True)                       
        fhandle.close()

############################################################################
class MonitorModel(object):
    def __init__(self):
        self.name = None
        self.manufacturer = None
        self.eisaid = None
        self.horizontalsync = None
        self.verticalsync = None
        self.dpms = False
    def copy(self):
        newmonitor = MonitorModel()
        newmonitor.name = self.name
        newmonitor.manufacturer = self.manufacturer
        newmonitor.eisaid = self.eisaid
        newmonitor.horizontalsync = self.horizontalsync
        newmonitor.verticalsync = self.verticalsync
        newmonitor.dpms = self.dpms
        return newmonitor
    def getName(self): return self.name
    def setName(self,name): self.name = name
    def getManufacturer(self): return self.manufacturer
    def setManufacturer(self,manufacturer): self.manufacturer = manufacturer
    def setEisaId(self,eisaid): self.eisaid = eisaid
    def getEisaId(self): return self.eisaid    
    def setDpms(self,on): self.dpms = on
    def getDpms(self): return self.dpms
    def getHorizontalSync(self): return self.horizontalsync
    def setHorizontalSync(self,horizontalsync): self.horizontalsync = horizontalsync
    def getVerticalSync(self): return self.verticalsync
    def setVerticalSync(self,verticalsync): self.verticalsync = verticalsync
    def __str__(self):
        return "{Name:"+self.name+"}"

############################################################################
monitor_model_db_instance = None # Singleton

def GetMonitorModelDB():
    """Returns a GetMonitorModelDB instance.
    """
    global monitor_model_db_instance
    if monitor_model_db_instance is None:
        monitor_model_db_instance = MonitorModelDB()
    return monitor_model_db_instance

############################################################################
class MonitorModelDB(object):
    def __init__(self,filename="/usr/share/ldetect-lst/MonitorsDB"):
        self.db = {}
        self.vendordb = {}
        self.genericdb = {}
        
        # Plug'n Play is a kind of fake entry for monitors that are detected but unknown.
        # It's frequency info is filled in by hardware detection or from the X server config.
        self._plugnplay = MonitorModel()
        self._plugnplay.setName("Plug'n Play")
        self._plugnplay.setManufacturer(self._plugnplay.getName())
        self.db[self._plugnplay.getName()] = self._plugnplay

        # Special location and handling for Debian
        # FIXME is kxconfig part of the standand Debian KDE install? should be dependancy 
        # be mentioned in the README?
        if not os.path.isfile(filename) and ( 
                os.path.isfile("/usr/share/apps/kxconfig/hwdata/Monitors")):
            # This one is tab-separated! 
            self.load("/usr/share/apps/kxconfig/hwdata/Monitors","\t")
        else:        
            self.load(filename)
            
    def load(self,filename,split=";"):
        fhandle = open(filename,'r')
        for line in fhandle.readlines():
            line = line.strip()
            if len(line)!=0:
                if line.startswith("#")==False:
                    try:
                        parts = line.split(split)
                        monitorobj = MonitorModel()
                        monitorobj.setManufacturer(parts[0].strip())
                        monitorobj.setName(parts[1].strip())
                        monitorobj.setEisaId(parts[2].strip())
                        monitorobj.setHorizontalSync(parts[3].strip())
                        monitorobj.setVerticalSync(parts[4].strip())
                        if len(parts)>=6:
                            monitorobj.setDpms(parts[5].strip()=='1')
                        self.db[monitorobj.getName()] = monitorobj
                        
                        if monitorobj.getManufacturer()=="Generic":
                            self.genericdb[monitorobj.getName()] = monitorobj
                        else:
                            if monitorobj.getManufacturer() not in self.vendordb:
                                self.vendordb[monitorobj.getManufacturer()] = {}
                            self.vendordb[monitorobj.getManufacturer()]\
                                                [monitorobj.getName()] = monitorobj
                        
                    except IndexError:
                        print "Bad monitor line:",line
        fhandle.close()
        
    def getMonitorByName(self,name):
        return self.db.get(name,None)
    
    def detect(self): # FIXME move this into the Screen class.
        """Detect the attached monitor.
        
        Returns a 'monitor' object on success, else None.
        """
        # FIXME do a better job of finding a ddc util, maybe xresprobe should be 
        # used preferably?
        
        if os.path.isfile("/usr/sbin/monitor-edid"):
            # This utility appeared in Mandriva 2005 LE
            output = ExecWithCapture("/usr/sbin/monitor-edid",["monitor-edid","-v"])
            eisaid = None
            hrange = None
            vrange = None
            for line in output.split("\n"):
                if "HorizSync" in line:
                    hrange = line.split()[1]
                elif "VertRefresh" in line:
                    vrange = line.split()[1]
                elif line.startswith("EISA ID:"):
                    eisaid = line[9:]

        elif os.path.isfile("/usr/sbin/ddcxinfos"):
            # This utility _was_ standard on Mandrake 10.1 and earlier.
            output = ExecWithCapture("/usr/sbin/ddcxinfos",["ddcxinfos"])
            eisaid = None
            hrange = None
            vrange = None
            for line in output.split("\n"):
                if "HorizSync" in line:
                    hrange = line.split()[0]
                elif "VertRefresh" in line:
                    vrange = line.split()[0]
                elif "EISA ID=" in line:
                    eisaid = line[line.find("EISA ID=")+8:]
                    
        elif os.path.isfile("/usr/sbin/xresprobe"):
            # on Debian
            """
            "xresprobe's" output looks like this:
            miro.root(/etc/X11): xresprobe dummy
            id:
            res: 1280x1024 1280x960 1024x768 800x600 640x480
            freq: 30-95 50-150
            disptype: crt
            """
            # We pass "dummy" as driver to xresprobe, that seems to work as good 
            # as using the right driver. Weird.
            output = ExecWithCapture("/usr/sbin/xresprobe",["xresprobe", "dummy"])
            eisaid = None
            hrange = None
            vrange = None
            for line in output.split("\n"):
                if line.startswith("id:"):
                    eisaid = line.split(":")[1].strip()
                elif line.startswith("freq:"):
                    hrange = line.split()[1].strip()
                    vrange = line.split()[2].strip()
        
        # Look up the EISAID in our database.
        if eisaid is not None:
            for monitor in self.db:
                if eisaid==self.db[monitor].getEisaId():
                    return monitor
        if hrange is not None and vrange is not None:
            self._plugnplay.setHorizontalSync(hrange)
            self._plugnplay.setVerticalSync(vrange)
            return self._plugnplay
        
############################################################################

SYNC_TOLERANCE = 0.01    # 1 percent 

class ModeLine(object):
    
    XF86CONF_PHSYNC = 0x0001
    XF86CONF_NHSYNC = 0x0002
    XF86CONF_PVSYNC = 0x0004
    XF86CONF_NVSYNC = 0x0008
    XF86CONF_INTERLACE = 0x0010
    XF86CONF_DBLSCAN = 0x0020
    XF86CONF_CSYNC = 0x0040
    XF86CONF_PCSYNC = 0x0080
    XF86CONF_NCSYNC = 0x0100
    XF86CONF_HSKEW = 0x0200     # hskew provided 
    XF86CONF_BCAST = 0x0400
    XF86CONF_CUSTOM = 0x0800    # timing numbers customized by editor
    XF86CONF_VSCAN = 0x1000
    flags = {"interlace": XF86CONF_INTERLACE,
        "doublescan": XF86CONF_DBLSCAN,
        "+hsync": XF86CONF_PHSYNC,
        "-hsync": XF86CONF_NHSYNC,
        "+vsync": XF86CONF_PVSYNC,
        "-vsync": XF86CONF_NVSYNC,
        "composite": XF86CONF_CSYNC,
        "+csync": XF86CONF_PCSYNC,
        "-csync": XF86CONF_NCSYNC }

    # Thanks go out to Redhat for this code donation. =)
    def __init__(self, elements):
        self.name = elements[1].strip('"')
        self.clock = float(elements[2])
        self.hdisp = int(elements[3])
        self.hsyncstart = int(elements[4])
        self.hsyncend = int(elements[5])
        self.htotal = int(elements[6])
        self.vdisp = int(elements[7])
        self.vsyncstart = int(elements[8])
        self.vsyncend = int(elements[9])
        self.vtotal = int(elements[10])

        self.flags = 0
        for i in range(11, len(elements)):
            try:
                self.flags |= ModeLine.flags[string.lower(elements[i])]
            except KeyError:
                pass
    
    def getVRefresh(self):
        vrefresh = self.clock * 1000000.0 / float(self.htotal * self.vtotal)
        if self.flags & ModeLine.XF86CONF_INTERLACE:
            vrefresh = vrefresh * 2.0;
        if self.flags & ModeLine.XF86CONF_DBLSCAN:
            vrefresh = vrefresh / 2.0;
        return vrefresh

    # Bascically copied from xf86CheckModeForMonitor
    def supports(self, monitor_hsync, monitor_vsync):
        hsync = self.clock * 1000 / self.htotal
        for freq in monitor_hsync:
            if hsync > freq[0] * (1.0 - SYNC_TOLERANCE) and hsync < freq[1] * (1.0 + SYNC_TOLERANCE):
                break;
        else:
            return False

        vrefresh = self.getVRefresh()
        for freq in monitor_vsync:
            if vrefresh > freq[0] * (1.0 - SYNC_TOLERANCE) and vrefresh < freq[1] * (1.0 + SYNC_TOLERANCE):
                return True
        return False

############################################################################
monitor_mode_db_instance = None # Singleton

def GetMonitorModeDB():
    """Returns a GetMonitorModeDB instance.
    """
    global monitor_mode_db_instance
    if monitor_mode_db_instance is None:
        monitor_mode_db_instance = MonitorModeDB()
    return monitor_mode_db_instance
    
############################################################################
class MonitorModeDB(object):    
    def __init__(self):
        self.db = {}

        module_dir = os.path.dirname(os.path.join(os.getcwd(),__file__))
        self.load(os.path.join(module_dir,"vesamodes"))
        self.load(os.path.join(module_dir,"extramodes"))
            
    def load(self,filename):
        fd = open(filename, 'r')
        lines = fd.readlines()
        fd.close()
    
        for line in lines:
            if line[0] != "#" and line[0] != '/':
                line = line.strip()
                elements = line.split()
                if line!="":
                    if len(elements) < 11 or string.lower(elements[0]) != "modeline":
                        print "Bad modeline found:",line
                        continue
                    name = elements[1][1:-1]
                    if self.db.has_key(name):
                        self.db[name].append(ModeLine(elements))
                    else:
                        self.db[name] = [ModeLine(elements)]
    
    def getAvailableModes(self,monitor):
        result = []
       
        hsync_list = self._list_from_string(monitor.horizontalsync)
        vsync_list = self._list_from_string(monitor.verticalsync)
        
        for modearray in self.db.values():
            res = []
            for modeline in modearray:
                if modeline.supports(hsync_list, vsync_list):
                    res.append(modeline)
            if len(res)!=0:
                result.append(res)
        return result
       
    def _list_from_string(self,src):
        l = []
        pieces = src.split(",")
        for piece in pieces:
            tmp = string.split(piece, "-")
            if len(tmp) == 1:
                l.append( (float(tmp[0]), float(tmp[0])) )
            else:
                l.append( (float(tmp[0]), float(tmp[1])) )
        return l

############################################################################

def ranges_to_string(array, length):
    stringobj = ""
    for i in range(length):
        r = array[i]
        if stringobj != "":
            stringobj = stringobj + ","
        if r[0] == r[1]:
            stringobj = stringobj + repr(r[0])
        else:
            stringobj = stringobj + repr(r[0]) + "-"  + repr(r[1])
    return stringobj

############################################################################
# FIXME move out into another file.
def ExecWithCapture(command, argv, searchPath = 0, root = '/', stdin = 0,
        catchfd = 1, closefd = -1):

    if not os.access(root + command, os.X_OK):
        raise RuntimeError, command + " can not be run"

    (read, write) = os.pipe()
    childpid = os.fork()
    if (not childpid):
        if (root and root != '/'): os.chroot(root)
        os.dup2(write, catchfd)
        os.close(write)
        os.close(read)
        
        if closefd != -1:
            os.close(closefd)
        if stdin:
            os.dup2(stdin, 0)
            os.close(stdin)
        if (searchPath):
            os.execvp(command, argv)
        else:
            os.execv(command, argv)
        sys.exit(1)
    os.close(write)

    rc = ""
    s = "1"
    while (s):
        select.select([read], [], [])
        s = os.read(read, 1000)
        rc = rc + s

    os.close(read)

    try:
        os.waitpid(childpid, 0)
    except OSError, (errno, msg):
        print __name__, "waitpid:", msg

    return rc


if __name__=='__main__':
    xs = XSetup('xorg.conf')
    print str(xs)
    print xs.getAllScreens()
    
    #screen1 = xs.getGfxCards()[0].getScreens()[0]
    #monitor_db = GetMonitorModelDB()
    #new_model = monitor_db.getMonitorByName('Samsung SyncMaster 15GL')
    #print new_model
    #screen1.setMonitorModel(new_model)
    
    #gfxcard_db = GetGfxCardModelDB()
    #new_gfxcard_model = gfxcard_db.getGfxCardModelByName('NVIDIA GeForce FX (generic)')
    ##'ATI Radeon 8500'
    ##'NVIDIA GeForce FX (generic)'
    #print new_gfxcard_model
    gfx_card = xs.getGfxCards()[0]
    gfx_card.setProprietaryDriver(False)
    #gfx_card.setGfxCardModel(new_gfxcard_model)
    xs.writeXOrgConfig('xorg.conf.test')
    
    
