#!/usr/bin/python

###########################################################################
# displayconfig.py - description                                          #
# ------------------------------                                          #
# begin     : Fri Mar 26 2004                                             #
# copyright : (C) 2004 by Simon Edwards                                   #
# email     : simon@simonzone.com                                         #
#                                                                         #
###########################################################################
#                                                                         #
#   This program is free software; you can redistribute it and/or modify  #
#   it under the terms of the GNU General Public License as published by  #
#   the Free Software Foundation; either version 2 of the License, or     #
#   (at your option) any later version.                                   #
#                                                                         #
###########################################################################

from qt import *
from kdecore import *
from kdeui import *
import xorgconfig
import xf86misc
import string
import math
import os
import select
import sys
import struct
import csv
import time
import signal
import shutil
from ktimerdialog import *
from displayconfigwidgets import *
from displayconfigabstraction import *

programname = "Display and Graphics Configuration"
version = "0.4.0"

# Are we running as a separate standalone application or in KControl?
standalone = __name__=='__main__'

# Running as the root user or not?
isroot = os.getuid()==0

############################################################################
class GfxCardDialog(KDialogBase):
    def __init__(self,parent):
        KDialogBase.__init__(self,parent,None,True,"Choose Graphics Card", 
                            KDialogBase.Ok|KDialogBase.Cancel, KDialogBase.Cancel)
        self.gfxcarddb = None
        self.updatingGUI = True
        self.card2listitem = {}
        
        topbox = QVBox(self)
        topbox.setSpacing(KDialog.spacingHint())
        self.setMainWidget(topbox)
        label = QLabel(topbox)
        label.setText("Select Graphics Card:")
        self.listview = KListView(topbox)
        self.listview.addColumn("")
        self.listview.header().hide()
        self.listview.setRootIsDecorated(True)
        self.connect(self.listview,SIGNAL("selectionChanged(QListViewItem *)"),self.slotListClicked)
        topbox.setStretchFactor(self.listview,1)
        
        self.driver = KListViewItem(self.listview)
        self.driver.setText(0,"Drivers")
        self.driver.setSelectable(False)
        
        self.manufacturer = KListViewItem(self.listview)
        self.manufacturer.setText(0,"Manufacturers")
        self.manufacturer.setSelectable(False)
        
        hbox = QHBox(topbox)
        topbox.setStretchFactor(hbox,0)
        vbox = QVBox(hbox)
        
        self.detected_label = QLabel("",vbox)
        
        self.detected_button = KPushButton(vbox)
        self.detected_button.setText("Select")
        self.connect(self.detected_button,SIGNAL("clicked()"),self.slotSelectDetectedClicked)
        
        spacer = QWidget(vbox)
        vbox.setStretchFactor(self.detected_button,0)
        vbox.setStretchFactor(spacer,1)

        hbox.setStretchFactor(vbox,0)
        spacer = QWidget(hbox)
        hbox.setStretchFactor(spacer,1)
        
        drivergrid = QGrid(2,hbox)
        drivergrid.setSpacing(KDialog.spacingHint())
        QLabel("Driver:",drivergrid)
        self.standarddriverradio = QRadioButton("Standard",drivergrid)
        self.connect(self.standarddriverradio,SIGNAL("clicked()"),self.slotStandardDriverClicked)
        QWidget(drivergrid)
        self.proprietarydriverradio = QRadioButton("Proprietary",drivergrid)
        self.connect(self.proprietarydriverradio,SIGNAL("clicked()"),self.slotProprietaryDriverClicked)
        
        self.updatingGUI = False
        self._setGfxCardDB(GetGfxCardModelDB())
        
    def _setGfxCardDB(self,gfxcarddb):
        self.updatingGUI = True
        self.gfxcarddb = gfxcarddb
        
        # Add the GfxCards under the Manufacturer item.
        keys = gfxcarddb.vendordb.keys()
        keys.sort()
        for key in keys:
            cardkeys = self.gfxcarddb.vendordb[key].keys()
            vendoritem = KListViewItem(self.manufacturer)
            vendoritem.setText(0,key)
            vendoritem.setSelectable(False)
            for cardkey in cardkeys:
                item = KListViewItem(vendoritem)
                item.setText(0,cardkey)
                self.card2listitem[self.gfxcarddb.vendordb[key][cardkey]] = item

        # Add the GfxCard _drivers_ under the Drivers item
        drivers = gfxcarddb.driverdb.keys()
        drivers.sort()
        for driver in drivers:
            driveritem = KListViewItem(self.driver)
            driveritem.setText(0,driver)
            self.card2listitem[gfxcarddb.driverdb[driver]] = driveritem

        self.updatingGUI = False
        
    def do(self,card,proprietarydriver,detected_card):
        self.updatingGUI = True
        item = self.card2listitem[card]
        self.listview.setSelected(item,True)
        self.listview.ensureItemVisible(item)

        if detected_card is None:
            self.detected_button.setEnabled(False)
            self.detected_label.setText("Detected graphics card:\n(unknown)")
        else:
            self.detected_button.setEnabled(True)
            self.detected_label.setText("Detected graphics card:\n'%s'." % detected_card.getName())

        self.__syncDriver(card,proprietarydriver)
        
        self.detected_card = detected_card
        self.selectedcard = card
        self.updatingGUI = False
        
        if self.exec_loop()==QDialog.Accepted:
            return (self.selectedcard, self.proprietarydriverradio.isChecked() and 
                                                    (self.selectedcard is not None) )
        else:
            return (card, proprietarydriver)

    def __syncDriver(self,card,proprietarydriver):
        if card.getProprietaryDriver() is None:
            self.standarddriverradio.setChecked(True)
            self.standarddriverradio.setEnabled(False)
            self.proprietarydriverradio.setEnabled(False)
        else:
            self.standarddriverradio.setEnabled(True)
            self.proprietarydriverradio.setEnabled(True)
            self.standarddriverradio.setChecked(not proprietarydriver)
            self.proprietarydriverradio.setChecked(proprietarydriver)
            
    def slotSelectDetectedClicked(self):
        self.updatingGUI = True
        item = self.card2listitem[self.detected_card]
        self.listview.setSelected(item,True)
        self.listview.ensureItemVisible(item)
        self.selectedcard = self.detected_card
        self.__syncDriver(self.selectedcard, self.proprietarydriverradio.isChecked())
        self.updatingGUI = False

    def slotListClicked(self,item):
        if self.updatingGUI:
            return
            
        for key in self.card2listitem:
            value = self.card2listitem[key]
            if value is item:
                self.selectedcard = key
                self.__syncDriver(self.selectedcard, self.proprietarydriverradio.isChecked())
                
    def slotStandardDriverClicked(self):
            self.proprietarydriverradio.setChecked(False)
            self.standarddriverradio.setChecked(True)
            
    def slotProprietaryDriverClicked(self):
            self.standarddriverradio.setChecked(False)
            self.proprietarydriverradio.setChecked(True)
            
############################################################################
class MonitorDialog(KDialogBase):
    def __init__(self,parent):
        KDialogBase.__init__(self,parent,None,True,"Choose Monitor", 
                                        KDialogBase.Ok|KDialogBase.Cancel, KDialogBase.Cancel)
        self.monitordb = None
        self.selectedmonitor = None
        self.plugnplaymonitor = None
        self.monitor2listitem = {}
        self.updatingGUI = True
        
        topbox = QVBox(self)
        topbox.setSpacing(KDialog.spacingHint())
        self.setMainWidget(topbox)
        label = QLabel(topbox)
        label.setText("Select Monitor:")
        self.listview = KListView(topbox)
        self.listview.addColumn("")
        self.listview.header().hide()
        self.listview.setRootIsDecorated(True)
        self.connect(self.listview,SIGNAL("selectionChanged(QListViewItem *)"),self.slotListClicked)
        
        self.generic = KListViewItem(self.listview)
        self.generic.setText(0,"Generic")
        self.generic.setSelectable(False)
        
        self.manufacturer = KListViewItem(self.listview)
        self.manufacturer.setText(0,"Manufacturers")
        self.manufacturer.setSelectable(False)

        self.plugnplay = KListViewItem(self.listview)
        self.plugnplay.setText(0,"Plug 'n' Play")
        
        
        grid = QGroupBox(4,QGroupBox.Horizontal,topbox)
        grid.setTitle("Details")
        
        label = QLabel(grid)
        label.setText("Horizontal Range:")
        
        self.horizrange = KLineEdit(grid)
        self.horizrange.setReadOnly(True)
        
        label = QLabel(grid)
        label.setText("Vertical Refresh:")

        self.vertrange = KLineEdit(grid)
        self.vertrange.setReadOnly(True)
        
        hbox = QHBox(topbox)

        detectbutton = KPushButton(hbox)
        detectbutton.setText("Detect Monitor") # FIXME better label/text?
        self.connect(detectbutton,SIGNAL("clicked()"),self.slotDetectClicked)

        spacer = QWidget(hbox)
        hbox.setStretchFactor(detectbutton,0)
        hbox.setStretchFactor(spacer,1)
        self.updatingGUI = False

    def setMonitorDB(self,monitordb):
        self.monitordb = monitordb
        
        # Add the Monitors
        vendors = monitordb.vendordb.keys()
        vendors.sort()
        for vendor in vendors:
            monitorkeys = self.monitordb.vendordb[vendor].keys()
            vendoritem = KListViewItem(self.manufacturer)
            vendoritem.setText(0,vendor)
            vendoritem.setSelectable(False)
            for monitorkey in monitorkeys:
                item = KListViewItem(vendoritem)
                item.setText(0,monitorkey)
                self.monitor2listitem[self.monitordb.vendordb[vendor][monitorkey]] = item

        generics = monitordb.genericdb.keys()
        generics.sort()
        for generic in generics:
            genericitem = KListViewItem(self.generic)
            genericitem.setText(0,generic)
            self.monitor2listitem[monitordb.genericdb[generic]] = genericitem
            
    def do(self,monitor):
        """Run the monitor selection dialog.
        
        Parameters:
        
        monitor - Currently selected 'Monitor' object.
        
        Returns the newly selected monitor object.
        """
        self.plugnplaymonitor = monitor.copy()
        self.monitor2listitem[self.plugnplaymonitor] = self.plugnplay
        
        self.selectedmonitor = monitor
        if monitor in self.monitor2listitem:
            item = self.monitor2listitem[monitor]
        else:
            # Not in the DB, assume Plug N Play.
            item = self.plugnplay
        
        self.listview.setSelected(item,True)
        self.listview.ensureItemVisible(item)

        self.updatingGUI = True
        self._syncGUI()
        self.updatingGUI = False

        if self.exec_loop()!=QDialog.Accepted:
            # Dialog was cancelled. Return the original monitor.
            self.selectedmonitor = monitor

        # Take our temp Plug N Play item out of the dict.
        del self.monitor2listitem[self.plugnplaymonitor]
        self.plugnplaymonitor = None
        
        return self.selectedmonitor

    def slotDetectClicked(self):
        detectedmonitor = self.monitordb.detect()
        if detectedmonitor is not None:
            if detectedmonitor not in self.monitordb.db.values():
                # Must be a Plug N Play monitor. Use this as our new PlugNPlay entry.
                del self.monitor2listitem[self.plugnplaymonitor]
                self.plugnplaymonitor = detectedmonitor
                self.monitor2listitem[self.plugnplaymonitor] = self.plugnplay
    
            self.selectedmonitor = detectedmonitor
            self._syncGUI()
        else:
            KMessageBox.error(self, "Sorry, the model and capabilities of your\nmonitor couldn't be detected.", \
                "Monitor detection failed")

    def slotListClicked(self,item):
        if self.updatingGUI:
            return
        self.updatingGUI = True
        for key in self.monitor2listitem:
            value = self.monitor2listitem[key]
            if value is item:
                self.selectedmonitor = key
                break
        self._syncGUI()
        self.updatingGUI = False
    
    def _syncGUI(self):
        if self.selectedmonitor in self.monitor2listitem:
            item = self.monitor2listitem[self.selectedmonitor]
        else:
            item = self.plugnplay
        self.listview.setSelected(item,True)
        self.listview.ensureItemVisible(item)
        self.vertrange.setText(self.selectedmonitor.getVerticalSync())
        self.horizrange.setText(self.selectedmonitor.getHorizontalSync())

############################################################################
if standalone:
    programbase = KDialogBase
else:
    programbase = KCModule
    
############################################################################
class DisplayApp(programbase):
    ########################################################################
    def __init__(self,parent=None,name=None):
        global standalone,isroot,kapp
        if standalone:
            KDialogBase.__init__(self,KJanusWidget.Tabbed,"Display Configuration",\
                KDialogBase.Apply|KDialogBase.User1|KDialogBase.User2|KDialogBase.Close, KDialogBase.Close)
            self.setButtonText(KDialogBase.User1,"Reset")
            self.setButtonText(KDialogBase.User2,"About")
            self.config = KConfig("displayconfigrc")
        else:
            KCModule.__init__(self,parent,name)
            # Create a configuration object.
            self.config = KConfig("displayconfigrc")
            self.setButtons(KCModule.Apply|KCModule.Reset)
            self.aboutdata = MakeAboutData()
            
            # This line has the effect of hiding the "Admin only" message and also forcing
            # the Apply/Reset buttons to be shown. Yippie! Only had to read the source
            # to work that out.
            self.setUseRootOnlyMsg(False) 
            
        KGlobal.iconLoader().addAppDir("guidance")

        global imagedir
        imagedir = str(KGlobal.dirs().findDirs("data","guidance/pics/displayconfig")[0])
        
        self.imagedir = imagedir

        self.xconfigchanged = False
        self.xconfigtested = True
        
        self.availabletargetgammas = ['1.4','1.6','1.8','2.0','2.2','2.4']
        self.lightimages = []
        self.mediumimages = []
        self.darkimages = []

        # X Server stuff
        self.xf86server = xf86misc.XF86Server()
        
        # Lookup location of X configfile
        # FIXME use executeWithCapture
        pipe = os.popen(os.popen("which xset").read()[:-1]+" q")
        for line in pipe.readlines():
            if line.strip().startswith("Config file"):
                self.xconfigpath = line.split(":")[1][:-1].strip()
        print "XConfigpath:",self.xconfigpath
        self.xsetup = XSetup(self.xconfigpath)
        
        self.updatingGUI = True
        self.gfxcarddb = GfxCardModelDB()
        self.monitordb = MonitorModelDB()
        self.monitormodedb = MonitorModeDB()
        
        # initialise some stuff.
        #self.screens = []
        #self.screensections = self.xset.getAllScreens() # TODO: obsolete
        #for i in range(len(self.screensections)):
        #    # Build up our list of screens for the Display comboboxes.
        #    print "Screen : " , i
        #    self.xsetup.addScreen(Screen(i, self.xf86server.screens[i], self.screensections[i].identifier))
        
        self._buildGUI()

        # Work out if the currently running Gfxdriver is safe enough that we
        # can test other drivers at the same time.
        self.badfbrestore = self._badFbRestore()
        self.testbutton.setEnabled(isroot and not self._badFbRestore())
        if isroot and not self._badFbRestore():
            self.testunavailablelabel.hide()
        else:
            self.testunavailablelabel.show()

        # Load up some of our databases, and initialise our state variables.
        if len(self.xsetup.getAllScreens()):
            self.currentsizescreen = self.xsetup.getAllScreens()[0]
            self.currentgammascreen = self.xsetup.getAllScreens()[0]
        else:
            # FIXME
            print "Houston, we have a problem: No screens found in configuration file, exiting. :("
            sys.exit(1)
        
        self.monitordialog.setMonitorDB(self.monitordb)
        
        self.aboutus = KAboutApplication(self)

        # For centering the timed Apply dialog.
        self.applytimerdialog = None
        self.connect(kapp.desktop(), SIGNAL("resized(int)"), self.slotDesktopResized)
        self.applydialogscreenindex = 0
        
        self.__loadImages()
        self._loadConfig()
        self._syncGUI()
        
        if standalone:
            self.enableButton(KDialogBase.User1,False) # Reset button
            self.enableButtonApply(False) # Apply button

        self.updatingGUI = False
        
    def _buildGUI(self):
        global standalone
        if not standalone:
            toplayout = QVBoxLayout( self, 0, KDialog.spacingHint() )
            tabcontrol = KTabCtl(self)
            toplayout.addWidget(tabcontrol)
            toplayout.setStretchFactor(tabcontrol,1)
            
        #--- Size, Orientation and Positioning ---
        tabname = "Size, Orientation && Positioning"
        if standalone:
            sopage = self.addGridPage(1,QGrid.Horizontal,tabname)
            sopage.setSpacing(0)
            self.SizePage = SizeOrientationPage(sopage,self.xsetup)
        else:
            self.SizePage = SizeOrientationPage(tabcontrol,self.xsetup)
        #self.SizePage.setScreen(self.xsetup.getAllScreens())
        
        # Connect all PYSIGNALs from SizeOrientationPage Widget to appropriate actions.
        self.connect(self.SizePage,PYSIGNAL("dualheadEnabled(bool)"),self.slotDualheadEnabled)
        self.connect(self.SizePage,PYSIGNAL("changedSignal()"),self._sendChangedSignal)
        self.connect(self.SizePage,PYSIGNAL("resolutionChange(int)"),self.slotResolutionChange)
        
        if not standalone:
            tabcontrol.addTab(self.SizePage,tabname)
        
        #--- Color & Gamma tab ---
        if standalone:
            gammapage = self.addVBoxPage("Color && Gamma")
            vbox = QVBox(gammapage)
        else:
            vbox = QVBox(tabcontrol)
            vbox.setMargin(KDialog.marginHint())
        vbox.setSpacing(KDialog.spacingHint())
        
        hbox = QHBox(vbox)
        hbox.setSpacing(KDialog.spacingHint())
        vbox.setStretchFactor(hbox,0)
        label = QLabel(hbox,"textLabel1")
        label.setText("Screen:")
        hbox.setStretchFactor(label,0)
        self.gammadisplaycombobox = QComboBox(0,hbox,"comboBox11")
        hbox.setStretchFactor(self.gammadisplaycombobox,0)
        spacer = QWidget(hbox)
        hbox.setStretchFactor(spacer,1)
        self.connect(self.gammadisplaycombobox,SIGNAL("activated(int)"),self.slotGammaScreenCombobox)
        
        # fill the combobox.
        for screen in self.xsetup.getAllScreens():
            self.gammadisplaycombobox.insertItem(screen.getIdentifier())
        
        nospacervbox = QVBox(vbox)
        hbox = QHBox(nospacervbox)
        spacer = QWidget(hbox)
        hbox.setStretchFactor(spacer,1)
        self.darkpic = QLabel(hbox)
        self.darkpic.setFixedSize(305,105)
        self.mediumpic = QLabel(hbox)
        self.mediumpic.setFixedSize(305,105)
        spacer = QWidget(hbox)
        hbox.setStretchFactor(spacer,1)
        vbox2 = QVBox(nospacervbox)
        hbox = QHBox(vbox2)
        spacer = QWidget(hbox)
        hbox.setStretchFactor(spacer,1)
        self.lightpic = QLabel(hbox)
        self.lightpic.setFixedSize(305,105)
        self.rangepic = QLabel(hbox)
        self.rangepic.setFixedSize(305,105)
        spacer = QWidget(hbox)
        hbox.setStretchFactor(spacer,1)
        
        vbox2.setStretchFactor(hbox,0)

        label = QLabel("Adjust the gamma correction sliders until the color of the squares \n" + \
            "above match their background as closely as possible.",vbox)
        label.setTextFormat(Qt.PlainText)
        vbox2.setStretchFactor(label,1)
        
        spacer = QWidget(vbox)
        vbox2.setStretchFactor(spacer,1)

        sliderspace = QWidget(vbox)
        
        grid = QGridLayout(sliderspace, 9, 4, 0, KDialog.spacingHint())
        grid.setSpacing(KDialog.spacingHint())
        grid.setColStretch(0,0)
        grid.setColStretch(1,0)
        grid.setColStretch(2,0)
        grid.setColStretch(3,1)

        label = QLabel("Gamma correction:",sliderspace)
        grid.addWidget(label, 0, 0)
        
        self.gammaradiogroup = QButtonGroup()
        self.gammaradiogroup.setRadioButtonExclusive(True)
        self.connect(self.gammaradiogroup,SIGNAL("clicked(int)"),self.slotGammaRadioClicked)
        
        self.allradio = QRadioButton(sliderspace)
        grid.addWidget(self.allradio, 0, 1, Qt.AlignTop)
        
        label = QLabel("All:",sliderspace)
        grid.addWidget(label, 0, 2)
        
        self.gammaslider = KDoubleNumInput(0.4, 3.5, 2.0, 0.05, 2, sliderspace, 'gammaslider')
        grid.addMultiCellWidget(self.gammaslider,0,1,3,3)
        self.gammaslider.setRange(0.5, 2.5, 0.05, True)
        self.connect(self.gammaslider, SIGNAL("valueChanged(double)"), self.slotGammaChanged)

        self.componentradio = QRadioButton(sliderspace)
        grid.addWidget(self.componentradio, 2, 1, Qt.AlignTop)

        label = QLabel("Red:",sliderspace)
        grid.addWidget(label, 2, 2)

        self.redslider = KDoubleNumInput(self.gammaslider,0.4, 3.5, 2.0, 0.05, 2, sliderspace, 'redslider')
        grid.addMultiCellWidget(self.redslider,2,3,3,3)
        self.redslider.setRange(0.5, 2.5, 0.05, True)
        self.connect(self.redslider, SIGNAL("valueChanged(double)"), self.slotRedChanged)

        label = QLabel("Green:",sliderspace)
        grid.addWidget(label, 4, 2)
        
        self.greenslider = KDoubleNumInput(self.redslider,0.4, 3.5, 2.0, 0.05, 2, sliderspace, 'greenslider')
        grid.addMultiCellWidget(self.greenslider,4,5,3,3)
        self.greenslider.setRange(0.5, 2.5, 0.05, True)
        self.connect(self.greenslider, SIGNAL("valueChanged(double)"), self.slotGreenChanged)

        label = QLabel("Blue:",sliderspace)
        grid.addWidget(label, 6, 2)

        self.blueslider = KDoubleNumInput(self.greenslider,0.4, 3.5, 2.0, 0.05, 2, sliderspace, 'blueslider')
        grid.addMultiCellWidget(self.blueslider,6,7,3,3)
        self.blueslider.setRange(0.5, 2.5, 0.05, True)
        self.connect(self.blueslider, SIGNAL("valueChanged(double)"), self.slotBlueChanged)

        self.gammaradiogroup.insert(self.allradio,0)
        self.gammaradiogroup.insert(self.componentradio,1)

        label = QLabel("Target gamma:",sliderspace)
        grid.addWidget(label, 8, 0)

        hbox = QHBox(sliderspace)
        self.targetgammacombo = KComboBox(False,hbox)
        self.targetgammacombo.insertItem('1.4')
        self.targetgammacombo.insertItem('1.6')
        self.targetgammacombo.insertItem('1.8 Apple Macintosh standard')
        self.targetgammacombo.insertItem('2.0 Recommend')
        self.targetgammacombo.insertItem('2.2 PC standard, sRGB')
        self.targetgammacombo.insertItem('2.4')
        hbox.setStretchFactor(self.targetgammacombo,0)
        spacer = QWidget(hbox)
        hbox.setStretchFactor(spacer,1)
        grid.addMultiCellWidget(hbox, 8, 8, 1, 3)
        
        self.connect(self.targetgammacombo,SIGNAL("activated(int)"),self.slotTargetGammaChanged)

        spacer = QWidget(vbox)
        vbox.setStretchFactor(spacer,1)

        if not standalone:
            tabcontrol.addTab(vbox,"Color && Gamma")
        
        #--- Hardware tab ---
        if standalone:
            hardwarepage = self.addVBoxPage("Hardware")
            vbox = QVBox(hardwarepage)
        else:
            vbox = QVBox(tabcontrol)
            vbox.setMargin(KDialog.marginHint())
        self.gfxcarddialog = GfxCardDialog(self)
        self.monitordialog = MonitorDialog(self)

        self.xscreenwidgets = []
        
        for gfxcard in self.xsetup.getGfxCards():
            w = GfxCardWidget(vbox,self.xsetup, gfxcard, self.gfxcarddialog, self.monitordialog)
            self.xscreenwidgets.append(w)
            self.connect(w,PYSIGNAL("xconfigChanged"),self.slotXConfigChanged)
            
        spacer = QWidget(vbox)
        vbox.setStretchFactor(spacer,1)

        if not isroot:
            QLabel("Changes on this tab require 'root' access.",vbox)
            if not standalone:
                QLabel("Click the \"Administrator Mode\" button to allow modifications on this tab.",vbox)
        
        hbox = QHBox(vbox)
        hbox.setSpacing(KDialog.spacingHint())
        self.testbutton = KPushButton("Test",hbox)
        self.connect(self.testbutton,SIGNAL("clicked()"),self.slotTestClicked)
        hbox.setStretchFactor(self.testbutton,0)
        
        self.testunavailablelabel = QHBox(hbox)
        self.testunavailablelabel.setSpacing(KDialog.spacingHint())
        tmplabel = QLabel(self.testunavailablelabel)
        self.testunavailablelabel.setStretchFactor(tmplabel,0)
        tmplabel.setPixmap(SmallIcon('info'))
        label = QLabel("This configuration cannot be safely tested.",self.testunavailablelabel)
        self.testunavailablelabel.setStretchFactor(label,1)
        self.testunavailablelabel.hide()
        
        spacer = QWidget(hbox)
        hbox.setStretchFactor(spacer,1)
        vbox.setStretchFactor(hbox,0)

        if not standalone:
            tabcontrol.addTab(vbox,"Hardware")
            
        #--- Display Power Saving ---
        tabname = "Power saving && screensaver"
        if standalone:
            powerpage = self.addGridPage(1,QGrid.Horizontal,tabname)
            #powerpage.setSpacing(0)
            self.dpmspage = DpmsPage(powerpage)
        else:
            self.dpmspage = DpmsPage(tabcontrol)
        #self.SizePage.setScreens(self.xsetup.getScreens())
        
        # Connect all PYSIGNALs from SizeOrientationPage Widget to appropriate actions.
        #self.connect(self.SizePage,PYSIGNAL("dualheadEnabled(bool)"),self.slotDualheadEnabled)
        self.connect(self.dpmspage,PYSIGNAL("changedSignal()"),self._sendChangedSignal)
        
        if not standalone:
            tabcontrol.addTab(self.dpmspage,tabname)

    def save(self): # KCModule
        # Check the Size & Orientation tab.
        if self.applytimerdialog is None:
            self.applytimerdialog = KTimerDialog(15000, KTimerDialog.CountDown, self, "mainKTimerDialog",
                True, "Confirm Display Setting Change", KTimerDialog.Ok | KTimerDialog.Cancel, \
                KTimerDialog.Cancel)
            self.applytimerdialog.setButtonOK(KGuiItem("&Keep", "button_ok"))
            self.applytimerdialog.setButtonCancel(KGuiItem("&Cancel", "button_cancel"))
            label = KActiveLabel("Trying new screen settings. Keep these new settings? (Automatically cancelling in 15 seconds.)", \
                self.applytimerdialog, "userSpecifiedLabel")
            self.applytimerdialog.setMainWidget(label)
        
        for s in self.xsetup.getAllScreens():
            if s.isChanged():
                s.apply()
                self.applydialogscreenindex = s.getIndex()
                KDialog.centerOnScreen(self.applytimerdialog, self.applydialogscreenindex)
                if self.applytimerdialog.exec_loop():
                    s.accept()
                else:
                    s.revert()
        self._sendChangedSignal()
        
        # Save the X server config.
        if isroot and self.xconfigchanged:
            if not self.xconfigtested:
                if self.badfbrestore or self._badFbRestore():
                    if KMessageBox.warningContinueCancel(self, \
                            "The selected driver and monitor configuration can not be safely\n" \
                            "tested on this computer.\nContinue with this configuration?", \
                            "Configuration not tested")!=KMessageBox.Continue:
                        return
                else:
                    if KMessageBox.warningContinueCancel(self, \
                            "The selected driver and monitor configuration has not been successfully\n" \
                            "tested on this computer.\nContinue with this configuration?", \
                            "Configuration not tested")!=KMessageBox.Continue:
                        return
            
            try:
                # Backup up the current config file.
                i = 1
                while os.path.exists("%s.%i" % (self.xconfigpath,i)):
                    i += 1
                shutil.copyfile(self.xconfigpath,"%s.%i" % (self.xconfigpath,i))
                
                # Write out the new config
                tmpfilename = self.xconfigpath + ".tmp"
                self.xconfig.writeConfig(tmpfilename)
                os.rename(tmpfilename,self.xconfigpath)
                
                # Let the user know that they should restart the X server.
                KMessageBox.information(self,\
                    "The new settings will take effect after you restart the X server or reboot.",\
                    "Settings","restartxmessage")
            except (IOError,TypeError):
                return
                # FIXME error
                
    # Called when the desktop is resized. Just center the confirm dialog.
    def slotDesktopResized(self):
        if self.applytimerdialog is not None:
            KDialog.centerOnScreen(self.applytimerdialog, self.applydialogscreenindex)

    def slotApply(self): # KDialogBase
        self.save()
        self._saveConfig()
        self.dpmspage.apply()
        
    def load(self): # KCModule
        self.__reset()
        self._syncGUI()
        self._sendChangedSignal()

    def slotUser1(self): # Reset button, KDialogBase
        self.load()
    
    def slotUser2(self): # About button, KDialogBase
        self.aboutus.show()
        
    def slotResolutionChange(self,i):
        print "Caught SIGNAL resolution change"
        self.currentsizescreen.setResolutionIndex(i)
        self._sendChangedSignal()
        
    def slotTargetGammaChanged(self,i):
        self.targetgamma = i
        self._selectGamma(self.targetgamma)
        self._sendChangedSignal()
    
    def slotGammaRadioClicked(self,i):
        self.settingall = i==0
        self.gammaslider.setDisabled(not self.settingall)
        self.redslider.setDisabled(self.settingall)
        self.greenslider.setDisabled(self.settingall)
        self.blueslider.setDisabled(self.settingall)
        
        if self.settingall:
            self.currentgammascreen.setAllGamma(self.currentgammascreen.getAllGamma())
        else:
            self.currentgammascreen.xscreen.setRedGamma(self.currentgammascreen.getRedGamma())
            self.currentgammascreen.xscreen.setGreenGamma(self.currentgammascreen.getGreenGamma())
            self.currentgammascreen.xscreen.setBlueGamma(self.currentgammascreen.getBlueGamma())
        self._sendChangedSignal()
        
    def slotGammaChanged(self,value):
        if self.updatingGUI:
            return
        self.currentgammascreen.setAllGamma(value)
        self._sendChangedSignal()

    def slotRedChanged(self,value):
        if self.updatingGUI:
            return
        self.currentgammascreen.setRedGamma(value)
        self._sendChangedSignal()
        
    def slotGreenChanged(self,value):
        if self.updatingGUI:
            return
        self.currentgammascreen.setGreenGamma(value)
        self._sendChangedSignal()

    def slotBlueChanged(self,value):
        if self.updatingGUI:
            return
        self.currentgammascreen.setBlueGamma(value)
        self._sendChangedSignal()
        
    def slotGammaScreenCombobox(self,i):
        self.currentgammascreen = self.xsetup.getAllScreens()[i]
        self._syncGUI()
        self._sendChangedSignal()

    def slotXConfigChanged(self):
        self.xconfigchanged = True
        self.xconfigtested = False
        
        # Check if the current X config can be tested.
        self._syncTestButton()
        self._sendChangedSignal()
        
    def slotTestClicked(self):
        self.xconfigtested = self.testX()
        
    def slotDualheadEnabled(self, enabled):
        # FIXME: Do necessary stuff outside of DualheadPage 
        if enabled: print "Dualhead mode is enabled"
        else: print "Dualhead mode is disabled"
        
    def testX(self):
        
        self.xserverbin = "/usr/X11R6/bin/XFree86"
        if not os.path.isfile(self.xserverbin):
            self.xserverbin = "/usr/X11R6/bin/Xorg"
        rc = False
        
        # Remove an stale X server lock
        try: os.remove("/tmp/.X9-lock")
        except OSError: pass
        
        # Try to find a safe tmp dir.
        tmpdir = None
        if os.environ.get("TMPDIR") is not None:
            tmpdir = os.environ.get("TMPDIR")
        if tmpdir is None or not os.path.isdir(tmpdir):
            tmpdir = os.path.join(os.environ.get("HOME"),"tmp")
            if not os.path.isdir(tmpdir):
                tmpdir = "/tmp"
        workingtmpdir = os.path.join(tmpdir,"guidance."+str(os.getpid()))
        errorfilename = os.path.join(workingtmpdir,"testserver.xoutput")
        configfilename = os.path.join(workingtmpdir,"testserver.config")

        print "errorfile is",errorfilename
        print "configfile is",configfilename
        
        # Start the Xserver up with the new config file.
        try:
            # Create our private dir.
            os.mkdir(workingtmpdir,0700)
            
            # Write out the new config file.
            self.xsetup.writeXOrgConfig(configfilename)
        
            os.system("xauth add :9 . `mcookie`")
            # FIXME:: -xf86config is nowhere in man X ??
            pid = os.spawnv(os.P_NOWAIT,"/bin/bash",\
                ["bash","-c","exec %s :9 -xf86config %s &> %s" % (self.xserverbin,configfilename,errorfilename)])
            print "Got pid",pid

            # Wait for the server to show up.
            print str(os.waitpid(pid,os.WNOHANG))
            
            time.sleep(1) # Wait a sec.
            testserver = None
            while True:
                # Try connecting to the server.
                try:
                    testserver = xf86misc.XF86Server(":9")
                    break
                except xf86misc.XF86Error:
                    testserver = None
                # Check if the server process is still alive.
                if os.waitpid(pid,os.WNOHANG) != (0,0):
                    break
                time.sleep(1) # Give the server some more time.
            
            print "checkpoint 1"
            print str(testserver)
            
            if testserver is not None:
                # Start the timed popup on the :9 display.
                servertestpy = str(KGlobal.dirs().findResource("data","guidance/servertestdialog.py"))
                pythonexe = str(KStandardDirs.findExe("python"))
                testrc = os.system(pythonexe + " " + servertestpy)
                rc = (rc >> 8) == 0 # Test is good if the return code was 0.
                testserver = None
                os.kill(pid,signal.SIGINT)
            else:
                # Server failed, read the error info.
                msg = ""
                try:
                    fhandle = open(errorfilename,'r')
                    for line in fhandle.readlines():
                        if (line.startswith("(EE)") and ("Disabling" not in line)) or line.startswith("Fatal"):
                            msg += line
                    msg = "Messages from the X server:\n" + msg
                except IOError:
                    msg += "Sorry, unable to capture the error messages from the X server."
                KMessageBox.detailedSorry(self,"Sorry, this configuration video card driver\nand monitor doesn't appear to work.",msg)

        finally:
            # Attempt some cleanup before we go.
            try: os.remove(errorfilename)
            except OSError: pass
            try: os.remove(configfilename)
            except OSError: pass
            try: os.rmdir(workingtmpdir)
            except OSError: pass
            
        return rc
        
    def _syncGUI(self):
        self.SizePage._syncGUI()

        # Sync the gamma tab.
        self.targetgammacombo.setCurrentItem(self.targetgamma)
        self._selectGamma(self.targetgamma)
        
        if self.currentgammascreen.isGammaEqual():
            self.gammaradiogroup.setButton(0)
        else:
            self.gammaradiogroup.setButton(1)
        self.gammaslider.setValue(self.currentgammascreen.getAllGamma())
        self.redslider.setValue(self.currentgammascreen.getRedGamma())
        self.greenslider.setValue(self.currentgammascreen.getGreenGamma())
        self.blueslider.setValue(self.currentgammascreen.getBlueGamma())

        self.gammaslider.setDisabled(not self.currentgammascreen.isGammaEqual())
        self.redslider.setDisabled(self.currentgammascreen.isGammaEqual())
        self.greenslider.setDisabled(self.currentgammascreen.isGammaEqual())
        self.blueslider.setDisabled(self.currentgammascreen.isGammaEqual())
        self._syncTestButton()
        
    def _syncTestButton(self):
        currentbadfbrestore = self._badFbRestore()
        self.testbutton.setEnabled(isroot and not (self.badfbrestore or currentbadfbrestore))
        if not isroot or (isroot and not (self.badfbrestore or currentbadfbrestore)):
            self.testunavailablelabel.hide()
        else:
            self.testunavailablelabel.show()
        
    def _loadConfig(self):
        self.config.setGroup("General")
        t = self.config.readEntry("targetgamma","2.0")
        if t in self.availabletargetgammas:
            t = '2.0'
        self.targetgamma = self.availabletargetgammas.index(t)
        
    def _saveConfig(self):
        self.config.setGroup("General")
        self.config.writeEntry("targetgamma",self.availabletargetgammas[self.targetgamma])
        for s in self.xsetup.getAllScreens():
            self._saveRandRConfig(s)
        self.config.sync()

    def _saveRandRConfig(self,screen):
        w,h = screen.getAvailableResolutions()[screen.getResolutionIndex()]
        self.config.setGroup("Screen"+str(screen.getIndex()))
        self.config.writeEntry("width",w)
        self.config.writeEntry("height",h)
        self.config.writeEntry("reflectX", int( (screen.getReflection() & screen.RR_Reflect_X)!=0) )
        self.config.writeEntry("reflectY", int((screen.getReflection() & screen.RR_Reflect_Y)!=0) )
        self.config.writeEntry("refresh", screen.getAvailableRefreshRates()[screen.getRefreshRateIndex()])
        rotationmap = {screen.RR_Rotate_0: "0", screen.RR_Rotate_90: "90",
                        screen.RR_Rotate_180:"180", screen.RR_Rotate_270: "270"}
        self.config.writeEntry("rotate", rotationmap[screen.getRotation()])
        
    def _selectGamma(self,i):
        self.lightpic.setPixmap(self.lightimages[i])
        self.mediumpic.setPixmap(self.mediumimages[i])
        self.darkpic.setPixmap(self.darkimages[i])
        
    def __loadImages(self):
        for g in ['14','16','18','20','22','24']:
            self.lightimages.append( QPixmap(self.imagedir+'gammapics/Gamma'+g+'.png') )
            self.mediumimages.append( QPixmap(self.imagedir+'gammapics/MGam'+g+'.png') )
            self.darkimages.append( QPixmap(self.imagedir+'gammapics/DGam'+g+'.png') )
        self.rangepic.setPixmap(QPixmap(self.imagedir+'gammapics/ranges.png'))

        self.previewscreen = QPixmap(self.imagedir+'monitor_screen_1280x1024.png')
        self.previewscreenportrait = QPixmap(self.imagedir+'monitor_screen_1024x1280.png')
    
    def __reset(self):
        # Reset the screen settings.
        for s in self.xsetup.getAllScreens():
            s.reset()
        for w in self.xscreenwidgets:
            w.reset()
            
    # Kcontrol expects updates about whether the contents have changed.
    # Also we fake the Apply and Reset buttons here when running outside kcontrol.
    def _sendChangedSignal(self):
        global standalone
        
        changed = False
        for s in self.xsetup.getAllScreens():
            changed = changed or s.isChanged()
        
        changed = self.xsetup.isXOrgConfigChanged()
        if self.dpmspage.isChanged():
            changed = True
            
        if standalone:
            self.enableButton(KDialogBase.User1,changed) # Reset button
            self.enableButtonApply(changed) # Apply button
        else:
            self.emit(SIGNAL("changed(bool)"), (changed,) )

    def _badFbRestore(self):
        bad_fb_restore = False
        for card in self.xsetup.getGfxCards():
            bad_fb_restore = bad_fb_restore or \
                ((card.getGfxCardModel() is not None) and card.getGfxCardModel().getBadFbRestore(card.isProprietaryDriver()))
        return bad_fb_restore
        
############################################################################
class SizeOrientationPage(QWidget):
    """
    A TabPage with all the settings for Size and Orientation of the screens,
    also features Refreshrates and Dualheadsettings.
    
    Emits the following signals:
    # FIXME?    dualheadEnabled(bool) checkbox "Use second monitor" changed.
        self.emit(PYSIGNAL("changedSignal()",())
        ...
        
        TODO:
            * Update __doc__ with emitted signals, connect these.
            * Choose screen (more than one preview)
            * insert second preview if dualhead enabled (SIGNAL?).
            * Select screen by click in Dualhead mode 
            * Relative positioning.
            * Call setRefreshCombo after switching screens.
    """
    def __init__(self,parent,xsetup):
        QWidget.__init__(self,parent)
        
        global imagedir
        self.xsetup = xsetup
        self.imagedir = imagedir
        self.parent = parent
        self.current_screen = self.xsetup.getAllScreens()[0]
        
        self._buildGUI()
        self.setRefreshRatesCombo()
        self.connect(self.MonitorPreview,PYSIGNAL("focussed()"),self.slotMonitorFocussed)
        
        self._syncGUI()
        # FIXME: Set enabled if dualhead is initially enabled.
        self.slotDualheadToggled(False)
    
    def _syncGUI(self):
        # Sync the size tab.
        self.resizeslider.setScreen(self.current_screen)
        # Sync the screen orientation.
        self._syncRANDR()
        
    def _syncRANDR(self):
        self.orientationradiogroup.setButton( \
            [Screen.RR_Rotate_0, Screen.RR_Rotate_90, Screen.RR_Rotate_270, 
                Screen.RR_Rotate_180].index(self.current_screen.getRotation()))
        # This construct above just maps an rotation to a radiobutton index.
        self.mirrorhorizontalcheckbox.setChecked(self.current_screen.getReflection() & Screen.RR_Reflect_X)
        self.mirrorverticalcheckbox.setChecked(self.current_screen.getReflection() & Screen.RR_Reflect_Y)
        self.slotResolutionChange(self.current_screen.originalsizeindex)
        
        # Enable/disable the rotation/reflection widgets.
        self.leftorientationradio.setEnabled((self.current_screen.getAvailableRotations() & Screen.RR_Rotate_90)!=0)
        self.rightorientationradio.setEnabled((self.current_screen.getAvailableRotations() & Screen.RR_Rotate_270)!=0)
        self.upsideorientationradio.setEnabled((self.current_screen.getAvailableRotations() & Screen.RR_Rotate_180)!=0)
        self.mirrorhorizontalcheckbox.setEnabled((self.current_screen.getAvailableReflections() & Screen.RR_Reflect_X)!=0)
        self.mirrorverticalcheckbox.setEnabled((self.current_screen.getAvailableRotations() & Screen.RR_Reflect_Y)!=0)
        
    def enableRANDR(self,enabled=True):
        if enabled:
            self.OrientationGroupBox.setEnabled(enabled)
            self._syncRANDR()
        else:
            self.orientationradiogroup.setButton(0)
            self.mirrorhorizontalcheckbox.setChecked(False)
            self.mirrorverticalcheckbox.setChecked(False)
            self.OrientationGroupBox.setEnabled(False)
    
    def setRefreshRatesCombo(self):
        """ Update refresh combobox """
        self.sizerefreshcombo.clear()
        for rate in self.current_screen.getAvailableRefreshRates():
            self.sizerefreshcombo.insertItem("%i Hz" % rate)
            print rate
        self.sizerefreshcombo.setCurrentItem(self.current_screen.getRefreshRateIndex())

    def slotMonitorFocussed(self):
        print "Monitor is focussed."
        self.resizeslider.setTitle("Screen size: " + self.dualmonitorpreview.getCurrentLabel())
        # TODO: DualMonitorPreview does not yet return the correct screen
        self.current_screen = self.dualmonitorpreview.getCurrentScreen()
        # FIXME: set currentsizescreen to newly selected one (can get that one from MonitorPreview, probably)

    def _sendChangedSignal(self):
        self.emit(PYSIGNAL("changedSignal()"),())
    
    def _updatePreview(self):
        if self.current_screen.getRotation()==Screen.RR_Rotate_0:
            self.MonitorPreview.setRotation(self.MonitorPreview.ROT_NORMAL)
        elif self.current_screen.getRotation()==Screen.RR_Rotate_90:
            self.MonitorPreview.setRotation(self.MonitorPreview.ROT_LEFT)
        elif self.current_screen.getRotation()==Screen.RR_Rotate_270:
            self.MonitorPreview.setRotation(self.MonitorPreview.ROT_RIGHT)
        else:
            self.MonitorPreview.setRotation(self.MonitorPreview.ROT_UPSIDEDOWN)
        self.MonitorPreview.changeResolution( \
            self.current_screen.getAvailableResolutions()[self.resizeslider.value()])

    def _buildGUI(self):
        """ Assemble all GUI elements """
        # Layout stuff.
        SizeOrientationTabLayout = QGridLayout(self,1,1,0,0,"SizeOrientationTabLayout")
        SizeOrientationTabLayout.setColStretch(1,1)
        LayoutPage = QHBoxLayout(None,0,6,"LayoutPage")
        
        # -- Left column with orientation and dualhead box.
        LayoutOrientDualhead = QVBoxLayout(None,0,6,"LayoutOrientDualhead")

        # -- Orientation group
        self.OrientationGroupBox = QVGroupBox(self)
        self.OrientationGroupBox.setTitle("Monitor Orientation")
        self.OrientationGroupBox.setInsideSpacing(KDialog.spacingHint())
        self.OrientationGroupBox.setInsideMargin(KDialog.marginHint())
        self.orientationradiogroup = QButtonGroup()
        self.orientationradiogroup.setRadioButtonExclusive(True)
 
        self.normalorientationradio = QRadioButton(self.OrientationGroupBox)
        self.normalorientationradio.setText("Normal")
        self.leftorientationradio = QRadioButton(self.OrientationGroupBox)
        self.leftorientationradio .setText("Left edge on top")
        self.rightorientationradio = QRadioButton(self.OrientationGroupBox)
        self.rightorientationradio.setText("Right edge on top")
        self.upsideorientationradio = QRadioButton(self.OrientationGroupBox)
        self.upsideorientationradio.setText("Upsidedown")

        self.mirrorhorizontalcheckbox = QCheckBox(self.OrientationGroupBox)
        self.mirrorhorizontalcheckbox.setText("Mirror horizontally")
        self.connect(self.mirrorhorizontalcheckbox,SIGNAL("toggled(bool)"),self.slotMirrorHorizontallyToggled)
        
        self.mirrorverticalcheckbox = QCheckBox(self.OrientationGroupBox)
        self.mirrorverticalcheckbox.setText("Mirror vertically")
        self.connect(self.mirrorverticalcheckbox,SIGNAL("toggled(bool)"),self.slotMirrorVerticallyToggled)
        
        self.orientationradiogroup.insert(self.normalorientationradio,0)
        self.orientationradiogroup.insert(self.leftorientationradio,1)
        self.orientationradiogroup.insert(self.rightorientationradio,2)
        self.orientationradiogroup.insert(self.upsideorientationradio,3)
        LayoutOrientDualhead.addWidget(self.OrientationGroupBox)
        self.connect(self.orientationradiogroup,SIGNAL("clicked(int)"),self.slotOrientationRadioClicked)

        # -- Dualhead Box.
        self.DualheadBox = QGroupBox(self,"DualheadBox")
        self.DualheadBox.setCheckable(True)
        self.DualheadBox.setChecked(False)
        self.DualheadBox.setColumnLayout(0,Qt.Vertical)
        self.DualheadBox.layout().setSpacing(6)
        self.DualheadBox.layout().setMargin(11)
        self.DualheadBox.setTitle("Use two monitors")
        self.connect(self.DualheadBox,SIGNAL("toggled(bool)"),self.slotDualheadToggled)
        
        DualheadBoxLayout = QVBoxLayout(self.DualheadBox.layout())
        DualheadBoxLayout.setAlignment(Qt.AlignTop)

        LayoutDualheadList = QVBoxLayout(None,0,6,"LayoutDualheadList")
        
        self.LinkScreenSizesCheck = QCheckBox(self.DualheadBox,"LinkScreenSizesCheck")
        self.LinkScreenSizesCheck.setText("Link screen sizes")
        LayoutDualheadList.addWidget(self.LinkScreenSizesCheck)

        self.IdentifyButton = QPushButton(self.DualheadBox,"IdentifyButton")
        self.IdentifyButton.setText("Identify monitors")
        LayoutDualheadList.addWidget(self.IdentifyButton)
        DualheadBoxLayout.addLayout(LayoutDualheadList)
        LayoutOrientDualhead.addWidget(self.DualheadBox)

        # -- Notification box.
        self.notifybox = QGroupBox(self,"notifybox")
        self.notifybox.setTitle("Information:")
        self.notifybox.setColumnLayout(0,Qt.Vertical)
        self.notifybox.layout().setSpacing(6)
        self.notifybox.layout().setMargin(11)
        self.notifybox.setAlignment(Qt.AlignTop)
        
        notifyboxlayout = QVBoxLayout(self.notifybox.layout())
        self.notify = QLabel(self.notifybox)
        self.notify.setText("Here go warning.\n\n\n\nAnother one.")
        notifyboxlayout.addWidget(self.notify)
        
        LayoutOrientDualhead.addWidget(self.notifybox)

        LeftVerSpacer = QSpacerItem(21,180,QSizePolicy.Minimum,QSizePolicy.Expanding)
        LayoutOrientDualhead.addItem(LeftVerSpacer)
        LayoutPage.addLayout(LayoutOrientDualhead)
        
        # -- Right columns with preview, size and refresh widgets.
        LayoutPreviewSize = QVBoxLayout(None,0,6,"LayoutPreviewSize")
        
        # -- Preview Box.
        self.MonitorPreviewStack = QWidgetStack(self)
        
        self.MonitorPreview = MonitorPreview(self,self.imagedir)
        # FIXME: Set Resolutions.
        #self.MonitorPreview.setAvailableResolutions(self.xsetup.getCurrentScreen().getAvailableResolutions())
        self.MonitorPreviewStack.addWidget(self.MonitorPreview)
        
        # FIXME: Only load this widget if we're actually dualheaded.
        size = w,h = 400,300
        self.dualmonitorpreview = DualMonitorPreview(self,size,self.imagedir)
        #self.dualmonitorpreview.setCurrentScreen(self.xsetup.getCurrentScreen())
        # FIXME: pass screens so the PreviewWidget can sort out number and resolutions itself.
        self.MonitorPreviewStack.addWidget(self.dualmonitorpreview)
        self.connect(self.dualmonitorpreview,PYSIGNAL("currentMonitorChanged()"),self.slotMonitorFocussed)
        
        LayoutPreviewSize.addWidget(self.MonitorPreviewStack)
        LayoutPreviewSize.setResizeMode(QLayout.Auto)
        
        # -- Size & Refresh Box.
        LayoutSize = QHBoxLayout(None,0,6,"LayoutSize")
        LayoutSize.setResizeMode(QLayout.Auto)

        self.resizeslider = ResizeSlider(self)
        self.resizeslider.setSizePolicy(QSizePolicy(QSizePolicy.Expanding,QSizePolicy.Fixed))
        self.resizeslider.setScreen(self.xsetup.getAllScreens()[0])
        self.connect(self.resizeslider,PYSIGNAL("resolutionChange(int)"),self.slotResolutionChange)
        
        LayoutSize.addWidget(self.resizeslider)
        LayoutPreviewSize.setStretchFactor(self.resizeslider,0)
        
        RefreshCombo = QHBoxLayout(None,0,6,"RefreshCombo")
        self.RefreshLabel = QLabel(self,"RefreshLabel")
        self.RefreshLabel.setText("Refresh:")
        RefreshCombo.addWidget(self.RefreshLabel)
        
        self.sizerefreshcombo = QComboBox(0,self,"comboBox1") # gets filled in setRefreshRates()
        
        RefreshCombo.addWidget(self.sizerefreshcombo)
        LayoutSize.addLayout(RefreshCombo)
        
        LayoutPreviewSize.addLayout(LayoutSize)
        
        LeftVerSpacerRight = QSpacerItem(21,130,QSizePolicy.Minimum,QSizePolicy.Expanding)
        LayoutPreviewSize.addItem(LeftVerSpacerRight)
        #LayoutPreviewSize.addLayout(LayoutOrientDualhead)
        
        LayoutPage.addLayout(LayoutPreviewSize)

        SizeOrientationTabLayout.addLayout(LayoutPage,0,0)
        self.resize(self.minimumSizeHint())
        #self.resize(QSize(732,436).expandedTo(self.minimumSizeHint()))
        self.clearWState(Qt.WState_Polished)
        
    def setNotification(self,text):
        self.notify.setText(text)
    
    def slotOrientationRadioClicked(self,i):
        self.current_screen.setRotation([Screen.RR_Rotate_0, Screen.RR_Rotate_90,
                                                Screen.RR_Rotate_270, Screen.RR_Rotate_180][i])
        self.emit(PYSIGNAL("orientationChanged(int)"),(i,))
        self._updatePreview()
        self._sendChangedSignal()
        
    def slotMirrorHorizontallyToggled(self,b):
        # Bit flippin'
        if b:
            self.current_screen.setReflection(self.current_screen.getReflection() | Screen.RR_Reflect_X)
        else:
            self.current_screen.setReflection(self.current_screen.getReflection() & ~Screen.RR_Reflect_X)
        self.MonitorPreview.mirrorHorizontally(b)
        self._updatePreview()
        self._sendChangedSignal()

    def slotMirrorVerticallyToggled(self,b):
        # Bit flippin'
        if b:
            self.current_screen.setReflection(self.current_screen.getReflection() | Screen.RR_Reflect_Y)
        else:
            self.current_screen.setReflection(self.current_screen.getReflection() & ~Screen.RR_Reflect_Y)
        self.MonitorPreview.mirrorVertically(b)
        self._updatePreview()
        self._sendChangedSignal()

    def slotResolutionChange(self,i):
        print "Some resolution changed.", i
        self.current_screen.setResolutionIndex(i)
        # FIXME? : Initial setting of resolution would fail since no widget is visible at that point.
        try:
            self.MonitorPreviewStack.visibleWidget().setMinimalResolution(
                                                    self.current_screen.getAvailableResolutions()[0])
            self.MonitorPreviewStack.visibleWidget().changeResolution(
                                                    self.current_screen.getAvailableResolutions()[i])
        except AttributeError:
            pass
        print "ResolutionChange in SizePage"
        self.emit(PYSIGNAL("resolutionChange(int)"),(i,)) # FIXME: Add screen?
        self._sendChangedSignal()
        
    def setScreen(self,screen):
        self.current_screen = screen
        self._syncGUI()
            
    def slotDualheadToggled(self,enabled):
        """ FIXME: enabling Dualhead should:
            - Update resolutions with "metamodes"
            * disable Rotation/mirroring stuff 
            - Add ServerLayout to xorg.conf
            - Add second Screen to xorg.conf
            - Alternatively (if both resolutions are the same enable MergedFB
            - Handle ResolutionSliders independently
        """
        if enabled:
            print "Dualhead Mode is enabled now."
            self.dualmonitorpreview.setCurrentScreen(self.current_screen)
            self.MonitorPreviewStack.raiseWidget(self.dualmonitorpreview)
        else:
            print "Dualhead Mode is disabled now."
            self.resizeslider.setTitle("Screen size: ")
            self.MonitorPreviewStack.raiseWidget(self.MonitorPreview)
        self.MonitorPreview.setAvailableResolutions(self.current_screen.getAvailableResolutions())    
        # Enable / disable all the wizard that are(n't) now useful.
        self.enableRANDR(not enabled)
        self.LinkScreenSizesCheck.setEnabled(enabled)
        self.IdentifyButton.setEnabled(enabled)
        self.emit(PYSIGNAL("dualheadEnabled(bool)"),(enabled,))
        
############################################################################
class DpmsPage(QWidget):

    # Mapping values in seconds to human-readable labels.
    intervals = (
            (60,'1  minute'), 
            (120,'2  minutes'),
            (180,'3  minutes'),
            (300,'5  minutes'),
            (600,'10 minutes'),
            (900,'15 minutes'),
            (1200,'20 minutes'),
            (1500,'25 minutes'),
            (1800,'30 minutes'),
            (2700,'45 minutes'),
            (3600,'1 hour'),
            (7200,'2 hours'),
            (10800,'3 hours'),
            (14400,'4 hours'),
            (18000,'5 hours'))

    def __init__(self,parent = None,name = None,modal = 0,fl = 0):
        global imagedir
        QWidget.__init__(self,parent)
        
        # Where to find xset.
        self.xset_bin = os.popen('which xset').read()[:-1]
        self.kcmshell_bin = os.popen('which kcmshell').read()[:-1]
            
        if not name:
            self.setName("DPMSTab")

        DPMSTabLayout = QVBoxLayout(self,11,6,"DPMSTabLayout")

        titlelayout = QHBoxLayout(None,0,6,"titlelayout")
        topspacer = QSpacerItem(221,20,QSizePolicy.Expanding,QSizePolicy.Minimum)
        titlelayout.addItem(topspacer)

        self.energystarpix = QLabel(self,"energystarpix")
        self.energystarpix.setSizePolicy(QSizePolicy(QSizePolicy.Fixed,QSizePolicy.Fixed,0,0,self.energystarpix.sizePolicy().hasHeightForWidth()))
        self.energystarpix.setMinimumSize(QSize(150,77))
        self.energystarpix.setPixmap(QPixmap(imagedir+"../energystar.png"))
        self.energystarpix.setScaledContents(1)
        titlelayout.addWidget(self.energystarpix)
        DPMSTabLayout.addLayout(titlelayout)

        self.screensavergroup = QGroupBox(self,"screensavergroup")
        self.screensavergroup.setColumnLayout(0,Qt.Vertical)
        self.screensavergroup.layout().setSpacing(6)
        self.screensavergroup.layout().setMargin(11)
        self.screensavergroup.setTitle("Screensaver")
        
        screensavergroupLayout = QHBoxLayout(self.screensavergroup.layout())
        screensavergroupLayout.setAlignment(Qt.AlignTop)
        spacer4 = QSpacerItem(101,20,QSizePolicy.Expanding,QSizePolicy.Minimum)
        screensavergroupLayout.addItem(spacer4)

        self.screensaverbutton = QPushButton(self.screensavergroup)
        self.screensaverbutton.setText("Configure screensaver")

        self.connect(self.screensaverbutton,SIGNAL("clicked()"),self.slotScreesaverbuttonClicked)
        
        screensavergroupLayout.addWidget(self.screensaverbutton)
        DPMSTabLayout.addWidget(self.screensavergroup)

        self.dpmsgroup = QGroupBox(self,"dpmsgroup")
        self.dpmsgroup.setCheckable(1)
        self.dpmsgroup.setColumnLayout(0,Qt.Vertical)
        self.dpmsgroup.layout().setSpacing(6)
        self.dpmsgroup.layout().setMargin(11)
        self.dpmsgroup.setTitle("Enable display power management")
        
        self.connect(self.dpmsgroup,SIGNAL("toggled(bool)"),self.slotDpmsToggled)
        
        dpmsgroupLayout = QHBoxLayout(self.dpmsgroup.layout())
        dpmsgroupLayout.setAlignment(Qt.AlignTop)
        spacer4_2 = QSpacerItem(244,20,QSizePolicy.Expanding,QSizePolicy.Minimum)
        dpmsgroupLayout.addItem(spacer4_2)

        self.dpmstext = QLabel(self.dpmsgroup,"dpmstext")
        self.dpmstext.setText("Switch off display after")
        dpmsgroupLayout.addWidget(self.dpmstext)

        self.dpmscombo = QComboBox(0,self.dpmsgroup,"dpmscombo")
        self.fillCombo(self.dpmscombo)
        self.connect(self.dpmscombo,SIGNAL("activated(int)"),self.slotDpmsActivated)
        
        dpmsgroupLayout.addWidget(self.dpmscombo)
        DPMSTabLayout.addWidget(self.dpmsgroup)
        bottomspacer = QSpacerItem(51,160,QSizePolicy.Minimum,QSizePolicy.Expanding)
        DPMSTabLayout.addItem(bottomspacer)

        self.resize(QSize(508,372).expandedTo(self.minimumSizeHint()))
        self.clearWState(Qt.WState_Polished)
        self.readDpms()
        
    def fillCombo(self,combo):
        """ Fill the combobox with the values from our list """
        for interval in self.intervals:
            combo.insertItem(interval[1])

    def slotDpmsActivated(self,index):
        """ Another dpms value has been chosen, update buttons at bottom. """
        self.emit(PYSIGNAL("changedSignal()"), ())
        
    def slotDpmsToggled(self,bool):
        """ Dpms checkbox has been toggled, update buttons at bottom. """
        self.emit(PYSIGNAL("changedSignal()"), ())
        
    def slotScreesaverbuttonClicked(self):
        os.system("%s screensaver &" % self.kcmshell_bin)
        
    def readDpms(self):
        """ Read output from xset -q and parse DPMS settings from it. """
        pipe = os.popen("%s -q" % self.xset_bin)
        for line in pipe.readlines():
            if line.strip().startswith("Standby"):
                off_time = int(line.strip().split()[5]) # TODO: More subtle exception handling. ;)
            if line.strip().startswith("DPMS is"):
                dpms_on = line.strip().split()[2]
                
        self.dpms_enabled = {"Enabled":True,"Disabled":False}[dpms_on]
        self.dpms_min = off_time
        
        self.dpmsgroup.setChecked(self.dpms_enabled)

        found = False
        for i in range(len(self.intervals)):
            if self.dpms_min == self.intervals[i][0]:
                self.dpmscombo.setCurrentItem(i)
                found = True
        if not found:
            # FIXME: We haven't got an entry in our list that fits current settings, what to do??
            # Set to default == 30 min for now.
            self.dpms_min = 1800
        
    def isChanged(self):
        """ Check if something has changed since startup or last apply(). """
        print "isChanged()",
        if self.intervals[self.dpmscombo.currentItem()][0] != self.dpms_min:
            print "True"
            return True
        if self.dpmsgroup.isChecked() != self.dpms_enabled:
            return True
        # [...] Check other options that might've changed.
        print "False"
        return False
        
        
    def applyDpms(self):
        """ Use xset to apply new dpms settings. """
        enabled = self.dpmsgroup.isChecked()
        minutes = self.intervals[self.dpmscombo.currentItem()][0]
        if enabled:
            # Switch dpms on and set timeout interval.
            cmd_on = "%s +dpms" % self.xset_bin
            cmd_set = "%s dpms %i %i %i" % (self.xset_bin, minutes,minutes,minutes)
            print cmd_set
            if os.system(cmd_set) != 0:
                print "DPMS command failed: ", cmd_set
        else:
            # Switch dpms off.
            cmd_on = "%s -dpms" % self.xset_bin
        if os.system(cmd_on) != 0:
            print "DPMS command failed: ", cmd_on
        self.readDpms()
        self.emit(PYSIGNAL("changedSignal()"), ())
        print cmd_on
    
    def apply(self):
        self.applyDpms()

############################################################################
def create_displayconfig(parent,name):
    """ Factory function for KControl """
    print "create_displayconfig"
    global kapp
    kapp = KApplication.kApplication()
    return DisplayApp(parent, name)

############################################################################
def MakeAboutData():
    aboutdata = KAboutData("guidance",programname,version, \
        "Display and Graphics Configuration Tool", KAboutData.License_GPL, \
        "Copyright (C) 2003-2005 Simon Edwards", \
        "Thanks go to  Phil Thompson, Jim Bublitz and David Boddie.")
    aboutdata.addAuthor("Simon Edwards","Developer","simon@simonzone.com", \
            "http://www.simonzone.com/software/")
    aboutdata.addCredit("Pete Andrews","Gamma calibration pictures/system",None, \
            "http://www.photoscientia.co.uk/Gamma.htm")
    return aboutdata

if standalone:
    aboutdata = MakeAboutData()
    KCmdLineArgs.init(sys.argv,aboutdata)

    kapp = KApplication()
    displayapp = DisplayApp()
    displayapp.exec_loop()
