###########################################################################
# ScanPCI.py -                                                            #
# ------------------------------                                          #
# copyright : (C) 2005 by Simon Edwards                                   #
# email     : simon@simonzone.com                                         #
#                                                                         #
###########################################################################
#                                                                         #
#   This program is free software; you can redistribute it and/or modify  #
#   it under the terms of the GNU General Public License as published by  #
#   the Free Software Foundation; either version 2 of the License, or     #
#   (at your option) any later version.                                   #
#                                                                         #
###########################################################################
"""Provides information about the devices attached to the PCI bus.
"""
import struct
import csv

###########################################################################
class PCIDevice(object):
    def __init__(self):
        self.vendor = None      # PCI vendor id
        self.device = None

        self.subvendor = None   # 0xffff if not probe_type'd or no subid
        self.subdevice = None   # 0xffff if not probe_type'd or no subid
        self.pciclass = None    # 'None' if not probe_type'd

        self.pci_bus = None     # pci bus id 8 bits wide
        self.pci_device = None  # pci device id 5 bits wide
        self.pci_function = None# pci function id 3 bits wide

        self.module = None
        self.text = None
        self.already_found = False

        self.class_ = None
        
    def isGfxCard(self):
        #print "Module: ",self.module
        return self.module is not None and self.module.startswith("Card:")
        
    def getModule(self):
        if self.isGfxCard():
            return self.module[5:]
        else:
            return self.module
        
    def __str__(self):
        s = "PCI:%i:%i:%i " % (self.pci_bus,self.pci_device,self.pci_function)
        s += "Vendor:%x Device:%x" % (self.vendor,self.device)
        if self.subvendor is not None:
            s += ", Subvendor:%x" % self.subvendor
        if self.subdevice is not None:
            s += ", Subdevice:%x" % self.subdevice
        if self.module is not None:
            s += ", Module:%s" % self.module
        if self.text is not None:
            s += ", Text:%s" % self.text
        return s

############################################################################
class PCIBus(object):
    PCI_CLASS_SERIAL_USB = 0x0c03
    PCI_CLASS_SERIAL_FIREWIRE = 0x0c00

    def __init__(self):
        self.devices = []

    def detect(self):
        # Shamelessly translated from ldetect's pci.c.
        fhandle = open("/proc/bus/pci/devices")
        for line in fhandle.readlines():
            #print "L:",line
            entry = PCIDevice()
            self.devices.append(entry)
            parts = line.split()
            try:
                #if len(parts[17]):
                #    print "Module might be: ", parts[17]
                entry.module = parts[17]
            except IndexError:
                pass
            devbusfn = int(parts[0],16)
            idbits = int(parts[1],16)
            entry.vendor = idbits >> 16
            entry.device = idbits & 0xffff
            entry.pci_bus = devbusfn >> 8
            entry.pci_device = (devbusfn & 0xff) >> 3
            entry.pci_function = (devbusfn & 0xff) & 0x07

            try:
                infohandle = open("/proc/bus/pci/%02x/%02x.%d" % (
                                entry.pci_bus, entry.pci_device, entry.pci_function),"r")
                # these files are 256 bytes but we only need first 48 bytes
                buf = infohandle.read(48)   
                (class_prog, entry.class_, entry.subvendor, entry.subdevice) = \
                        struct.unpack("@xxxxxxxxxBHxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxHH",buf)
                #print "STRUCT: ",struct.unpack("@xxxxxxxxxBHxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxHH",buf)
                if (entry.subvendor==0 and entry.subdevice==0) or \
                        (entry.subvendor==entry.vendor and entry.subdevice==entry.device):
                    entry.subvendor = 0xffff
                    entry.subdevice = 0xffff
                if entry.class_==PCIBus.PCI_CLASS_SERIAL_USB:
                    # taken from kudzu's pci.c
                    if class_prog == 0:
                        entry.module = "usb-uhci"
                    elif class_prog == 0x10:
                        entry.module = "usb-ohci"
                    elif class_prog == 0x20:
                        entry.module = "ehci-hcd"
                if entry.class_ == PCIBus.PCI_CLASS_SERIAL_FIREWIRE:
                    # taken from kudzu's pci.c
                    if class_prog == 0x10:
                       entry.module = "ohci1394"
                infohandle.close()
            except IOError:
                pass
        fhandle.close()

        # Scan the PCI database.
        fhandle = open("/usr/share/ldetect-lst/pcitable","r")
        
        # This class is just for skipping comment lines in the database file.
        # This whole class is just an iterator wrapper that we put around our file iterator.
        class commentskipperiterator(object):
            def __init__(self,fhandle):
                self.fhandle = iter(fhandle)
            def __iter__(self):
                return self
            def next(self):
                line = self.fhandle.next()
                while line[0]=="#":
                    line = self.fhandle.next()
                return line
                
        unknowndevices = self.devices[:]
        
        # Process each row of the DB.
        for row in csv.reader(commentskipperiterator(fhandle),delimiter='\t'):
            if len(row)==4:
                (vendor,device,module,text) = row
            elif len(row)==6:
                (vendor, device, subvendor, subdevice, module, text) = row
                subvendor = int(subvendor[2:],16)
                subdevice = int(subdevice[2:],16)
            else:
                continue
            vendor = int(vendor[2:],16) # parse hex numbers of the form 0x1abc
            device = int(device[2:],16)
            
            i = 0
            while i<len(unknowndevices):
                pcidevice = unknowndevices[i]
                if pcidevice.vendor==vendor and pcidevice.device==device \
                        and (len(row)==4 \
                        or (pcidevice.subvendor==subvendor and pcidevice.subdevice==subdevice)):
                    pcidevice.module = module
                    pcidevice.text = text
                    if len(row)==6: # Close match, also matched on subdevice/subvendor ids.
                        del unknowndevices[i]
                    else:
                        i += 1
                else:
                    i += 1
            
        fhandle.close()

    def __str__(self):
        return "\n".join([str(x) for x in self.devices])
        
############################################################################
if __name__=='__main__':
    bus = PCIBus()
    bus.detect()
    print bus
    
