#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
from imagestore.lib.service import (
    ThreadedService, ServiceTask, ServiceError, taskHandlerInThread)


class StorageServiceError(ServiceError):
    pass


class StorageServiceTask(ServiceTask):
    pass

class StoreImagesTask(StorageServiceTask):

    def __init__(self, images):
        self.images = images

class GetStoredImagesTask(StorageServiceTask):

    def __init__(self, uris):
        self.uris = uris

class GetStoredImageStatesTask(StorageServiceTask):

    def __init__(self, uris):
        self.uris = uris

class SetErrorMessageTask(StorageServiceTask):

    def __init__(self, imageUri, errorMessage):
        self.imageUri = imageUri
        self.errorMessage = errorMessage

class ClearErrorMessageTask(StorageServiceTask):

    def __init__(self, imageUri):
        self.imageUri = imageUri

class SetUpgradeImagesTask(StorageServiceTask):

    def __init__(self, images):
        self.images = images

class SetImageRegistrationTask(StorageServiceTask):

    def __init__(self, imageRegistration):
        self.imageRegistration = imageRegistration

class GetInstalledImageUrisTask(StorageServiceTask):
    pass


class StorageService(ThreadedService):

    def __init__(self, reactor, storageFactory):
        ThreadedService.__init__(self, reactor)
        self._storageFactory = storageFactory

    def initializeThread(self):
        self._storage = self._storageFactory()

    def destructThread(self):
        self._storage.close()

    @taskHandlerInThread(StoreImagesTask)
    def _storeImages(self, task):
        for image in task.images:
            self._storage.addImage(image)
        self._storage.commit()

    @taskHandlerInThread(GetStoredImagesTask)
    def _getStoredImages(self, task):
        # XXX Optimize this to use a single query inside the storage.
        return [self._storage.getImage(uri) for uri in task.uris]

    @taskHandlerInThread(GetStoredImageStatesTask)
    def _getStoredImageStates(self, task):
        # XXX Optimize this to use a single query inside the storage.
        return [self._storage.getImageState(uri) for uri in task.uris]

    @taskHandlerInThread(SetErrorMessageTask)
    def _setErrorMessage(self, task):
        self._storage.setErrorMessage(task.imageUri, task.errorMessage)
        self._storage.commit()

    @taskHandlerInThread(ClearErrorMessageTask)
    def _clearErrorMessage(self, task):
        self._storage.clearErrorMessage(task.imageUri)
        self._storage.commit()

    @taskHandlerInThread(SetUpgradeImagesTask)
    def _setUpgradeImages(self, task):
        self._storage.setUpgrades([image["uri"] for image in task.images])
        self._storage.commit()

    @taskHandlerInThread(SetImageRegistrationTask)
    def _setImageRegistration(self, task):
        self._storage.setImageRegistration(task.imageRegistration)
        self._storage.commit()

    @taskHandlerInThread(GetInstalledImageUrisTask)
    def _getInstalledImageUris(self, task):
        return self._storage.getInstalledImageUris()
