#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
try:
    from pysqlite2 import dbapi2 as sqlite
except ImportError:
    from sqlite3 import dbapi2 as sqlite

from imagestore.lib.sqlitepatcher import SQLitePatcher
from imagestore.model import Image, ImageRegistration, ImageState


PATCHER = SQLitePatcher()


def withCursor(method, commit=False):
    def internalMethod(self, *args, **kwargs):
        cursor = self._connection.cursor()
        try:
            kwargs["cursor"] = cursor
            return method(self, *args, **kwargs)
        finally:
            cursor.close()
    return internalMethod

_auto = object() # Just a documentation aid.


class SQLiteStorage(object):

    def __init__(self, filename):
        PATCHER.patch(filename)
        self._connection = sqlite.connect(filename)

    def commit(self):
        self._connection.commit()

    def rollback(self):
        self._connection.rollback()

    def close(self):
        self._connection.close()

    def commitAndClose(self):
        self.commit()
        self.close()

    @withCursor
    def addImage(self, image, cursor=_auto):
        assert isinstance(image, Image)
        cursor.execute("REPLACE INTO image (uri, json) VALUES (?, ?)",
                       (image["uri"], str(image)))

    @withCursor
    def getImage(self, uri, cursor=_auto):
        cursor.execute("SELECT json FROM image WHERE uri=?", (uri,))
        row = cursor.fetchone()
        if row:
            return Image(row[0])
        return None

    @withCursor
    def getImageState(self, imageUri, cursor=_auto):
        state = ImageState({"status": "uninstalled",
                            "image-uri": imageUri})
        cursor.execute("SELECT kind, eid "
                       "FROM image_registration, image_registration_part "
                       "WHERE image_registration.image_uri=? AND "
                       "(image_registration_part.id = image_registration.eki OR"
                       " image_registration_part.id = image_registration.eri OR"
                       " image_registration_part.id = image_registration.emi)",
                       (imageUri,))
        rows = cursor.fetchall()
        if rows:
            state["status"] = "installed"
            kinds = {"kernel": "eki", "ramdisk": "eri", "image": "emi"}
            for row in rows:
                state[kinds[row[0]]] = row[1]
        else:
            # If it's not installed, it may be an upgrade.
            cursor.execute("SELECT image_uri FROM upgrade_image "
                           "WHERE image_uri=?", (imageUri,))
            row = cursor.fetchone()
            if row:
                state["is-upgrade"] = True
        cursor.execute("SELECT error_message FROM image_error "
                       "WHERE image_uri=?", (imageUri,))
        row = cursor.fetchone()
        if row:
            state["error-message"] = row[0]
        return state

    @withCursor
    def setUpgrades(self, imageUris, cursor=_auto):
        cursor.execute("DELETE FROM upgrade_image")
        for imageUri in imageUris:
            cursor.execute("INSERT INTO upgrade_image (image_uri) "
                           "VALUES (?)", (imageUri,))

    @withCursor
    def getUpgrades(self, cursor=_auto):
        cursor.execute("SELECT image_uri FROM upgrade_image")
        return [row[0] for row in cursor.fetchall()]

    @withCursor
    def setErrorMessage(self, imageUri, errorMessage, cursor=_auto):
        cursor.execute("REPLACE INTO image_error VALUES (?, ?)",
                       (imageUri, errorMessage))

    @withCursor
    def clearErrorMessage(self, imageUri, cursor=_auto):
        cursor.execute("DELETE FROM image_error WHERE image_uri=?",
                       (imageUri,))

    @withCursor
    def getErrorMessage(self, imageUri, cursor=_auto):
        cursor.execute("SELECT error_message FROM image_error "
                       "WHERE image_uri=?", (imageUri,))
        row = cursor.fetchone()
        if row:
            return row[0]
        return None

    @withCursor
    def setImageRegistration(self, imageRegistration, cursor=_auto):
        self.addImage(imageRegistration.image)
        part_id = {}
        for name in ("eki", "eri", "emi"):
            imagePart = getattr(imageRegistration, name)
            cursor.execute(
                "INSERT INTO image_registration_part (eid, kind, size, sha256) "
                "VALUES (?, ?, ?, ?)",
                (imagePart.eid, imagePart.kind,
                 imagePart.size, imagePart.sha256))
            part_id[name] = cursor.lastrowid

        cursor.execute("REPLACE INTO image_registration "
                       "(image_uri, eki, eri, emi) VALUES (?, ?, ?, ?)",
                       (imageRegistration.image["uri"],
                        part_id["eki"], part_id["eri"], part_id["emi"]))

    @withCursor
    def getInstalledImageUris(self, cursor=_auto):
        cursor.execute("SELECT image_uri FROM image_registration")
        return [row[0] for row in cursor.fetchall()]


PATCHER.setFinalSchema([
# We use separate tables because it's easier and faster to handle
# independent insertions, replacements and removals like this.
"""
CREATE TABLE image (
    uri TEXT PRIMARY KEY,
    json TEXT NOT NULL)
""",
"""
CREATE TABLE upgrade_image (
    image_uri TEXT PRIMARY KEY)
""",
"""
CREATE TABLE image_error (
    image_uri TEXT PRIMARY KEY,
    error_message TEXT NOT NULL)
""",
"""
CREATE TABLE image_registration (
    image_uri TEXT PRIMARY KEY,
    eki INTEGER NOT NULL,
    eri INTEGER NOT NULL,
    emi INTEGER NOT NULL)
""",
"""
CREATE TABLE image_registration_part (
    id INTEGER PRIMARY KEY,
    eid TEXT NOT NULL,
    kind TEXT NOT NULL,
    size INTEGER NOT NULL,
    sha256 TEXT NOT NULL)
""",
])
