/* grepmap
 *
 * util.c - utility functions
 *
 * Copyright © 2005 Canonical Ltd.
 * Author: Scott James Remnant <scott@ubuntu.com>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include "grepmap.h"


/**
 * fgets_alloc:
 * @stream: stdio stream to read from.
 *
 * Reads from stream up to EOF or a newline, without any line-length
 * limitations.
 *
 * Returns: static string containing the entire line WITHOUT the
 * terminating newline, or NULL if end of file is reached and nothing
 * was read.
 **/
char *
fgets_alloc (FILE *stream)
{
	static char   *buf = NULL;
	static size_t  buf_sz = 0;
	size_t         buf_len = 0;

	for (;;) {
		char *ret, *pos;

		if (buf_sz <= (buf_len + 1)) {
			buf_sz += BUFSIZ;
			buf = realloc (buf, buf_sz);
			if (! buf)
				abort();
		}

		ret = fgets (buf + buf_len, buf_sz - buf_len, stream);
		if ((! ret) && (! buf_len)) {
			return NULL;
		} else if (! ret) {
			return buf;
		}

		buf_len += strlen (ret);
		pos = strchr (ret, '\n');
		if (pos) {
			*pos = '\0';
			break;
		}
	}

	return buf;
}

/**
 * parse_hex:
 * @str: string to parse.
 * @hex: variable to write value to.
 *
 * Parses the string as a hexadecimal value, optionally beginning with
 * a variety of silly prefixes.
 *
 * Returns: 0 on success, 1 on failure.
 **/
int
parse_hex (const char   *str,
	   unsigned int *hex)
{
	int end;

	if (! strncmp (str, "0x", 2)) {
		str += 2;
	} else if (! strncmp (str, "&#x", 3)) {
		str += 3;
	} else if (! strncmp (str, "&", 1)) {
		str += 1;
	}

	if (sscanf (str, "%x%n", hex, &end) < 1)
		return 1;

	str += end;
	if (*str && (! strchr (" \t\r\n", *str)))
		return 1;

	return 0;
}

/**
 * parse_array:
 * @str: string to parse.
 * @array: integer array to fill.
 * @array_len: number of elements of integer array.
 *
 * Scans @str and reads a colon separated array of hexadecimal numbers.
 *
 * These arrays are a little strange, the kernel actually prints them
 * backwards starting from the first non-zero part; so they may not actually
 * be complete.
 *
 * Returns: 0 on success, 1 on failure.
 **/
int
parse_array (const char   *str,
	     BitArray     *array,
	     int           array_len)
{
	int i, j;

	memset (array, 0, array_len * sizeof (BitArray));
	i = array_len;

	while (*str && i) {
		int end;

#if LONG_ARRAYS
		if (sscanf (str, "%lx%n", &array[--i], &end) < 1)
			break;
#else
		if (sscanf (str, "%x%n", &array[--i], &end) < 1)
			break;
#endif

		str += end;
		if (*str == ':') {
			str++;
		} else if (*str && (! strchr (" \t\r\n", *str))) {
			return 1;
		} else {
			break;
		}
	}

	for (j = 0; j < array_len; j++, i++)
		array[j] = (i < array_len ? array[i] : 0);

	return 0;
}

/**
 * parse_line:
 * @line: line to parse.
 * @format: expected values.
 *
 * Parses the line according to the @format string, after skipping any
 * initial module name.  The format string should be a sequence of 'x'
 * or 'a' characters; for each 'x' character an unsigned int pointer
 * should be passed and a hexadecimal number parsed from the line will
 * be placed in it;  for each 'a' character an unsigned int pointer
 * should be passed with enough space for as many elements as specified
 * by an int parameter also passed.
 *
 * Example:
 *   parse_line(line, "xxa", &hex1, &hex2, array, 4)
 *
 * Returns: 0 if all values are pased, 1 if not.
 **/
int
parse_line (const char *line,
	    const char *format,
	    ...)
{
	const char *c;
	va_list     args;

	va_start (args, format);

	for (c = format; *c; c++) {
		unsigned int *int_val;
		BitArray     *array_val;
		int           array_len;

		line += strcspn (line, " \t\r\n");
		line += strspn (line, " \t\r\n");

		switch (*c) {
		case 'x':
			int_val = va_arg (args, unsigned int *);
			if (parse_hex (line, int_val))
				return 1;

			break;
		case 'a':
			array_val = va_arg (args, BitArray *);
			array_len = va_arg (args, int);
			if (parse_array (line, array_val, array_len))
				return 1;

			break;
		}
	}

	va_end (args);

	return 0;
}

/**
 * match_array:
 * @mask: bits that must be present in the array.
 * @array: array to check.
 * @array_len: number of elements in both arrays.
 *
 * Check that @array has all of the bits present in @mask.
 *
 * Returns: 0 if the arrays match, non-zero if not.
 **/
int
match_array (const BitArray *mask,
	     const BitArray *array,
	     int             array_len)
{
	int i;

	for (i = 0; i < array_len; i++) {
		if (! FLAG_SET(array[i], mask[i]))
			return 1;
	}

	return 0;
}
