/* grepmap
 *
 * usbmap.c - parse modules.usbmap
 *
 * Copyright © 2005 Canonical Ltd.
 * Author: Scott James Remnant <scott@ubuntu.com>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "grepmap.h"


/**
 * usbmap_parse_args:
 * @argc: number of arguments,
 * @argv: arguments array,
 * @dev: device structure to fill.
 *
 * Parses the command-line arguments to fill a device structure.
 *
 * Returns: 0 on success, non-zero on failure.
 **/
int
usbmap_parse_args (int        argc,
		   char      *argv[],
		   UsbDevice *dev)
{
	int ret = 0;

	if (argc < 9) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("insufficient arguments"));
		suggest_help ();
		return 2;
	}

	ret |= parse_hex (argv[0], &dev->vendor);
	ret |= parse_hex (argv[1], &dev->product);
	ret |= parse_hex (argv[2], &dev->dev);
	ret |= parse_hex (argv[3], &dev->dev_class);
	ret |= parse_hex (argv[4], &dev->dev_subclass);
	ret |= parse_hex (argv[5], &dev->dev_protocol);
	ret |= parse_hex (argv[6], &dev->int_class);
	ret |= parse_hex (argv[7], &dev->int_subclass);
	ret |= parse_hex (argv[8], &dev->int_protocol);
	if (ret) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("arguments must be in hexadecimal format"));
		suggest_help ();
		return 2;
	}

	if (argc > 9)
		fprintf (stderr, "%s: %s\n", program_name,
			 _("some arguments ignored"));

	return 0;
}

/**
 * usbmap_udev_event:
 * @environ: environment array,
 * @dev: device structure to fill.
 *
 * Fills the device structure by reading a udev event from the environment.
 *
 * Returns: 0 on success, non-zero on failure.
 **/
int
usbmap_udev_event (char      **environ,
		   UsbDevice  *dev)
{
	char **envvar;
	int    ret = 0;

	for (envvar = environ; *envvar; envvar++) {
		char *key, *value;

		key = strdup (*envvar);
		value = strchr (key, '=');
		if (! value)
			goto next;
		*(value++) = 0;

		if (! strcmp (key, "PRODUCT")) {
			char *ptr1, *ptr2;

			ptr1 = strchr (value, '/');
			if (! ptr1) {
				ret = 1;
				goto next;
			}
			*(ptr1++) = 0;

			ptr2 = strchr (ptr1, '/');
			if (! ptr2) {
				ret = 1;
				goto next;
			}
			*(ptr2++) = 0;

			ret |= parse_hex (value, &dev->vendor);
			ret |= parse_hex (ptr1, &dev->product);
			ret |= parse_hex (ptr2, &dev->dev);

		} else if (! strcmp (key, "TYPE")) {
			char *ptr1, *ptr2;

			ptr1 = strchr (value, '/');
			if (! ptr1) {
				ret = 1;
				goto next;
			}
			*(ptr1++) = 0;

			ptr2 = strchr (ptr1, '/');
			if (! ptr2) {
				ret = 1;
				goto next;
			}
			*(ptr2++) = 0;

			ret |= parse_hex (value, &dev->dev_class);
			ret |= parse_hex (ptr1, &dev->dev_subclass);
			ret |= parse_hex (ptr2, &dev->dev_protocol);

		} else if (! strcmp (key, "INTERFACE")) {
			char *ptr1, *ptr2;

			ptr1 = strchr (value, '/');
			if (! ptr1) {
				ret = 1;
				goto next;
			}
			*(ptr1++) = 0;

			ptr2 = strchr (ptr1, '/');
			if (! ptr2) {
				ret = 1;
				goto next;
			}
			*(ptr2++) = 0;

			ret |= parse_hex (value, &dev->int_class);
			ret |= parse_hex (ptr1, &dev->int_subclass);
			ret |= parse_hex (ptr2, &dev->int_protocol);

		}

	next:
		free (key);
	}

	if (ret) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("udev event not in expected format"));
		return 2;
	}

	return 0;
}

/**
 * usbmap_modules:
 * @mapf: open map file to read from,
 * @file: filename or map file, or '-' for standard input,
 * @dev: device to look for.
 *
 * Handles the modules.usbmap file looking for a matching entry for
 * the device given, printing the module name to stdout if found.
 *
 * Returns: 0 on success, non-zero on failure.
 **/
int
usbmap_modules (FILE       *mapf,
		const char *file,
		UsbDevice  *dev)
{
	char *line;
	int   lineno = 0, ret = 0;

	ret = 1;
	while ((line = fgets_alloc (mapf)) != NULL) {
		UsbDevice map;

		++lineno;
		if ((line[0] == '#') || (line[0] == '\0'))
			continue;

		if (parse_line (line, "xxxxxxxxxxx",
				(unsigned int *)&map.match_flags,
				&map.vendor, &map.product,
				&map.dev_lo, &map.dev_hi,
				&map.dev_class, &map.dev_subclass,
				&map.dev_protocol,
				&map.int_class, &map.int_subclass,
				&map.int_protocol))
		{
			fprintf (stderr, "%s:%s:%d: %s\n", program_name, file,
				 lineno, _("unrecognised line format"));
			continue;
		}

		if (FLAG_SET(map.match_flags, USB_MATCH_VENDOR)
		    && (map.vendor != dev->vendor))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_PRODUCT)
		    && (map.product != dev->product))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_DEV_LO)
		    && (map.dev_lo > dev->dev))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_DEV_HI)
		    && (map.dev_hi < dev->dev))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_DEV_CLASS)
		    && (map.dev_class != dev->dev_class))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_DEV_SUBCLASS)
		    && (map.dev_subclass != dev->dev_subclass))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_DEV_PROTOCOL)
		    && (map.dev_protocol != dev->dev_protocol))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_INT_CLASS)
		    && (map.int_class != dev->int_class))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_INT_SUBCLASS)
		    && (map.int_subclass != dev->int_subclass))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_INT_PROTOCOL)
		    && (map.int_protocol != dev->int_protocol))
			continue;

		line[strcspn (line, " \t\r\n")] = '\0';
		printf ("%s\n", line);
		ret = 0;
	}

	return ret;
}
