/*
 * Copyright (C) 2001, 2002 Free Software Foundation
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Authors : Carlos Garca Campos <carlosgc@gnome.org>
 */


#include <libgnomevfs/gnome-vfs.h>
#include <gnome.h>
#include "cpufreq-applet.h"
#include "cpufreq.h"

static gchar *cpufreq_get_human_readble_freq (gint freq);
static gchar *cpufreq_get_human_readble_unit (gint freq);
static gchar *cpufreq_get_human_readble_perc (gint fmax, gint fmin);


static gchar *
cpufreq_get_human_readble_freq (gint freq)
{
	   gint divisor;

	   if (freq > 999999) /* freq (kHz) */
			 divisor = (1000 * 1000);
	   else
			 divisor = 1000;

	   if (((freq % divisor) == 0) || divisor == 1000) /* integer */
			 return g_strdup_printf ("%d", freq / divisor);
	   else /* float */
			 return g_strdup_printf ("%3.2f", ((gfloat)freq / divisor));
}

static gchar *
cpufreq_get_human_readble_unit (gint freq)
{
	   if (freq > 999999) /* freq (kHz) */
			 return g_strdup ("GHz");
	   else
			 return g_strdup ("MHz");
}

static gchar *
cpufreq_get_human_readble_perc (gint fmax, gint fmin)
{
	   return g_strdup_printf ("%d%%", (fmin * 100) / fmax);
}

gboolean
cpufreq_get_from_procfs (gpointer gdata)
{
	   GnomeVFSHandle *handle;
	   GnomeVFSFileSize bytes_read;
	   GnomeVFSResult result;
	   gchar *uri, *file;
	   gchar **lines;
	   gchar buffer[256];
	   gint i;
	   gint fmax, fmin, cpu;
	   gint pmin, pmax;
	   gchar mode[21];
	   CPUFreqApplet *applet;
	   CPUFreqMode cpufreq_mode;

	   applet = (CPUFreqApplet *) gdata;

	   g_return_val_if_fail (PANEL_IS_APPLET (PANEL_APPLET (applet)), FALSE);

	   uri = gnome_vfs_get_uri_from_local_path ("/proc/cpufreq");

	   result = gnome_vfs_open (&handle, uri, GNOME_VFS_OPEN_READ);
	   if (result != GNOME_VFS_OK) {
			 if (uri) g_free (uri);
			 
			 return FALSE;
	   }

	   g_free (uri);

	   result = gnome_vfs_read (handle, buffer, 256, &bytes_read);
	   file = g_strndup (buffer, bytes_read);
	   if (result != GNOME_VFS_OK) {
			 g_free (file);
			 
			 return FALSE;
	   }

	   result = gnome_vfs_close (handle);
	   if (result != GNOME_VFS_OK) {
			 g_free (file);
			 
			 return FALSE;
	   }

	   lines = g_strsplit (file, "\n", -1);
	   for (i=0; lines[i]; i++) {
			 if (lines[i][0] == 'C') {
				    /* CPU  0       650000 kHz ( 81 %)  -     800000 kHz (100 %)  -  powersave */
				    sscanf (lines[i], "CPU %d %d kHz (%d %%) - %d kHz (%d %%) - %20s",
						  &cpu, &fmin, &pmin, &fmax, &pmax, mode);
				    
				    if (cpu == applet->cpu)
						  break;
			 }
	   }

	   g_strfreev (lines);
	   g_free (file);

	   if (applet->freq) {
			 g_free (applet->freq);
			 applet->freq = NULL;
	   }
	   
	   if (applet->perc) {
			 g_free (applet->perc);
			 applet->perc = NULL;
	   }
	   
	   if (applet->unit) {
			 g_free (applet->unit);
			 applet->unit = NULL;
	   }
	   
	   if (g_strcasecmp (mode, "powersave") == 0) {
			 cpufreq_mode = POWERSAVE;
			 applet->freq = cpufreq_get_human_readble_freq (fmin);
			 applet->perc = g_strdup_printf ("%d%%", pmin);
			 applet->unit = cpufreq_get_human_readble_unit (fmin);
	   } else {
			 cpufreq_mode = PERFORMANCE;
			 applet->freq = cpufreq_get_human_readble_freq (fmax);
			 applet->perc = g_strdup_printf ("%d%%", pmax);
			 applet->unit = cpufreq_get_human_readble_unit (fmax);
	   }

	   if (applet->freq == NULL)
			 return FALSE;
	   if (applet->perc == NULL)
			 return FALSE;
	   if (applet->unit == NULL)
			 return FALSE;

	   cpufreq_applet_update (applet, cpufreq_mode);

	   return TRUE;
}

gboolean
cpufreq_get_from_sysfs (gpointer gdata)
{
	   GnomeVFSHandle *handle;
	   GnomeVFSFileSize bytes_read;
	   GnomeVFSResult result;
	   gchar *uri;
	   gchar buffer[20];
	   gint i;
	   CPUFreqApplet *applet;
	   CPUFreqMode cpufreq_mode;
	   gchar *files[] = {
			 "scaling_max_freq",
			 "scaling_min_freq",
			 "scaling_governor",
			 "cpuinfo_max_freq",
			 "scaling_setspeed",
			 NULL };
	   gchar **cpufreq_data;
	   gchar *path;
	   
	   enum {
			 SCALING_MAX,
			 SCALING_MIN,
			 GOVERNOR,
			 CPUINFO_MAX,
			 SCALING_SETSPEED,
			 LAST
	   };

	   applet = (CPUFreqApplet *) gdata;

	   g_return_val_if_fail (PANEL_IS_APPLET (PANEL_APPLET (applet)), FALSE);

	   /* /sys/devices/system/cpu/cpu[0]/cpufreq/scaling_max_freq
	    * /sys/devices/system/cpu/cpu[0]/cpufreq/scaling_min_freq
	    * /sys/devices/system/cpu/cpu[0]/cpufreq/scaling_governor
	    * /sys/devices/system/cpu/cpu[0]/cpufreq/cpuinfo_max_freq
	    * /sys/devices/system/cpu/cpu[0]/cpufreq/scaling_setspeed (userspace)
	    */

	   cpufreq_data = g_new (gchar *, LAST + 1);
	   
	   for (i=0; i<LAST; i++) {
			 cpufreq_data[i] = NULL;
			 /* SCALING_SETSPEED only exists in userspace governors */
			 if ((i == SCALING_SETSPEED) &&
				(g_strcasecmp (cpufreq_data[GOVERNOR], "userspace") != 0)) {
				    cpufreq_data[i] = g_strdup ("");
				    buffer[0] = '\0';
				    continue;
			 }

			 path = g_strdup_printf ("/sys/devices/system/cpu/cpu%d/cpufreq/%s",
								applet->cpu, files[i]);
			 uri = gnome_vfs_get_uri_from_local_path (path);
			 g_free (path);
			 
			 result = gnome_vfs_open (&handle, uri, GNOME_VFS_OPEN_READ);
			 if (result != GNOME_VFS_OK) {
				    if (uri) g_free (uri);
				    g_strfreev (cpufreq_data);
				    
				    return FALSE;
			 }

			 g_free (uri);

			 result = gnome_vfs_read (handle, buffer, sizeof (buffer), &bytes_read);
			 
			 /* bytes_read - 1 in order to remove the \n character */
			 cpufreq_data[i] = g_strndup (buffer, bytes_read - 1);

			 if (result != GNOME_VFS_OK) {
				    g_strfreev (cpufreq_data);
				    gnome_vfs_close (handle);
				    
				    return FALSE;
			 }

			 result = gnome_vfs_close (handle);
			 if (result != GNOME_VFS_OK) {
				    g_strfreev (cpufreq_data);
				    
				    return FALSE;
			 }
			 buffer[0] = '\0';
	   }
	   cpufreq_data[LAST] = NULL;

	   if (applet->freq) {
			 g_free (applet->freq);
			 applet->freq = NULL;
	   }
	   
	   if (applet->perc) {
			 g_free (applet->perc);
			 applet->perc = NULL;
	   }
	   
	   if (applet->unit) {
			 g_free (applet->unit);
			 applet->unit = NULL;
	   }

	   if (g_strcasecmp (cpufreq_data[GOVERNOR], "userspace") == 0) {
			 cpufreq_mode = USERSPACE;
			 applet->freq = cpufreq_get_human_readble_freq (atoi (cpufreq_data[SCALING_SETSPEED]));
			 applet->perc = cpufreq_get_human_readble_perc (atoi (cpufreq_data[CPUINFO_MAX]),
												   atoi (cpufreq_data[SCALING_SETSPEED]));
			 applet->unit = cpufreq_get_human_readble_unit (atoi (cpufreq_data[SCALING_SETSPEED]));
	   } else if (g_strcasecmp (cpufreq_data[GOVERNOR], "powersave") == 0) {
			 cpufreq_mode = POWERSAVE;
			 applet->freq = cpufreq_get_human_readble_freq (atoi (cpufreq_data[SCALING_MIN]));
			 applet->perc = cpufreq_get_human_readble_perc (atoi (cpufreq_data[CPUINFO_MAX]),
												   atoi (cpufreq_data[SCALING_MIN]));
			 applet->unit = cpufreq_get_human_readble_unit (atoi (cpufreq_data[SCALING_MIN]));
	   } else {
			 cpufreq_mode = PERFORMANCE;
			 applet->freq = cpufreq_get_human_readble_freq (atoi (cpufreq_data[SCALING_MAX]));
			 applet->perc = cpufreq_get_human_readble_perc (atoi (cpufreq_data[CPUINFO_MAX]),
												   atoi (cpufreq_data[SCALING_MAX]));
			 applet->unit = cpufreq_get_human_readble_unit (atoi (cpufreq_data[SCALING_MAX]));
	   }

	   if (applet->freq == NULL) {
			 g_strfreev (cpufreq_data);
			 
			 return FALSE;
	   }

	   if (applet->perc == NULL) {
			 g_strfreev (cpufreq_data);
			 
			 return FALSE;
	   }

	   if (applet->unit == NULL) {
			 g_strfreev (cpufreq_data);

			 return FALSE;
	   }

	   g_strfreev (cpufreq_data);

	   cpufreq_applet_update (applet, cpufreq_mode);

	   return TRUE;
}

gboolean
cpufreq_get_from_procfs_cpuinfo (CPUFreqApplet *applet)
{
	   GnomeVFSHandle *handle;
	   GnomeVFSFileSize bytes_read;
	   GnomeVFSResult result;
	   gchar *uri, *file;
	   gchar **lines;
	   gchar buffer[256];
	   gchar *p;
	   gint cpu, i;
	   CPUFreqMode cpufreq_mode;

	   g_return_val_if_fail (PANEL_IS_APPLET (PANEL_APPLET (applet)), FALSE);

	   uri = gnome_vfs_get_uri_from_local_path ("/proc/cpuinfo");

	   result = gnome_vfs_open (&handle, uri, GNOME_VFS_OPEN_READ);
	   if (result != GNOME_VFS_OK) {
			 if (uri) g_free (uri);
			 
			 return FALSE;
	   }

	   g_free (uri);

	   result = gnome_vfs_read (handle, buffer, 256, &bytes_read);
	   file = g_strndup (buffer, bytes_read);
	   if (result != GNOME_VFS_OK) {
			 g_free (file);
			 gnome_vfs_close (handle);
			 
			 return FALSE;
	   }

	   result = gnome_vfs_close (handle);
	   if (result != GNOME_VFS_OK) {
			 g_free (file);
			 
			 return FALSE;
	   }

	   lines = g_strsplit (file, "\n", -1);
	   for (i=0; lines[i]; i++) {
			 if (g_ascii_strncasecmp ("cpu MHz", lines[i], strlen ("cpu MHz")) == 0) {
				    p = g_strrstr (lines[i], ":");

				    if (p == NULL) {
						  g_strfreev (lines);
						  g_free (file);
						  
						  return FALSE;
				    }

				    if (strlen (lines[i]) < (p - lines[i])) {
						  g_strfreev (lines);
						  g_free (file);
						  
						  return FALSE;
				    }

				    if ((sscanf(p + 1, "%d.", &cpu)) != 1) {
						  g_strfreev (lines);
						  g_free (file);
						  
						  return FALSE;
				    }

				    break;
			 }
	   }

	   g_strfreev (lines);
	   g_free (file);

	   if (applet->freq) {
			 g_free (applet->freq);
			 applet->freq = NULL;
	   }

	   if (applet->perc) {
			 g_free (applet->perc);
			 applet->perc = NULL;
	   }

	   if (applet->unit) {
			 g_free (applet->unit);
			 applet->unit = NULL;
	   }

	   cpufreq_mode = PERFORMANCE;
	   applet->freq = cpufreq_get_human_readble_freq (cpu * 1000); /* kHz are expected*/
	   applet->unit = cpufreq_get_human_readble_unit (cpu * 1000); /* kHz are expected*/
	   applet->perc = g_strdup ("100%");

	   if (applet->freq == NULL)
			 return FALSE;
	   if (applet->perc == NULL)
			 return FALSE;
	   if (applet->unit == NULL)
			 return FALSE;

	   cpufreq_applet_update (applet, cpufreq_mode);

	   return TRUE;
}
