/* gcompris - python.c
 *
 * Copyright (C) 2003 GCompris Developpement Team
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <Python.h>
#include <pygobject.h>
#include "gcompris/gcompris.h"
#include "py-gcompris-board.h"
#include "py-mod-gcompris.h"
#include "py-gcompris-profile.h"

static GcomprisBoard *gcomprisBoard = NULL;

static PyObject* python_gcomprisBoard = NULL;
static PyObject* python_board_module = NULL;
static PyObject* python_board_instance = NULL;

static GcomprisBoard *gcomprisBoard_config = NULL;
static PyObject* python_gcomprisBoard_config = NULL;;
static PyObject* python_board_config_module = NULL;
static PyObject* python_board_config_instance = NULL;

static void	 pythonboard_init (GcomprisBoard *agcomprisBoard);
static void	 pythonboard_start (GcomprisBoard *agcomprisBoard);
static void	 pythonboard_pause (gboolean pause);
static void	 pythonboard_end (void);

static gboolean	 pythonboard_is_our_board (GcomprisBoard *agcomprisBoard);

static gint	 pythonboard_key_press (guint keyval, gchar *commit_str, gchar *preedit_str);
static void	 pythonboard_ok (void);
static void	 pythonboard_set_level (guint level);
static void	 pythonboard_config(void);
static void	 pythonboard_repeat (void);
static void	 pythonboard_config_start (GcomprisBoard *agcomprisBoard,
					   GcomprisProfile *aProfile);
static void	 pythonboard_config_stop (void);

static gboolean  pythonboard_is_ready = FALSE;


/* Description of this plugin */
static BoardPlugin menu_bp =
  {
    NULL,
    NULL,
    N_("Python Board"),
    N_("Special board that embeds python into gcompris."),
    "Olivier Samyn <osamyn@ulb.ac.be>",
    pythonboard_init,
    NULL,
    NULL,
    NULL,
    pythonboard_start,
    pythonboard_pause,
    pythonboard_end,
    pythonboard_is_our_board,
    pythonboard_key_press,
    pythonboard_ok,
    pythonboard_set_level,
    pythonboard_config,
    pythonboard_repeat,
    pythonboard_config_start,
    pythonboard_config_stop
  };

static BoardPlugin *bp_board = NULL;

/*
 * Return the plugin structure (Common to all gcompris boards)
 */

GET_BPLUGIN_INFO(python)


/*
 * Tests if a python interpreter is available
 * and that all required imports can be loaded.
 */

static GList *config_boards= NULL;

GList *
get_pythonboards_list()
{
  GList *pythonboards_list = NULL;
  GList *boards_list = gcompris_get_boards_list();
  GList *list;
  GcomprisBoard *board;

  for (list = boards_list; list != NULL; list = list->next){
    board = (GcomprisBoard *) list->data;
    if (g_ascii_strncasecmp(board->type, "python", 6)==0)
      pythonboards_list = g_list_append(pythonboards_list, board);
  }

  return pythonboards_list;
}

static void
pythonboard_init (GcomprisBoard *agcomprisBoard){
  PyObject* main_module;
  PyObject* globals;
  gchar* execstr;
  gchar* userplugindir;


  gboolean has_config = TRUE;
  char* board_file_name;
  char* boardclass;
  gchar *boarddir;
  PyObject* module_dict;
  PyObject* py_boardclass;
  PyObject* py_boardclass_args;
  PyObject* py_config_start;

  GcomprisProperties	*properties = gcompris_get_properties();

  if (pythonboard_is_ready)
    return ;

  /* Initialize the python interpreter */
  Py_Initialize();

  static char *python_args[]={ "" };
  PySys_SetArgv( 1, python_args);

  pythonboard_is_ready = TRUE;

  main_module = PyImport_AddModule("__main__"); /* Borrowed reference */
  globals = PyModule_GetDict(main_module); /* Borrowed reference */

  if(globals==NULL){
    g_warning("! Python disabled: Cannot get info from the python interpreter.\n");
    pythonboard_is_ready = FALSE;
  } else {
    /* Add the python plugins dir to the python's search path */
    execstr = g_strdup_printf("import sys; sys.path.append('%s')",PYTHON_PLUGIN_DIR);
#ifndef DISABLE_USER_PLUGIN_DIR
    userplugindir = g_strconcat(g_get_home_dir(), "/.gcompris/Plugins/", NULL);
    execstr = g_strdup_printf("import sys; sys.path.append('%s/python'); sys.path.append('%s')",
			      userplugindir, PYTHON_PLUGIN_DIR);
    g_free(userplugindir);
#else
    execstr = g_strdup_printf("import sys; sys.path.append('%s')",PYTHON_PLUGIN_DIR );
#endif
    if(PyRun_SimpleString(execstr)!=0){
      pythonboard_is_ready = FALSE;
      g_warning("! Python disabled: Cannot add plugins dir into search path\n");
    } else {
      /* Load the gcompris modules */
      python_gcompris_module_init();

      /* Try to import pygtk modules */
      g_free(execstr);
      execstr = g_strdup("import gtk; import gtk.gdk");
      if(PyRun_SimpleString(execstr)!=0){
	pythonboard_is_ready = FALSE;
	g_warning("! Python disabled: Cannot import pygtk modules\n");
      } else {
	/* Try to import gnome-python modules */
	g_free(execstr);
	execstr = g_strdup("import gnome; import gnome.canvas");
	if(PyRun_SimpleString(execstr)!=0){
	  pythonboard_is_ready = FALSE;
	  g_warning("! Python disabled: Cannot import gnome-python modules\n");
	} else {
	  /* Try to import gcompris modules */
	  g_free(execstr);
	  execstr = g_strdup("import gcompris; import gcompris.bonus; "
			     "import gcompris.score; import gcompris.sound;"
			     "import gcompris.skin; import gcompris.timer;"
			     "import gcompris.utils; import gcompris.anim");
	  if(PyRun_SimpleString(execstr)!=0){
	    pythonboard_is_ready = FALSE;
	    g_warning("! Python disabled: Cannot import gcompris modules\n");
	  } else {
	    GList *python_boards;
	    GList *list;

	    /* Load the gcompris modules */
	    python_gcompris_module_init();

	    /* Get the list of python boards */
	    python_boards = get_pythonboards_list();

	    /* Search in the list each one with a config entry */
	    for(list = python_boards; list != NULL; list = list->next) {
	      GcomprisBoard *board = (GcomprisBoard *) list->data;

	      /* Python is now initialized we create some usefull variables */
	      board_file_name = strchr(board->type, ':')+1;
	      boardclass = g_strdup_printf("Gcompris_%s", board_file_name);

	      /* Test if board come with -L option */

	      g_warning("board_dir: %s package_data_dir %s",
			board->board_dir,
			properties->package_data_dir);

	      if (strcmp(board->board_dir, properties->package_data_dir)!=0){ 
		boarddir = g_strdup_printf("sys.path.append('%s/../python/')", board->board_dir);
	      
		PyRun_SimpleString(boarddir);
		g_free(boarddir);
	      }
	      
	      /* Insert the board module into the python's interpreter */
	      python_board_module = PyImport_ImportModuleEx(board_file_name,
							    globals,
							    globals,
							    NULL);

	      if(python_board_module!=NULL){
		/* Get the module dictionnary */
		module_dict = PyModule_GetDict(python_board_module);
	      
		/* Get the python board class */
		py_boardclass = PyDict_GetItemString(module_dict, boardclass);

		if (PyObject_HasAttrString( py_boardclass, "config_start")) {
		  config_boards = g_list_append(config_boards, board);
		  g_warning("The board '%s' has a configuration entry", 
			    board_file_name);
		}
	      }

	      g_free(boardclass);
	    }

	    g_list_free(python_boards);
	  }
	}
      }
    }
    g_free(execstr);

  }

  /* Finalize the python interpreter */
  Py_Finalize();

}

/*
 * Start the board.
 * In this case:
 * - initialize python interpreter
 * - import gcompris functions/objects
 * - import gtk/gnome functions/objects
 * - load the python written board
 * - call the board start function
 */
static void
pythonboard_start (GcomprisBoard *agcomprisBoard){
  PyObject* main_module;
  PyObject* py_function_result;
  PyObject* module_dict;
  PyObject* py_boardclass;
  PyObject* py_boardclass_args;
  PyObject* globals;
  static char *python_args[]={ "" };
  static char* python_prog_name="gcompris";
  char* boarddir;
  char* boardclass;
  char* board_file_name;
  gchar *userplugindir;

  GcomprisProperties	*properties = gcompris_get_properties();

  if(agcomprisBoard!=NULL){
    /* Initialize the python interpreter */
    Py_SetProgramName(python_prog_name);
    Py_Initialize();

    PySys_SetArgv(1, python_args);

    init_pygobject();

    main_module = PyImport_AddModule("__main__");
    globals = PyModule_GetDict(main_module);

    if(globals==NULL){
      g_print("Cannot get info from the python interpreter. Seems there is a problem with this one.\n");
      return;
    } else {
      gcomprisBoard = agcomprisBoard;
    }

    /* Add the python plugins dir to the python's search path */
#ifndef DISABLE_USER_PLUGIN_DIR
    userplugindir = g_strconcat(g_get_home_dir(), "/.gcompris/Plugins/", NULL);
    boarddir = g_strdup_printf("import sys; sys.path.append('%s/python'); sys.path.append('%s')",
			       userplugindir,
			       PYTHON_PLUGIN_DIR);
#else
    boarddir = g_strdup_printf("import sys; sys.path.append('%s')",PYTHON_PLUGIN_DIR );
#endif

    PyRun_SimpleString(boarddir);
    g_free(boarddir);


    /* Test if board come with -L option */
    if (strcmp(gcomprisBoard->board_dir, properties->package_data_dir)!=0){ 
      boarddir = g_strdup_printf("sys.path.append('%s/../python/')", gcomprisBoard->board_dir);
      
      PyRun_SimpleString(boarddir);
      g_free(boarddir);
    }


#ifndef DISABLE_USER_PLUGIN_DIR
    g_free(userplugindir);
#endif

    /* Load the gcompris modules */
    python_gcompris_module_init();

    /* Python is now initialized we create some usefull variables */
    board_file_name = strchr(agcomprisBoard->type, ':')+1;
    boardclass = g_strdup_printf("Gcompris_%s", board_file_name);

    /* Insert the board module into the python's interpreter */
    python_board_module = PyImport_ImportModuleEx(board_file_name,
 						  globals,
 						  globals,
 						  NULL);

    if(python_board_module!=NULL){
      /* Get the module dictionnary */
      module_dict = PyModule_GetDict(python_board_module);

      /* Get the python board class */
      py_boardclass = PyDict_GetItemString(module_dict, boardclass);

      /* Create a python gcompris board */
      python_gcomprisBoard=gcompris_new_pyGcomprisBoardObject(agcomprisBoard);

      /* Create an instance of the board class */
      py_boardclass_args = PyTuple_New(1);
      Py_INCREF(python_gcomprisBoard);
      PyTuple_SetItem(py_boardclass_args, 0, python_gcomprisBoard);
      python_board_instance = PyInstance_New(py_boardclass, py_boardclass_args, NULL);
      Py_DECREF(py_boardclass_args);

      /* Call the function */
      py_function_result = PyObject_CallMethod(python_board_instance, "start", NULL);
      if( py_function_result != NULL){
	Py_DECREF(py_function_result);
      } else {
	PyErr_Print();
      }
    } else {
      PyErr_Print();
    }

    g_free(boardclass);
  }
}

/*
 * Pause the board.
 */
static void pythonboard_pause (gboolean pause){
  PyObject* result = NULL;

  result = PyObject_CallMethod(python_board_instance, "pause", "i", pause);
  if( result != NULL){
    Py_DECREF(result);
  } else {
    PyErr_Print();
  }
}

/*
 * End the board.
 * In this case:
 * - call the board end function
 * - finalise python interpreter
 */
static void pythonboard_end (void){
  PyObject* result = NULL;

  if(python_gcomprisBoard!=NULL){
    result = PyObject_CallMethod(python_board_instance, "end", NULL);
    if( result == NULL){
      PyErr_Print();
    } else {
      Py_DECREF(result);
    }
    Py_XDECREF(python_board_module);
    Py_XDECREF(python_board_instance);
    Py_XDECREF(python_gcomprisBoard);
    Py_Finalize();
  }
}

/*
 * Return TRUE if the board is a python one.
 */
static gboolean pythonboard_is_our_board (GcomprisBoard *agcomprisBoard){

  if (agcomprisBoard->plugin)
    return TRUE;

  if(pythonboard_is_ready) {
    if (agcomprisBoard!=NULL) {

      if (g_ascii_strncasecmp(agcomprisBoard->type, "python", 6)==0) {
  
	bp_board = g_malloc0(sizeof(BoardPlugin));
	
	bp_board->handle        = menu_bp.handle;
	bp_board->filename      = menu_bp.filename;
	bp_board->name          = menu_bp.name;
	bp_board->description   = menu_bp.description;
	bp_board->author        = menu_bp.author;
	bp_board->init          = menu_bp.init;
	bp_board->cleanup       = menu_bp.cleanup;
	bp_board->about         = menu_bp.about;
	bp_board->configure     = menu_bp.configure;
	bp_board->start_board   = menu_bp.start_board;
	bp_board->pause_board   = menu_bp.pause_board;
	bp_board->end_board     = menu_bp.end_board;
	bp_board->is_our_board  = menu_bp.is_our_board;
	bp_board->key_press     = menu_bp.key_press;
	bp_board->ok            = menu_bp.ok;
	bp_board->set_level     = menu_bp.set_level;
	bp_board->config        = menu_bp.config;
	bp_board->repeat        = menu_bp.repeat;
  
	if (g_list_find (config_boards, agcomprisBoard)){
	  bp_board->config_start  = menu_bp.config_start;
	  bp_board->config_stop   = menu_bp.config_stop;
	} else {
	  bp_board->config_start  = NULL;
	  bp_board->config_stop   = NULL;
	}


	/* Set the plugin entry */
	agcomprisBoard->plugin = bp_board;
	
	bp_board = NULL;
	
	//g_print("pythonboard: is our board = TRUE\n");
	
	return TRUE;
      }
    }
  }
  agcomprisBoard->plugin=NULL;
  return FALSE;
}

/*
 * Key press
 */
static gint pythonboard_key_press (guint keyval, gchar *commit_str, gchar *preedit_str){
  PyObject* result = NULL;

  result = PyObject_CallMethod(python_board_instance, "key_press", "iss", keyval, commit_str, preedit_str);

  if (result==NULL) return FALSE;

  if (PyInt_Check(result) && (PyInt_AsLong(result)>0)){
    Py_DECREF(result);
    return TRUE;
  } else {
    Py_DECREF(result);
    return FALSE;
  }
}

/*
 * OK button pressed
 */
static void pythonboard_ok (void){
  PyObject* result = NULL;
  result = PyObject_CallMethod(python_board_instance, "ok", NULL);
  if( result != NULL){
    Py_DECREF(result);
  } else {
    PyErr_Print();
  }
}

/*
 * Set Level
 */
static void pythonboard_set_level (guint level){
  PyObject* result = NULL;

  result = PyObject_CallMethod(python_board_instance, "set_level", "i", level);
  if( result != NULL){
    Py_DECREF(result);
  } else {
    PyErr_Print();
  }
}

/*
 * Config
 */
static void pythonboard_config(void){
  PyObject* result = NULL;
  result = PyObject_CallMethod(python_board_instance, "config", NULL);
  if( result != NULL){
    Py_DECREF(result);
  } else {
    PyErr_Print();
  }
}

/*
 * Repeat
 */
static void pythonboard_repeat (void){
  PyObject* result = NULL;
  result = PyObject_CallMethod(python_board_instance, "repeat", NULL);
  if( result != NULL){
    Py_DECREF(result);
  } else {
    PyErr_Print();
  }
}

/*
 * Start the board config_start.
 * In this case:
 * - initialize python interpreter
 * - import gcompris functions/objects
 * - import gtk/gnome functions/objects
 * - load the python written board
 * - call the board start function
 */

/*
 * Normally python in already runningwhen config_start is called. If not config_stop has to stop it.
 */

static gboolean python_run_by_config = FALSE;

static void
pythonboard_config_start (GcomprisBoard *agcomprisBoard,
			  GcomprisProfile *aProfile
			  ){
  PyObject* py_function_result;
  PyObject* module_dict;
  PyObject* py_boardclass;
  PyObject* py_boardclass_args;
  PyObject* globals;
  static char *python_args[]={ "" };
  static char* python_prog_name="gcompris";
  char* boardclass;
  char* board_file_name;
  PyObject* main_module;
  char* boarddir;
  gchar *userplugindir;

  g_assert (agcomprisBoard != NULL);
 
  if(!Py_IsInitialized()){
    /* Initialize the python interpreter */
    Py_SetProgramName(python_prog_name);
    Py_Initialize();
    
    PySys_SetArgv(1, python_args);
    
    init_pygobject();

    main_module = PyImport_AddModule("__main__");
    globals = PyModule_GetDict(main_module);
    
    if(globals==NULL){
      g_print("Cannot get info from the python interpreter. Seems there is a problem with this one.\n");
      return;
    } else {
      gcomprisBoard_config = agcomprisBoard;
    }

    /* Add the python plugins dir to the python's search path */
#ifndef DISABLE_USER_PLUGIN_DIR
    userplugindir = g_strconcat(g_get_home_dir(), "/.gcompris/Plugins/", NULL);
    boarddir = g_strdup_printf("import sys; sys.path.append('%s/python'); sys.path.append('%s'); sys.path.append('%s')",
			       userplugindir,
			       PYTHON_PLUGIN_DIR,
			       gcomprisBoard_config->board_dir);
#else
    boarddir = g_strdup_printf("import sys; sys.path.append('%s')",PYTHON_PLUGIN_DIR );
#endif

    PyRun_SimpleString(boarddir);
    g_free(boarddir);

#ifndef DISABLE_USER_PLUGIN_DIR
    g_free(userplugindir);
#endif

    /* Load the gcompris modules */
    python_gcompris_module_init();

    python_run_by_config = TRUE;
   
  }
  else {
    main_module = PyImport_AddModule("__main__"); /* Borrowed reference */
    globals = PyModule_GetDict(main_module); /* Borrowed reference */
  }

  /* Python is now initialized we create some usefull variables */
  board_file_name = strchr(agcomprisBoard->type, ':')+1;
  boardclass = g_strdup_printf("Gcompris_%s", board_file_name);
  
  /* Insert the board module into the python's interpreter */
  python_board_config_module = PyImport_ImportModuleEx(board_file_name,
						       globals,
						       globals,
						       NULL);

  if(python_board_config_module!=NULL){
    /* Get the module dictionnary */
    module_dict = PyModule_GetDict(python_board_config_module);

    /* Get the python board class */
    py_boardclass = PyDict_GetItemString(module_dict, boardclass);

    /* Create a python gcompris board */
    python_gcomprisBoard_config=gcompris_new_pyGcomprisBoardObject(agcomprisBoard);
    /* Create an instance of the board class */
    py_boardclass_args = PyTuple_New(1);
    Py_INCREF(python_gcomprisBoard_config);
    PyTuple_SetItem(py_boardclass_args, 0, python_gcomprisBoard_config);
    python_board_config_instance = PyInstance_New(py_boardclass, py_boardclass_args, NULL);
    Py_DECREF(py_boardclass_args);

    py_function_result = PyObject_CallMethod(python_board_config_instance, 
					     "config_start", 
					     "O", 
					     gcompris_new_pyGcomprisProfileObject(aProfile));

    if( py_function_result != NULL){
      Py_DECREF(py_function_result);
    } else {
      PyErr_Print();
    }
  } else {
    PyErr_Print();
  }
  
  g_free(boardclass);
}

/*
 * End the board.
 * In this case:
 * - call the board end function
 * - finalise python interpreter
 */
static void pythonboard_config_stop (void){
  PyObject* result = NULL;

  if(python_gcomprisBoard_config!=NULL){
    result = PyObject_CallMethod(python_board_config_instance, "config_stop", NULL);
    if( result == NULL){
      PyErr_Print();
    } else {
      Py_DECREF(result);
    }
    Py_XDECREF(python_board_config_module);
    Py_XDECREF(python_board_config_instance);
    Py_XDECREF(python_gcomprisBoard_config);
    if (python_run_by_config){
      Py_Finalize();
      python_run_by_config = FALSE;
    }
  }
}
