/*
---------------------------------------------------------------------------
 $Id: tspc.c,v 1.8.2.1 2003/09/10 05:44:23 jpicard Exp $
---------------------------------------------------------------------------
Copyright (c) 2001-2003 Hexago Inc. All rights reserved.

     The contents of this file are subject to the Hexago
     Public License Version 1.0 (the "License"); you may not
     use this file except in compliance with the License.

     Software distributed under the License is distributed on
     an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either
     express or implied. See the License for the specific
     language governing rights and limitations under the
     License.

     The Original Code is _source code of the tunnel server
     protocol-client side_.

     The Initial Developer of the Original Code is Hexago .

     All Rights Reserved.

     Contributor(s): ______________________________________.

---------------------------------------------------------------------------
*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#ifdef sun
#include <sys/sockio.h>
#endif
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <regex.h>
#include <netdb.h>
#include <ctype.h>

#include "protocol.h"
#include "config.h"
#include "xmltsp.h"
#include "log.h"
#include "network.h"
#include "version.h"
#include "auth.h"
#include "tspc.h"

#define Identification "tspc - Tunnel Server Protocol Client\n"

char *FileName  = "tspc.conf";
char *LogFile   = "tspc.log";
char *ScriptDir = NULL;
char *LogFileName = NULL;
tConf Conf;
tTunnel t;

int Verbose = 0;

/* #define Display(a,b) if(Verbose >= a) printf(b) */

#ifdef WIN32
#  include <getopt.h>
#  include <process.h>
   char *TspHomeDir  = "C:\\TSP";
   char DirSeparator = '\\';
   int  RootUid = 0;
#  define ScriptExtension "bat"
#else
   char *TspHomeDir  = "/usr/local/etc/tsp";
   char DirSeparator = '/';
   int  RootUid = 0;
#  define ScriptInterpretor "/bin/sh "
#  define ScriptExtension "sh"
#endif

#ifdef linux
  void SetEnv(char *Variable, char *Value, int Flag) {
    if(Verbose) printf("%-30.30s   %s\n", Variable, Value);
      setenv(Variable, Value, Flag);
    }
#else
  void SetEnv(char *Variable, char *Value, int Flag)
  {
    char *buf;
    if(Value) {
      int size=(strlen(Variable) + strlen(Value) + 2);
        if((buf=malloc(size)) == NULL) {
          Display(1, ELError, "SetEnv", "Not enough memory!");
          exit(1);
        }
      snprintf(buf, size, "%s=%s", Variable, Value);
      putenv(buf);

#  ifndef solaris
#    ifndef WIN32
      free(buf);
#    endif 
#  endif
      if(Verbose) printf("%-30.30s   %s\n", Variable, Value);
    }
  }
#endif


#define Filename	"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_."
#define ServerString	"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-.:"
#define DNSLegal	"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-."
#define Alphanum	"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"
#define Numeric		"0123456789"
#define IPv6Addr	"ABCDEFabcdef:0123456789"
#define IPv4Addr	".0123456789"

void Display(int VerboseLevel, enum tErrorLevel el, char *func, char *format, ...) {
  va_list     argp;

  int i, j;
  char fmt[5000];
  char clean[5000];

  va_start(argp, format);
  vsnprintf(fmt, sizeof fmt, format, argp);
  va_end(argp);

  /* Change CRLF to LF for log output */
  for (i = 0, j = 0; i < sizeof fmt; i++) {
    if (fmt[i] == '\r' && fmt[i + 1] == '\n')
      continue;
    clean[j++] = fmt[i];
    if (fmt[i] == '\0')
      break;
  }

  /*
     output to stdout if verbose on
  */
  if(VerboseLevel <= Verbose) {
     printf("%s\n", clean);
  }
  LogPrintf(el, func, clean);

}

void PrintUsage(char *Message, ...) {
   printf(Identification);
   if(Message) {
      va_list     argp;
      va_start(argp, Message);
      vprintf(Message, argp);
      va_end(argp);
   }

   printf("usage: tspc [options] [-f config_file] [-r seconds]\n");
   printf("  where options are :\n");
   printf("    -v    set verbose level (-v,-vv or -vvv)\n");
   printf("    -i    gif interface to use for tunnel\n");
   printf("    -s    interface to query to get IPv4 source address\n");
   printf("    -f    Read this config file instead of %s \n", FileName);
   printf("    -r    Retry after n seconds until success\n");
   printf("    -h    help\n");
   printf("    -?    help\n");
   printf("\n");
   exit(1);
}

void ParseArguments(int argc, char *argv[], tConf *Conf) {
    int ch;

   while ((ch = getopt(argc, argv, "h?vf:r:i:s:")) != -1) {
      switch (ch) {
      case 'v':
         Verbose++;
         break;
      case 's':
         Conf->if_source = optarg;
         break;
      case 'i':
         Conf->if_tunnel = optarg;
         break;
      case 'f':
         FileName = optarg;
         break;
      case 'r':
	 Conf->retry = atoi(optarg);
	 break;
      case '?':
      case 'h':
         PrintUsage(NULL);
      default:
         PrintUsage("Invalid option");
      }
   }
}

/*
   Initialize with default values, read configuration file and override
   defaults with config file values.
*/
int Initialise(int argc, char *argv[], tConf *Conf)
{
  tConf CmdLine;
  char *Templ = "template";

  Conf->tsp_dir      = NULL;
  Conf->server       = "";
  Conf->userid       = "anonymous";
  Conf->passwd       = "";
  Conf->client_v4    = "auto";
  Conf->tsp_version  = "";
  Conf->template     = "";
  Conf->dns_server   = "";
  Conf->auth_method  = "any";
  Conf->protocol     = "default_route";
  Conf->if_tunnel    = "";
  Conf->if_source    = "";
  Conf->routing_info = "";
  Conf->if_prefix    = "";
  Conf->host_type    = "host";
  Conf->prefixlen    = 0;
  Conf->retry	     = 0;

  memset(&CmdLine, 0, sizeof(CmdLine));

  if(argc>1) {
      ParseArguments(argc, argv, &CmdLine);
  }

  if(ReadConfigFile(FileName, Conf))
     return 1;
  if(CmdLine.if_tunnel) Conf->if_tunnel = CmdLine.if_tunnel;
  if(CmdLine.if_source) Conf->if_source = CmdLine.if_source;
  if(Conf->tsp_dir) {
     TspHomeDir = Conf->tsp_dir;
     if((ScriptDir = (char *)malloc((size_t)(strlen(Conf->tsp_dir)+strlen(Templ)+2)))==NULL) {
        Display(1, ELError, "Initialise", "Not enough memory!");
        return 1;
     }
     sprintf(ScriptDir, "%s%c%s", Conf->tsp_dir, DirSeparator, Templ);
     if((LogFileName = (char *)malloc((size_t)(strlen(Conf->tsp_dir)+strlen(LogFile)+2)))==NULL) {
        Display(1, ELError, "Initialise", "Not enough memory!");
        return 1;
     }
     sprintf(LogFileName, "%s%c%s", Conf->tsp_dir, DirSeparator, LogFile);
  } else {
     if((ScriptDir = (char *)malloc((size_t)(strlen(TspHomeDir)+strlen(Templ)+2)))==NULL) {
        Display(1, ELError, "Initialise", "Not enough memory!");
        return 1;
     }
     sprintf(ScriptDir, "%s%c%s", TspHomeDir, DirSeparator, Templ);
     LogFileName = LogFile;
  }
     
  return 0;
}

/*
   Check if all characters in Value are within AllowedChars list.
*/
int IsAll(char *AllowedChars, char *Value)
{
   if(Value) {
      for(;*Value; Value++) {
         if(strchr(AllowedChars, *Value) == NULL)
            return NULL;
      }
   } else {
      return NULL;
   }
   return 1;
}

/*
   Check to see if there is a value in the char *
   If not, then the value was not supplied.
*/
int IsPresent(char *Value)
{
   if(Value)
      if(strlen(Value))
         return 1;

   return 0;
}

/*
  Validate IPv4 address
*/
int validate_ipv4(char *Addr) {
  char *cur;
  int value=0;
  int digit=1;

  if (Addr == NULL)
    return -1;

  for(cur=Addr;*cur;cur++) {
    if (isdigit(*cur)) {
      value = value*10 + (*cur - '0');
      if (value > 255)
	return -1;
    }
    else if (*cur == '.') {
      digit++;
      if (digit > 4)
	return -1;
      value=0;
    }
    else {
      return -1;
    }
  }
  if (digit != 4)
    return -1;
  return(0);
}

/*
   Validate all members of tConf.
*/

int VerifyConfig(tConf *Conf)
{
  int status;
# define eq(a,b) (strcmp(a,b) == 0)

  status = 0;

  if(IsPresent(Conf->tsp_version)) {
    if (strcmp(Conf->tsp_version, TSP_VERSION)) {
       Display(1, ELError, "VerifyConfig", "Bad config version number, expecting '%s', read '%s'", TSP_VERSION, Conf->tsp_version);
      status = 1;
    }
  } else {
    Display(1, ELError, "VerifyConfig", "tsp_version mandatory");
   status = 1;
  }

  if(IsPresent(Conf->client_v4)) {
     if(strcmp(Conf->client_v4, "auto")) { /* If not auto then check if ip address is ok. */
        if(validate_ipv4(Conf->client_v4) == -1) {
           Display(1, ELError, "VerifyConfig", "invalid ipv4 address in client_v4 ->[%s]", Conf->client_v4);
           status = 1;
        }
     } else {
        if(IsPresent(Conf->if_source)) {
           struct ifreq request;
           struct in_addr in;
           int fd;

           fd = socket(AF_INET, SOCK_DGRAM, 0);
  
           /* get the IP address */
           if(strlen(Conf->if_source) < (sizeof request.ifr_name)-1 )
              strcpy(request.ifr_name, Conf->if_source);
           else {
              Display(1, ELError, "VerifyConfig", "invalid source interface name [%s].", Conf->if_source);
              status = 1;
           }
           if (ioctl(fd, SIOCGIFADDR, &request) < 0) {
              Display(1, ELError, "VerifyConfig", "invalid source interface name [%s].", Conf->if_source);
              status = 1;
           }
           in.s_addr = ((struct sockaddr_in *) (&request.ifr_addr))->sin_addr.s_addr;
           Conf->client_v4 = inet_ntoa(in);
           Display(1, ELInfo, "VerifyConfig", "Using [%s] as source IPv4 address taken from %s.", Conf->client_v4, Conf->if_source);
        } /* else {
           Display(1, ELError, "VerifyConfig", "Source interface must be specified for client_v4=auto");
           status = 1;
        } */
     }
  } else {
    Display(1, ELError, "VerifyConfig", "client_v4 must be specified!");
    status = 1;
  }

  if(IsPresent(Conf->server)) {
     if(IsAll(ServerString, Conf->server) == 0) {
        Display(1, ELError, "VerifyConfig", "invalid server name.");
        status = 1;
     }
  } else {
    Display(1, ELError, "VerifyConfig", "server must be specified!");
    status = 1;
  }
  if(IsPresent(Conf->userid)) {
     if(strlen(Conf->userid) > 63) {
        Display(1, ELError, "VerifyConfig", "userid too long: must be less than 63 bytes");
        status = 1;
     } else if(IsAll(DNSLegal, Conf->userid) == 0) {
        Display(1, ELError, "VerifyConfig", "illegal chars in userid.");
        status = 1;
     }
     if(strcmp(Conf->userid, "anonymous")) {
        if(IsPresent(Conf->passwd)==0) {
           Display(1, ELError, "VerifyConfig", "passwd must be specified!");
           status = 1;
        }
     }
  } else {
    Display(1, ELError, "VerifyConfig", "userid must be specified!");
    status = 1;
  }
  if(IsPresent(Conf->if_tunnel)) {
     if(IsAll(DNSLegal, Conf->if_tunnel) == 0) {
        Display(1, ELError, "VerifyConfig", "illegal chars in if_tunnel.");
        status = 1;
     }
  } else {
    Display(1, ELError, "VerifyConfig", "if_tunnel must be specified!");
    status = 1;
  }

  if(IsPresent(Conf->template)) {
     if(IsAll(Filename, Conf->template) == 0) {
        Display(1, ELError, "VerifyConfig", "invalid template file name.");
        status = 1;
     }
  } else {
    Display(1, ELError, "VerifyConfig", "template must be specified!");
    status = 1;
  }

  if(IsPresent(Conf->host_type)) {
     if (!(eq(Conf->host_type, "host") ||  eq(Conf->host_type, "router"))) {
       Display(1, ELError, "VerifyConfig", "Bad host_type, expecting 'host' or 'router', read '%s'", Conf->host_type);
       status = 1;
     }
  }
  
  if  ( !((Conf->prefixlen == 0) ||
	  (Conf->prefixlen == 48) || 
	  (Conf->prefixlen == 64))) {
    Display(1,ELError, "VerifyConfig", "Bad prefix length, expecting '0', '48' or '64', read '%d'", Conf->prefixlen);
    status = 1;
  }

  if(IsPresent(Conf->host_type)) {
     if(!(eq(Conf->protocol, "default_route") ||
   	eq(Conf->protocol, "bgp"))) {
       Display(1,ELError, "VerifyConfig", "Bad routing protocol, expecting 'default_route' or 'bgp'");
       status = 1;
     }
  }
  if(IsPresent(Conf->dns_server)) {
     char *Server;
     char *dns = strdup(Conf->dns_server);
     if (eq(Conf->host_type, "host")) {
       Display(1,ELError, "VerifyConfig", "DNS delegation is not support for host_type=host");
       status = 1;
     }
     for(Server = strtok(dns, ":");Server; Server = strtok(NULL, ":")) {
        if(gethostbyname(Server) == NULL) {
           Display(1,ELError, "VerifyConfig", "DNS server name %s is not resolving.", Server);
           status = 1;
        }
     }
     free(dns);
  }

  return status;  
}

/*
   Create XML request for tunnel
*/
char *BuildCreateRequest()
{
  static char *Request[5000], Buffer[1024];

  /* XXX: This routine may overflow Request */
  memset(Request, 0, sizeof(Request));
  strcpy((char *)Request, "<tunnel action=\"create\" type=\"v6v4\">\r\n");
  strcat((char *)Request, " <client>\r\n");

  snprintf(Buffer, sizeof Buffer,
           "  <address type=\"ipv4\">%s</address>\r\n", Conf.client_v4);
  strcat((char *)Request, Buffer);

  /* ------------------------------------------------------- */
  /*                 ROUTER SECTION                          */
  /* ------------------------------------------------------- */
  if (strcmp(Conf.host_type, "router") == 0) {

    strcat((char *)Request, "  <router");

    if (strcmp(Conf.protocol, "default_route") != 0) {
      snprintf(Buffer, sizeof Buffer,
	      " protocol=\"%s\"",
	      Conf.protocol);
      strcat((char *)Request, Buffer);
    }

    strcat((char *)Request, ">\r\n");

    if (Conf.prefixlen==0) {
       Conf.prefixlen = 48; /* default to 48*/
    }
    snprintf(Buffer, sizeof Buffer,
	      "   <prefix length=\"%d\"/>\r\n",
	      Conf.prefixlen);
    strcat((char *)Request, Buffer);

    /* ------------------------------------------------------- */
    /*                 REVERSE DNS DELEGATION                  */
    /* ------------------------------------------------------- */
    if(strlen(Conf.dns_server)) {
       char *Server;
       strcat((char *)Request, "   <dns_server>\r\n");
       for(Server = strtok(Conf.dns_server, ":");Server; Server = strtok(NULL, ":")) {
          snprintf(Buffer, sizeof Buffer,
              "     <address type=\"dn\">%s</address>\r\n", Server);
          strcat((char *)Request, Buffer);
       }
       strcat((char *)Request, "   </dns_server>\r\n");
    }

    /* ------------------------------------------------------- */
    /*                 ROUTING PROTOCOL SECTION                */
    /* ------------------------------------------------------- */
    if (strcmp(Conf.protocol, "default_route") == 0) {
      if (strlen(Conf.routing_info) > 0) {
	snprintf(Buffer, sizeof Buffer,
		"    %s\r\n", 
		Conf.routing_info);
	strcat((char *)Request, Buffer);
      }
    }

    strcat((char *)Request, "  </router>\r\n");
  }

  strcat((char *)Request, " </client>\r\n");
  strcat((char *)Request, "</tunnel>\r\n");

  return (char *)Request;
}

/*
   Call the XML parser. Data will be contained in the
   structure t (struct tTunnel)
*/
int ExtractPayload(char *Payload)
{
  char *p; /* Fisrt byte of payload. */
  int   rc;

  memset(&t, 0, sizeof(t));
  
  Display(1, ELInfo, "ExtractPayload", "Process response from server\n");
  Display(2, ELInfo, "ExtractPayload", "Response:\n");
  Display(2, ELInfo, "ExtractPayload", Payload);
  if((p = strchr(strchr(Payload, '\n'), '<')) == NULL)
     return 1;
  if((rc = tspXMLParse(p, &t)) != 0)
     return 1;
  
/*  ShowInfo(&t); */

  return(0);
}

/*
   Setup environnement variables and call template script to setup tunnel
*/
int SetUpInterface()
{
  char buf[1024];
  int Err = 0;

#ifndef WIN32
  if(getuid() != RootUid) {
     Display(0, ELInfo, "SetUpInterface", "You are not root. Interface will not be configured.");
     return 0;
  }
#endif

  SetEnv("TSP_HOST_TYPE",           Conf.host_type, 1);
  SetEnv("TSP_TUNNEL_INTERFACE",    Conf.if_tunnel, 1);
  SetEnv("TSP_HOME_INTERFACE",      Conf.if_prefix, 1);
  if(IsAll(IPv4Addr, t.client_address_ipv4))
     SetEnv("TSP_CLIENT_ADDRESS_IPV4", t.client_address_ipv4, 1);
  else {
     Display(1, ELError, "SetUpInterface", "Bad value received from server for client_address_ipv4.\n");
     Err++;
  }
  if(IsAll(IPv6Addr, t.client_address_ipv6))
     SetEnv("TSP_CLIENT_ADDRESS_IPV6", t.client_address_ipv6, 1);
  else {
     Display(1, ELError, "SetUpInterface", "Bad value received from server for client_address_ipv6.\n");
     Err++;
  }
  if(IsAll(IPv4Addr, t.server_address_ipv4))
     SetEnv("TSP_SERVER_ADDRESS_IPV4", t.server_address_ipv4, 1);
  else {
     Display(1, ELError, "SetUpInterface", "Bad value received from server for server_address_ipv4.\n");
     Err++;
  }
  if(IsAll(IPv6Addr, t.server_address_ipv6))
     SetEnv("TSP_SERVER_ADDRESS_IPV6", t.server_address_ipv6, 1);
  else {
     Display(1, ELError, "SetUpInterface", "Bad value received from server for server_address_ipv6.\n");
     Err++;
  }
  SetEnv("TSP_TUNNEL_PREFIXLEN",    "128", 1);
  if(t.prefix) {
     if(IsAll(IPv6Addr, t.prefix)) {
        char chPrefix[128];
        int len, sep;

        len = (atoi(t.prefix_length) %  4) ? (atoi(t.prefix_length) /  4) +1 : (atoi(t.prefix_length) / 4);
        sep = (atoi(t.prefix_length) % 16) ? (atoi(t.prefix_length) / 16) : (atoi(t.prefix_length) / 16)-1;

        memset(chPrefix, 0, 128);
        memcpy(chPrefix, t.prefix, len+sep);
        SetEnv("TSP_PREFIX",              chPrefix, 1);
     } else {
        Display(1, ELError, "SetUpInterface", "Bad value received from server for prefix.\n");
        Err++;
     }
     if(IsAll(Numeric, t.prefix_length))
        SetEnv("TSP_PREFIXLEN",           t.prefix_length, 1);
     else {
        Display(1, ELError, "SetUpInterface", "Bad value received from server for prefix_length.\n");
        Err++;
     }
  }

  if(Err) {
    Display(1,ELError, "SetUpInterface", "Errors in server response.");
    return 1;
  }
  snprintf(buf, sizeof buf, "%d", Verbose);
  SetEnv("TSP_VERBOSE",           buf, 1);
  SetEnv("TSP_HOME_DIR",          TspHomeDir, 1);

#ifdef WIN32
  snprintf(buf, sizeof buf, "%s%c%s.%s", ScriptDir, DirSeparator, Conf.template, ScriptExtension);
#else
  snprintf(buf, sizeof buf, "%s %s%c%s.%s", ScriptInterpretor, ScriptDir, DirSeparator, Conf.template, ScriptExtension);
#endif

  Display(2, ELInfo, "SetUpInterface", "Executing configuration script.\n");
  Display(2, ELInfo, "SetUpInterface", buf);
  Display(2, ELInfo, "SetUpInterface", "\n");
#ifdef WIN32
/* return code not good with cygwin.*/
/*if(spawnl(0, buf, NULL)!=0) */
  spawnl(_P_WAIT, buf, buf, NULL);
#else
/*if(system(buf))*/
  system(buf);
#endif
/*
  {
    Display(1,ELError, "SetUpInterface", "Error while executing script %s", buf);
    return 1;
  }
*/

  return 0;
}

/* Connect to the specified server */
int TryServer(char *server)
{
  int      socket;
  int      status;
  tPayload Payload;

  if ((socket = tspConnect(server)) == 0) {
    Display(1,ELError, "TryServer", "Unable to connect to server %s", server);
    return -1;
  }

  if (Authenticate(socket, Conf.auth_method, Conf.userid, Conf.passwd, Conf.server) != 0) {
    tspClose(socket); 
    return 1;
  }

  if(strcmp(Conf.client_v4, "auto")==0) {
     struct sockaddr_in addr;
     int len;

     len = sizeof addr;
     if (getsockname(socket, (struct sockaddr *)&addr, &len) < 0) {
        perror("getsockname");
        Display(1, ELError, "TryServer", "Error while trying to find source ip address.");
        NetClose(socket);
        return 1;
     }
     Conf.client_v4 = inet_ntoa(addr.sin_addr);
     Display(1, ELInfo, "TryServer", "Using [%s] as source IPv4 address.", Conf.client_v4);
  }
  memset(&Payload, 0, sizeof(tPayload));
  Payload.payload = tspAddPayloadString(&Payload, BuildCreateRequest());
  Display(1, ELInfo, "TryServer", "Send request\n");
  Display(2, ELInfo, "TryServer", "Request:\n");
  Display(2, ELInfo, "TryServer", Payload.payload);

  if (tspSend(socket, &Payload) != 0) {
    tspClose(socket);
    return 1;
  }

  if (tspReceive(socket, &Payload) != 0) {
    tspClose(socket);
    return 1;
  }

  if ((status = tspGetStatusCode(Payload.payload)) != 200) {
    tspClose(socket);
 
    Display(0,ELError, "TryServer", "Status code from server: %s", tspGetStatusMsg(Payload.payload));
    if(status >= 1000) {
      extern int TryOtherServers(char *payload);
      Display(1,ELInfo, "TryServer", "Trying other servers...");
      return TryOtherServers(Payload.payload);
    }

    return 1;
  }

  tspClose(socket);

  ExtractPayload(Payload.payload);
  if(SetUpInterface())
     return -1;
   
  return 0;
}

/* Extract list of alternate server from XML response and try to connect to them. */
int TryOtherServers(char *payload)
{
  tLinkedList *server;
  int not_done = -1;

  ExtractPayload(payload);
  for(server = t.broker_address_ipv4; server && not_done; server = server->next) {
      if(server->Value) {
        Display(1,ELInfo, "TryOtherServers", "Trying suggested server : %s", server->Value);
        not_done = TryServer(server->Value);
      }
  }
  return not_done;
}

int main(int argc, char *argv[])
{
  int status=1;

  LogInit("TSPClient", LogFile);  
  Display(1,ELInfo, "main", "----- TSP Client Version %s Initializing -------------------------", TSP_VERSION);

  if((status = Initialise(argc, argv, &Conf)) != 0)
     goto endtspc;
  Display( 1, ELInfo, "main", Identification); 

  Display( 1, ELInfo, "main", "Loading configuration file\n"); 
  if((status = VerifyConfig(&Conf)) != 0)
     goto endtspc;

  Display( 1, ELInfo, "main", "Connecting to server\n"); 
  status = TryServer(Conf.server);
  if (Conf.retry != 0) {
	  while (status == -1) {
		  Display(1, ELInfo, "main", "\nRetry to connect to server in %d seconds...\n", Conf.retry);
		  sleep(Conf.retry);
		  status = TryServer(Conf.server);
	  }
  }

  endtspc:
  if(status) 
	  Display(0,ELInfo, "main", "No ipv6 tunnel has been set (use -h for help)\n");
  LogClose();

  if(Verbose) printf("Exiting with return code : %d (0 = no error)\n", status);
  exit(status);
}

/*----- tspc.c ----------------------------------------------------------------------------------*/
