/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdos.so
 *
 *   File: discovery.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "ptables.h"
#include "defsegmgr.h"
#include "discovery.h"
#include "segs.h"
#include "checks.h"
#include "commit.h"
#include "display.h"
#include "os2dlat.h"
#include "sn.h"
#include "embedded.h"
#include "bsd.h"
#include "unixware.h"
#include "solarisX86.h"
#include "seg_geometry.h"
#include "dm.h"
#include "resolver.h"

/*
 *  Called to see if the drive has a MS-DOS partitioning scheme. This is
 *  a simple MBR+EBR chain ... and possibly ... OS2 dlat sectors.
 */
static boolean isa_disk_with_msdos_partitions( LOGICALDISK *ld, Master_Boot_Record  *mbr )
{
        int rc=-1;
        DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
        geometry_t  guess;


        LOG_ENTRY();

        // first off ... look for msdos signature
        if ( has_msdos_signature( mbr )==FALSE) {
                LOG_DEBUG("disk %s has no msdos signature\n", ld->name);
                LOG_EXIT_BOOL(FALSE);
                return FALSE;
        }

        /*
         * Check for a disk that hasnt been partitioned yet. We will not
         * claim such a disk unless explicitly assigned to the disk at
         * a later time by an ASSIGN task.
         */
        if ( ptable_has_partition_records(ld,mbr) == TRUE ) {

                // Look for chain of valid OS/2 dlat sectors
                if ( isa_os2_partitioned_disk(ld, mbr, 0, 0)== TRUE ) {

                        disk_pdata->flags |= DISK_HAS_OS2_DLAT_TABLES;

                }

                // test if the disk is using LBA addressing for the partition records
                if ( disk_uses_lba_addressing(ld) == TRUE ) {

                        disk_pdata->flags |= DISK_USES_LBA_ADDRESSING;

                }

                // All that remains is to check the partition tables
                if ( disk_pdata->flags & DISK_USES_LBA_ADDRESSING ) {
                        rc = isa_valid_partition_table_chain(ld, mbr, 0, 0, FALSE, 0, TRUE);  // final call cuz not using CHS
                }
                else {
                        rc = isa_valid_partition_table_chain(ld, mbr, 0, 0, FALSE, 0, FALSE); // not final cuz we can try alternalte
                                                                                              // geometry for the drive.
                }

                if ( rc == EAGAIN ) {   // if we failed and can try again ...

                        rc = seg_geometry_guess(ld, &guess );
                        if (rc == 0) {
				char number_buffer1[64];
				char number_buffer2[64];

                                // success, so switch to new geometry
                                disk_pdata->geometry.cylinders         = guess.cylinders;
                                disk_pdata->geometry.sectors_per_track = guess.sectors_per_track;
                                disk_pdata->geometry.heads             = guess.heads;

				sprintf(number_buffer1, "%"PRIu64, ld->geometry.cylinders);
				sprintf(number_buffer2, "%"PRIu64, disk_pdata->geometry.cylinders);
                                MESSAGE( _("Warning:  Using an alternate geometry (Cylinders, Heads, Sectors) for drive %s.\n"
					   "The kernel reported drive geometry is: C= %s H= %d S= %d\n"
					   "The partition records report a geometry of: C= %s H= %d S= %d\n"
					   "Using the alternate geometry reported by the partition records.\n"),
                                         ld->name,
                                         number_buffer1,
                                         ld->geometry.heads,
                                         ld->geometry.sectors_per_track,
                                         number_buffer2,
                                         disk_pdata->geometry.heads,
                                         disk_pdata->geometry.sectors_per_track );

                        }

                        else if (rc == EPROTO) {

                                // partial success, had to ignore CHS errors
                                disk_pdata->geometry.cylinders         = guess.cylinders;
                                disk_pdata->geometry.sectors_per_track = guess.sectors_per_track;
                                disk_pdata->geometry.heads             = guess.heads;

                                rc = 0;
                        }


                        // last attempt will be to use LBA addressing for the drive
                        if (rc==-1) {

                                disk_pdata->flags |= DISK_USES_LBA_ADDRESSING;

                                rc = isa_valid_partition_table_chain(ld, mbr, 0, 0, FALSE, 0, TRUE);
                                if ( rc == 0 ) {

                                        disk_pdata->flags |= DISK_HAS_FORCED_LBA_ADDRESSING;

                                        MESSAGE(_("Warning:  Forced to use LBA addressing on drive %s due to inconsistant geometry.\nChanges will not be committed on this drive.\n"),
                                                ld->name);

                                }

                        }

                }

        }
        else {
                LOG_DETAILS("%s is not partitioned\n",ld->name);

                // Look for chain of valid OS/2 dlat sectors
                if ( isa_os2_partitioned_disk(ld, mbr, 0, 0)== TRUE ) {

                        disk_pdata->flags |= DISK_HAS_OS2_DLAT_TABLES;

                }

                rc = 0; // return success and claim this disk that doesnt
                        // seem to have any partitions on it yet.

        }

        if (rc == 0) {
                LOG_EXIT_BOOL(TRUE);
                return TRUE;
        }
        else {
                LOG_EXIT_BOOL(FALSE);
                return FALSE;
        }

}






/*
 *  Called by get_DiskSegments() when an extended partition is encountered.
 *  during disk segment discovery.
 *
 *  Part of walking a drives partition tables. This routine will walk an
 *  EBR chain, hanging DOS logical drives off the evms logical disk.
 *
 *  A DOS logical drives consists of an EBR sector and a logical partition.
 *  The EBR will actually consume the entire track because the partitions
 *  start on track boundaries.
 *
 *  This routine will hang the following segments off the logical disk:
 *
 *  EBR Meta Data Seg - describes the EBR track
 *  Evms Data Seg     - describes a logical partition
 *
 */
static int get_logical_drives(  LOGICALDISK           *ld,                  // drive
                                Extended_Boot_Record  *boot_sector,         // EBR boot sector buffer
                                DISKSEG               *ebr )                // EBR DISKSEG
{
        struct plugin_functions_s  *dft;
        Partition_Record     *part   = NULL;

        DISKSEG              *log_part_seg = NULL;
        SEG_PRIVATE_DATA     *log_part_pdata = NULL;

        SEG_PRIVATE_DATA     *ebr_pdata = (SEG_PRIVATE_DATA *) ebr->private_data;

        DISK_PRIVATE_DATA    *disk_pdata = get_disk_private_data(ld);

        DISKSEG              *new_ebr_seg;
        SEG_PRIVATE_DATA     *new_ebr_seg_pdata;

        Extended_Boot_Record *ebr_buffer;
        lba_t                 ebr_lba;

        int                   i;
        int                   rc;



        LOG_ENTRY();

        // get disk manager function table
        dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
        if (dft==NULL) {
                LOG_EXIT_INT(ENODEV);
                return ENODEV;
        }

        // hang logical partitions first ...
        for (i=0; i<4; i++) {
                part = &boot_sector->Partition_Table[i];

                if (  ( isa_null_partition_record( part )==TRUE ) ||
                      ( isa_ebr_partition_record(part) == TRUE ) ) {
                        continue;
                }

                log_part_seg = build_diskseg_from_partition_record(ld, part, ebr, i, FALSE );
                if (log_part_seg==NULL) {
                        LOG_EXIT_INT(ENOMEM);
                        return ENOMEM;
                }

                log_part_pdata = (SEG_PRIVATE_DATA *) log_part_seg->private_data;

                // increment count of logical drives
                ++disk_pdata->logical_drive_count;

                // find partition number for this logical partition, they
                // are numbered 5-n
                log_part_pdata->part_number = ebr_pdata->ebr_number + 5;

                // hang seg off disk
                if ( insert_diskseg_into_list(ld->parent_objects, log_part_seg) == NULL ) {
                        free_disk_segment(log_part_seg);
                        LOG_EXIT_INT(ENOMEM);
                        return ENOMEM;
                }

                // if the logical partition is offset from its EBR then shrink the
                // EBR segment to a single sector ... like other partitioning tools.
                if ( (log_part_seg->start - ebr->start) > disk_pdata->geometry.sectors_per_track ) {
                        ebr->size = 1;
                }

        }


        //   Now follow the EBR chain. Look for an extended partition record
        //   that chains us to another logical drive.
        for (i=0; i<4; i++) {

                part = &boot_sector->Partition_Table[i];

                if ( isa_ebr_partition_record(part) == TRUE ) {
                        break;
                }
        }

        // did we find an extended partition record ?
        if ( isa_ebr_partition_record(part) == TRUE ) {

                /*
                 * add the starting LBA of this disks extended partitoin to get a
                 * disk relative starting lba for the logical drive.
                 */
                ebr_lba = DISK_TO_CPU32(START_LBA(part)) + disk_pdata->extd_partition_lba;

                // malloc a buffer and read the Extended Boot Record
                ebr_buffer = (Extended_Boot_Record *) malloc(ld->geometry.bytes_per_sector);
                if (ebr_buffer) {

                        rc = dft->read(ld, ebr_lba, 1, (void *) ebr_buffer );
                        if ( rc == 0 ) {

                                // Look for BUGS in the EBR chain ...
                                //
                                // (1) Check if this extended partition record points to a null ptable
                                //     and nuke the terminal extended partition record, finishing the
                                //     logical drive discovery.
                                //
                                // (2) Check if this extended partition record points to a ptable that
                                //     -ONLY- chains to another ebr and nuke the -DO NOTHING- ebr and
                                //     continue following the chain.
                                //
                                if ( ptable_has_partition_records( ld, ebr_buffer ) == TRUE ) {

                                        if ( ptable_has_data_partition_record( ebr_buffer ) == TRUE) {

                                                new_ebr_seg = build_ebr_disk_segment( ld, part, ebr, ebr_lba, i, FALSE );
                                                if (new_ebr_seg == NULL) {
                                                        LOG_EXIT_INT(ENOMEM);
                                                        return ENOMEM;
                                                }

                                                new_ebr_seg_pdata = (SEG_PRIVATE_DATA *)new_ebr_seg->private_data;
                                                new_ebr_seg_pdata->ebr_number = ebr_pdata->ebr_number + 1;

                                                if ( insert_diskseg_into_list(ld->parent_objects, new_ebr_seg) == NULL ) {
                                                        free_disk_segment(new_ebr_seg);
                                                        LOG_EXIT_INT(ENOMEM);
                                                        return ENOMEM;
                                                }

                                                // follow the EBR chain with the new ebr
                                                rc = get_logical_drives(  ld, ebr_buffer, new_ebr_seg );
                                        }
                                        else {

                                                // follow the EBR chain ... loosing the DO_NOTHING ebr
                                                rc = get_logical_drives(  ld, ebr_buffer, ebr );
                                        }

                                }


                        }

                        free(ebr_buffer);
                }
                else {
                        rc = ENOMEM;
                }
        }
        else {
                rc = 0;
        }

        LOG_EXIT_INT(rc);
        return rc;
}





/*
 *  Called by segment discovery code.
 *
 *  After learning that we own the partitioning scheme on the specified
 *  drive, we now need to walk the drive's partition tables and create
 *  DISKSEG structs for each partition record that we find.
 *
 *  This routine will hang the following segments off the logical disk:
 *
 *  MBR MetaData - describes the MBR track
 *  EBR MetaData - describes an EBR track
 *  Data         - could be a primary or logical partition
 *  FreeSpace    - a segment that describes an unallocated region on the disk
 *
 *  This routine is called after examining the disk and determining
 *  that we own responsibility for the partitioning scheme on the disk.
 *  So, we know the partitioning is Valid and need not do very exact
 *  checking on its layout while creating evms segment objects.
 *
 */
static int get_disk_segments(  LOGICALDISK         *ld,                  // drive
                               Master_Boot_Record  *boot_sector )        // master boot sector
{
        struct plugin_functions_s  *dft;
        DISK_PRIVATE_DATA          *disk_pdata    = get_disk_private_data(ld);
        Partition_Record           *part = NULL;
        DISKSEG                    *mbr  = NULL;
        DISKSEG                    *pri_part_seg;
        SEG_PRIVATE_DATA           *ebr_pdata;
        Extended_Boot_Record       *ebr_buffer;
        DISKSEG                    *ebr;
        int                         i,rc;


        LOG_ENTRY();

        // get disk manager function table
        dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
        if (dft==NULL) {
                LOG_EXIT_INT(ENODEV);
                return ENODEV;
        }

        // hang a MBR meta data segment and DLAT off the logical drive
        mbr = build_mbr_disk_segment( ld );
        if (mbr == NULL) {
                LOG_EXIT_INT(ENOMEM);
                return ENOMEM;
        }
        else {

                if ( insert_diskseg_into_list(ld->parent_objects, mbr) == NULL ) {
                        free_disk_segment(mbr);
                        LOG_EXIT_INT(ENOMEM);
                        return ENOMEM;
                }

        }

        // create segment objects for primary partitions ...
        for (i=0; i<4; i++) {

                part = &boot_sector->Partition_Table[i];

                if (  ( isa_null_partition_record( part )==TRUE ) ||
                      ( isa_ebr_partition_record(part) == TRUE ) ) {
                        continue;
                }

                pri_part_seg = build_diskseg_from_partition_record(ld, part, mbr, i, TRUE );
                if (pri_part_seg==NULL) {
                        LOG_EXIT_INT(ENOMEM);
                        return ENOMEM;
                }

                // get partition number for this primary
                ((SEG_PRIVATE_DATA *)pri_part_seg->private_data)->part_number = i+1;

                // insert into segment list
                if ( insert_diskseg_into_list(ld->parent_objects, pri_part_seg) == NULL ) {
                        free_disk_segment(pri_part_seg);
                        LOG_EXIT_INT(ENOMEM);
                        return ENOMEM;
                }

        }

        // look for an extended partition record ...
        for (i=0; i<4; i++) {

                part = &boot_sector->Partition_Table[i];

                if ( isa_ebr_partition_record(part) == TRUE ) {
                        break;
                }

        }

        // if we found an extended partition then try to follow the ebr chain ...
        if ( isa_ebr_partition_record(part) == TRUE ) {

                // save extended partition information
                disk_pdata->flags                 |= DISK_HAS_EXTENDED_PARTITION;
                disk_pdata->extd_partition_size    = DISK_TO_CPU32(NR_SECTS(part));
                disk_pdata->extd_partition_lba     = DISK_TO_CPU32(START_LBA(part));
                disk_pdata->extd_partition_end_lba = DISK_TO_CPU32(START_LBA(part)) + DISK_TO_CPU32(NR_SECTS(part)) - 1;
                disk_pdata->extd_partition_sys_ind = SYS_IND(part);

                // malloc a buffer and read the Extended Boot Sector
                ebr_buffer = (Extended_Boot_Record *) malloc(ld->geometry.bytes_per_sector);
                if (ebr_buffer) {

                        rc = dft->read( ld, DISK_TO_CPU32(START_LBA(part)), 1,(void *) ebr_buffer );
                        if ( rc == 0 ) {

                                // look for logical drives in the extended partition ...
                                if ( ptable_has_partition_records( ld, ebr_buffer ) == TRUE ) {

                                        // build the extended partition segment object
                                        ebr = build_ebr_disk_segment( ld, part, mbr, (lba_t) DISK_TO_CPU32(START_LBA(part)), i, FALSE );
                                        if (ebr == NULL) {
                                                LOG_ERROR("error, unable to build extd partition segment object\n");
                                                LOG_EXIT_INT(ENOMEM);
                                                return ENOMEM;
                                        }

                                        // extended partition is sequence number 0 in ebr chain
                                        ebr_pdata = (SEG_PRIVATE_DATA *)ebr->private_data;
                                        ebr_pdata->ebr_number = 0;

                                        // insert into segment list
                                        if ( insert_diskseg_into_list(ld->parent_objects, ebr) == NULL ) {
                                                free_disk_segment(ebr);
                                                LOG_EXIT_INT(ENOMEM);
                                                return ENOMEM;
                                        }

                                        // now discover logical drives found in the extended partition
                                        rc = get_logical_drives(  ld,                     // disk
                                                                  ebr_buffer,             // ebr boot sector
                                                                  ebr );                  // ebr DISKSEG *
                                }
                        }

                        free(ebr_buffer);
                }
                else {
                        rc = ENOMEM;
                }
        }
        else {
                rc = 0;
        }


        // create segment objects for embedded partitions ...
        for (i=0; (i<4)&&(rc==0); i++) {

                part = &boot_sector->Partition_Table[i];

                switch ( SYS_IND(part) ) {

                case BSD_PARTITION:
                case NETBSD_PARTITION:
                case OPENBSD_PARTITION:
                        rc = do_bsd_partition_discover(ld, part);
                        break;
                case UNIXWARE_PARTITION:
                        rc = do_unixware_partition_discover(ld, part);
                        break;
                case SOLARIS_X86_PARTITION:
                        rc = do_solaris_x86_partition_discover(ld, part);
                        break;
                default:
                        break;
                }
                if (rc) {
                        LOG_WARNING("Clearing error from embedded partition discovery to complete segment discovery.\n");
                        rc = 0;  // clear error
                }

        }

        // look for guid partitioned disk and warn user
        if ( has_guid_partition_table( boot_sector )==TRUE) {

                MESSAGE(_("Warning: Disk %s seems to have a GUID Partition Table.  "
			  "Making any changes to this disk may damage the GPT partitions."),
                        ld->name );
        }


        // If successful, find all the freespace on the disk and then
        // make sure the ebr metadata segments are chained properly.
        if (rc == 0) {

                rc = find_freespace_on_disk( ld );

                if (rc == 0) {
                        rc = fixup_EBR_Chain( ld );
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *   Called by SEG_Discover() ... once for each item in the input_object list
 *
 *   Walks the partition table of the disk and if it is our partitioning scheme we will
 *   create storage object segments for the partition table records.
 */
int dos_segment_discovery( storage_object_t  *object,
                           list_anchor_t     output_objects,
                           uint              *object_count )
{
        LOGICALDISK                *ld;
        DISK_PRIVATE_DATA          *disk_pdata=NULL;
        SEG_PRIVATE_DATA           *pdata=NULL;
        Master_Boot_Record         *mbr = NULL;
        boolean                     success=FALSE;
        int                         rc=0;
        list_element_t              iter,iter2;
        struct plugin_functions_s  *dft;
        boolean                     acceptable_object = FALSE;
        int                         segcount=0;
        DISKSEG                    *seg;

        LOG_ENTRY();
        LOG_DEBUG("examining object %s\n", object->name );

        ld    = (LOGICALDISK *) object;
        pdata = (SEG_PRIVATE_DATA *) ld->private_data;


        // check for a segment object that disallows further discovery by checking
        // the TOP_SEGMENT bit in the common flags field.
        //
        // we only work on DISK storage objects so filter out other storage objects that
        // may have been in the list.
        //
        // add checks for plugin record and data seg types in order to support
        // msdos partitions on s390.
        //
        if ( ( ld->object_type == SEGMENT ) &&
             ( pdata->cflags & SEG_CFLAG_TOP_SEGMENT )) {
                acceptable_object = FALSE;
        }
        else if ( ld->object_type == DISK ) {
                acceptable_object = TRUE;
        }
        else if ( ( ld->plugin != Seg_My_PluginRecord_Ptr ) &&
                  ( ld->object_type == SEGMENT && ld->data_type == DATA_TYPE ) ) {
                acceptable_object = TRUE;
        }

        if ( acceptable_object == FALSE ) {
                if (EngFncs->insert_thing(output_objects,ld,INSERT_BEFORE,NULL)) {
                        rc = 0;
                }
                else {
                        rc = EPERM;
                }

                LOG_DEBUG("object is not acceptable\n" );
                LOG_EXIT_INT(rc);
                return rc;
        }

        // get disk manager function table
        dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
        if (dft==NULL) {
                MESSAGE(_("Error:  No device manager function table for drive %s\n"), ld->name);
                rc = EINVAL;
        }

        // read MBR off drive
        if (!rc) {
                mbr = (Master_Boot_Record *) malloc(ld->geometry.bytes_per_sector);
                if (mbr) {
                        rc = dft->read( ld, 0, 1, (void *) mbr );
                }
                else {
                        rc = ENOMEM;
                }
        }

        // create disk private data area for this drive
        if (!rc) {
                if ( create_disk_private_data(ld) ) {
                        MESSAGE(_("Error:  Unable to malloc a sector sized buffer for reading sectors off drive %s\n"),
				ld->name );
                        rc = ENOMEM;
                }
                else {
                        disk_pdata = get_disk_private_data(ld);
                        if (disk_pdata==NULL) {
                                rc = ENOMEM;
                        }
                }
        }

        // look for any errors -OR- GPT Protective Master Boot Record (PMBR) and let the
        // gpt segment manager have the disk.
        if ( (rc!=0) || has_guid_partition_table( mbr )==TRUE) {

                LOG_DEBUG("Discovery, Failure ... not my disk or no partitions\n");

                if (EngFncs->insert_thing(output_objects, ld,INSERT_BEFORE,NULL)) {
                        rc = 0;
                }
                else {
                        rc = EPERM;
                }

                // get rid of the disk private data area we built
                delete_disk_private_data(ld);

                // free mbr i/o buffer
                if (mbr) free(mbr);

                LOG_EXIT_INT(rc);
                return rc;
        }


        if ( isa_disk_with_msdos_partitions(ld, mbr ) == TRUE ) {

                // create evms segment storage objects from mbr/ebr chain
                rc = get_disk_segments(ld, mbr );

                if (rc == 0) {

                        segcount = EngFncs->list_count( ld->parent_objects );

                        if (segcount > 0) {

                                LOG_DETAILS( "Discovery, Success ...I found partitions and created %d segments for disk: %s\n",
                                             segcount,
                                             ld->name);

                                DisplayDiskSegmentList(ld);

                                // someone could use another patitioning tool and get us out
                                // of sync with the DM kernel objects. So, we need to resolve
                                // these differences and obtain status on the segment storage
                                // objects ...
                                resolve_partitions_with_device_mapper(ld);

                                // copy storage objects to the output list
                                rc = EngFncs->concatenate_lists( output_objects, ld->parent_objects );

                                // mark segment discovery as successful
                                if (rc==0) {

                                        // return number of segments found
                                        *object_count += segcount;

                                        success = TRUE;
                                }

                        }

                }

        }
        else {   // errors but if it has an msdos signature we will need to do an assign
                 // operation and popup further messages as needed.

                DISKSEG *freespace=NULL;
                DISKSEG *mbrseg=NULL;
                char * choices[] = {_("Yes"), _("No"), NULL};
                int answer = 1;     // Initialize to Revert

                if ( has_msdos_signature(mbr) == TRUE ) {

                        // create freespace on the disk
                        rc = find_freespace_on_disk( ld );
                        if ( rc == 0) {

                                // now get the freespace segment we just created
                                freespace = get_first_freespace_seg_in_list( ld->parent_objects );
                                if ( freespace ) {

                                        // create an MBR storage object for the new disk
                                        rc = create_mbr_For_Disk( ld, "no name", FALSE );
                                        if (rc == 0) {

                                                DisplayDiskSegmentList(ld);

                                                // copy storage objects to the output list
                                                rc = EngFncs->concatenate_lists( output_objects, ld->parent_objects );

                                                // mark segment discovery as successful
                                                if (rc==0) {

                                                        // return number of segments found ... 2 ... mbr seg plus freespace seg
                                                        *object_count += 2;

                                                        QUESTION( &answer, choices,
                                                                  "Errors were found with the partition information on drive %s.\n\n"
                                                                  "Due to the errors, the drive will appear as though all the partitions were removed, with just an mbr "
                                                                  "and freespace segments showing.\n\n"
                                                                  "You can keep this change by answering YES to the following question, but you will later need to commit "
                                                                  "this change when you exit from evms. Committing this change will cause any existing partition information "
                                                                  "to be discarded and an empty partition table created on the drive.\n\n"
                                                                  "If you choose NO, you should exit evms and correct the problem on the drive.\n\n"
                                                                  "Question: Would you like to mark the drive dirty so that the partitions can be discarded?\n",
                                                                  ld->name);

                                                        if (answer != 0) {
                                                                freespace->flags &= ~SOFLAG_DIRTY;
                                                                mbrseg = get_mbr_from_seglist(ld->parent_objects);
                                                                if (mbrseg) mbrseg->flags &= ~SOFLAG_DIRTY;
                                                        }

                                                        success = TRUE;

                                                }

                                        }

                                }

                        }

                }

        }

        if (success == FALSE) {

                LOG_DEBUG("Discovery, Failure ... not my disk or no partitions\n");

                // remove any storage objects we may have created
                LIST_FOR_EACH_SAFE( ld->parent_objects, iter, iter2, seg ) {
                        if (seg->plugin == Seg_My_PluginRecord_Ptr ) {
                                free_disk_segment(seg);
                        }
                        EngFncs->delete_element( iter );
                }

                // place the disk storage object back onto the discovery list
                if (EngFncs->insert_thing(output_objects, ld, INSERT_BEFORE, NULL)) {
                        rc = 0;
                }
                else {
                        rc = EPERM;
                }

                // get rid of the disk private data area we built
                delete_disk_private_data(ld);
        }


        free(mbr);
        LOG_EXIT_INT(0);
        return 0;
}
