/*
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: ntfsfsim.c
 */

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <fcntl.h>
#include <ctype.h>

#include <plugin.h>

#include "ntfsfsim.h"
#include "utils.h"

#define MIN_NTFS_SIZE	((1024 * 1024) >> EVMS_VSECTOR_SIZE_SHIFT)

static plugin_record_t  ntfs_plugrec;
plugin_record_t       * my_plugin_record = &ntfs_plugrec;
engine_functions_t    * EngFncs;
boolean			have_mkntfs = FALSE;
boolean			have_ntfsfix = FALSE;
boolean			have_ntfsinfo = FALSE;
boolean			have_ntfsclone = FALSE;
boolean			have_ntfsresize = FALSE;
char 			utils_version[12];	// Really the mkntfs version


static int ntfs_setup(engine_functions_t *engine_function_table) {

	int rc = 0;

	EngFncs = engine_function_table;

	LOG_ENTRY();

	if (try_run_get_version("mkntfs", utils_version) == 0) {
		have_mkntfs = TRUE;
	}

	if (try_run("ntfsfix") == 0) {
		have_ntfsfix = TRUE;
	}

	if (try_run("ntfsinfo") == 0) {
		have_ntfsinfo = TRUE;
	}

	if (try_run("ntfsclone") == 0) {
		have_ntfsclone = TRUE;
	}

	if (try_run("ntfsresize") == 0) {
		have_ntfsresize = TRUE;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Free all of the private data we have left on volumes.
 */
static void ntfs_cleanup() {
	int rc = 0;
	list_anchor_t global_volumes;
	list_element_t iter;
	logical_volume_t * vol;
	LOG_ENTRY();

	rc = EngFncs->get_volume_list(my_plugin_record, NULL, 0, &global_volumes);
	if (!rc) {
		LIST_FOR_EACH(global_volumes, iter, vol) {
			free_private_data(vol);
		}

		EngFncs->destroy_list(global_volumes);
		global_volumes = NULL;
	}

	have_mkntfs = FALSE;
	have_ntfsfix = FALSE;
	have_ntfsinfo = FALSE;
	have_ntfsclone = FALSE;
	have_ntfsresize = FALSE;

	LOG_EXIT_VOID();
}


/*
 * Does this FSIM manage the file system on this volume?
 * Return 0 for "yes", else a reason code.
 */
static int ntfs_probe(logical_volume_t * volume) {

	int  rc = 0;

	LOG_ENTRY();

	rc = has_ntfs_boot_sector(volume);

	if (rc == 0) {
		volume->private_data = EngFncs->engine_alloc(sizeof(private_data_t));

		if (volume->private_data != NULL) {
			fill_private_data(volume);

		} else {
			rc = ENOMEM;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Can mkfs this volume?
 */
static int ntfs_can_mkfs(logical_volume_t * volume) {

	LOG_ENTRY();

	if (!have_mkntfs) {
		LOG_DEBUG("The mkntfs utility is not installed.\n");
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	if (volume->vol_size < MIN_NTFS_SIZE) {
		LOG_DETAILS("Volume %s is too small.  NTFS volumes must be at least 1MB in size.\n",
			    volume->name);
		LOG_EXIT_INT(EINVAL);
		return EBUSY;
	}

	LOG_EXIT_INT(0);
	return 0;
}


/*
 * Can unmkfs this volume?
 */
static int ntfs_can_unmkfs(logical_volume_t * volume) {

	private_data_t * pd = (private_data_t *) volume->private_data;

	LOG_ENTRY();

	if (volume->file_system_manager != my_plugin_record) {
		/* It's not my volume. */
		LOG_DEBUG("Volume %s does not have NTFS on it.\n", volume->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* If mounted, can't unmkfs. */
	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		LOG_DEBUG("Volume %s is mounted.\n", volume->name);
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	/*
	 * If scheduled to be cloned, can't unmkfs.  The clone target
	 * must be "unmkfsed" first.
	 */
	if (pd->flags & PDFLAG_NTFSCLONE_SOURCE) {
		LOG_DEBUG("Volume %s is scheduled to be cloned.\n", volume->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	LOG_EXIT_INT(0);
	return 0;
}


/*
 * Can fsck this volume?
 */
static int ntfs_can_fsck(logical_volume_t * volume) {

	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}


/*
 * Can this volume be expanded?
 */
static int ntfs_can_expand_by(logical_volume_t * volume,
			      sector_count_t   * delta_size) {

	private_data_t * pd = volume->private_data;

	LOG_ENTRY();

	if (!have_ntfsresize) {
		LOG_DETAILS("The ntfsresize utility is not installed.  I can't expand.\n");
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	/*
	 * Can't expand a clone target since it doesn't have a file system
	 * on it yet.
	 */
	if (pd->flags & PDFLAG_NTFSCLONE_TARGET) {
		LOG_DETAILS("Volume %s is the target of a pending clone operation.\n",
			    volume->name);
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	/* Expand must be done offline. */
	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		LOG_DETAILS("Volume %s is mounted.\n", volume->name);
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	if (*delta_size > pd->max_fs_size - pd->fs_size) {
		*delta_size = pd->max_fs_size - pd->fs_size;
	}

	LOG_EXIT_INT(0);
	return 0;
}


/*
 * Can this volume be shrunk?
 */
static int ntfs_can_shrink_by(logical_volume_t * volume,
			      sector_count_t   * delta_size) {

	private_data_t * pd = (private_data_t *) volume->private_data;
	sector_count_t min_fs_size;

	LOG_ENTRY();

	if (!have_ntfsresize) {
		LOG_DETAILS("The ntfsresize utility is not installed.  I can't shrink.\n");
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	/*
	 * Can't shrink a clone target since it doesn't have a file system
	 * on it yet.
	 */
	if (pd->flags & PDFLAG_NTFSCLONE_TARGET) {
		LOG_DETAILS("Volume %s is the target of a pending clone operation.\n",
			    volume->name);
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	/* Shrink must be done offline. */
	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		LOG_DETAILS("Volume %s is mounted.\n", volume->name);
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	min_fs_size = get_min_fs_size(volume);
	
	if (min_fs_size >= volume->vol_size) {
		/* No room to shrink. */
		LOG_DETAILS("Volume %s has no room to shrink.\n", volume->name);
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	if (*delta_size > volume->vol_size - min_fs_size) {
		*delta_size = volume->vol_size - min_fs_size;
	}

	LOG_EXIT_INT(0);
	return 0;
}


/*
 * Get the current size of this volume
 */
static int ntfs_get_fs_size(logical_volume_t * volume,
			    sector_count_t   * size) {

	private_data_t * pd = volume->private_data;

	LOG_ENTRY();

        *size =	pd->fs_size;
	LOG_DEBUG("Size of file system on volume %s is %"PRIu64"\n", volume->name, pd->fs_size);

	LOG_EXIT_INT(0);
	return 0;
}


/*
 * Get the size limits for this volume.
 */
static int ntfs_get_fs_limits(logical_volume_t * volume,
			      sector_count_t   * min_size,
			      sector_count_t   * max_volume_size,
			      sector_count_t   * max_object_size) {
	
	private_data_t * pd = volume->private_data;

	LOG_ENTRY();

	if (pd->flags & PDFLAG_NTFSCLONE_TARGET) {
		/*
		 * The clone target does not have a file system on it yet.
		 * Get its minimum file system size from the clone source.
		 */
		*min_size = get_min_fs_size(pd->clone_source);

	} else {
		*min_size = get_min_fs_size(volume);
	}

	*max_volume_size = pd->max_fs_size;
	*max_object_size = pd->max_fs_size;

	LOG_EXIT_INT(0);
	return 0;
}


/*
 * mkfs has been scheduled.  Do any setup work such as claiming another
 * volume for an external log.
 */
static int ntfs_mkfs_setup(logical_volume_t * volume,
			   option_array_t   * options) {

	int rc = 0;

	LOG_ENTRY();

	volume->private_data = EngFncs->engine_alloc(sizeof(private_data_t));

	if (volume->private_data == NULL) {
		LOG_CRITICAL("Unable to get memory for private data.\n");
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * NAME: set_mkntfs_options
 *
 * FUNCTION: Build options array (argv) for mkntfs
 *
 * PARAMETERS:
 *      options   - options array passed from EVMS engine
 *      argv      - mkfs options array
 *      vol_name  - volume on which program will be executed
 *
 */
static void set_ntfs_mkfs_options(option_array_t   * options,
				  char           * * argv,
				  logical_volume_t * volume) {
	
	int i;
	int bufsize;
	int opt_count;
	char *buf;

	LOG_ENTRY();

	argv[0] = "mkntfs";

	/* 'quiet' and 'Force' options */
	argv[1] = "-qF";

	opt_count = 2;

	for (i = 0; i < options->count; i++) {

		if (!options->option[i].is_number_based) {

			if (!strcmp(options->option[i].name, MKFS_LABEL_NAME)) {
				options->option[i].number = MKFS_LABEL_INDEX;
			} else if (!strcmp(options->option[i].name, MKFS_CLUSTER_SIZE_NAME)) {
				options->option[i].number = MKFS_CLUSTER_SIZE_INDEX;
			} else if (!strcmp(options->option[i].name, MKFS_MFT_ZONE_MULT_NAME)) {
				options->option[i].number = MKFS_MFT_ZONE_MULT_INDEX;
			} else if (!strcmp(options->option[i].name, MKFS_COMPRESS_NAME)) {
				options->option[i].number = MKFS_COMPRESS_INDEX;
			} else if (!strcmp(options->option[i].name, MKFS_QUICK_NAME)) {
				options->option[i].number = MKFS_QUICK_INDEX;
			} else {
				/* Unknown. Ignore. */
				continue;
			}
		}

		switch (options->option[i].number) {
			
			case MKFS_LABEL_INDEX:
				if (options->option[i].value.s != NULL) {
					argv[opt_count++] = "-L";
					argv[opt_count++] = options->option[i].value.s;
				}
				break;

			case MKFS_CLUSTER_SIZE_INDEX:
				/*
				 * Make sure the requested cluster size is not
				 * smaller than the sector size.
				 */
				if (options->option[i].value.ui < volume->object->geometry.bytes_per_sector) {
					options->option[i].value.ui = volume->object->geometry.bytes_per_sector;
				}

				if (options->option[i].value.ui == 512) {
					argv[opt_count++] = "-c";
					argv[opt_count++] = "512";
				} else if (options->option[i].value.ui == 1024) {
					argv[opt_count++] = "-c";
					argv[opt_count++] = "1024";
				} else if (options->option[i].value.ui == 2048) {
					argv[opt_count++] = "-c";
					argv[opt_count++] = "2048";
				} else if (options->option[i].value.ui == 4096) {
					argv[opt_count++] = "-c";
					argv[opt_count++] = "4096";
				}
				break;

			case MKFS_MFT_ZONE_MULT_INDEX:
				if (options->option[i].value.r32 == 25) {
					argv[opt_count++] = "-z";
					argv[opt_count++] = "2";
				} else if (options->option[i].value.r32 == 37.5) {
					argv[opt_count++] = "-z";
					argv[opt_count++] = "3";
				} else if (options->option[i].value.r32 == 50) {
					argv[opt_count++] = "-z";
					argv[opt_count++] = "4";
				}
				break;

			case MKFS_COMPRESS_INDEX:
				if (options->option[i].value.b == TRUE) {
					argv[opt_count++] = "-C";
				}
				break;

			case MKFS_QUICK_INDEX:
				if (options->option[i].value.b == TRUE) {
					argv[opt_count++] = "-Q";
				}
				break;

			default:
				break;
		}
	}

	argv[opt_count++] = volume->dev_node;
	argv[opt_count] = NULL;

	bufsize = 0;
	for (i = 0; argv[i]; i++) {
		bufsize += strlen(argv[i]) + 1;
	}
	buf = EngFncs->engine_alloc(bufsize + 1);
	if (buf != NULL) {
		buf[0] = 0;
		for (i=0; argv[i]; i++) {
			strcat(buf, argv[i]);
			strcat(buf, " ");
		}

		LOG_DEBUG("mkntfs command: %s\n", buf);
		EngFncs->engine_free(buf);
	}

	LOG_EXIT_VOID();
	return;
}


/*
 * Put the file system on the volume.
 */
static int ntfs_mkfs(logical_volume_t * volume, option_array_t * options) {
	
	int    rc = 0;
	char * argv[MKFS_OPTIONS_STRING_COUNT + 3];
	pid_t  pidm;
	int    status;
	int    fds[2];
	char * buffer = NULL;
	int    bytes_read = 0;

	LOG_ENTRY();

	if (!have_mkntfs) {
		MESSAGE(_("The mkntfs utility is not installed on this machine.  "
			  "The NTFS FSIM uses mkntfs to make the NTFS file system on the volume.  "
			  "Get the latest version of the NTFS utilities from http://sourceforge.net/projects/linux-ntfs/\n"));
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN);
	if (buffer == NULL) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	status = pipe(fds);
	if (status < 0) {
		EngFncs->engine_free(buffer);
		LOG_EXIT_INT(errno);
		return errno;
	}

	set_ntfs_mkfs_options(options, argv, volume);

	fcntl(fds[0], F_SETFL, fcntl(fds[0], F_GETFL, 0) | O_NONBLOCK);
	fcntl(fds[1], F_SETFL, fcntl(fds[1], F_GETFL, 0) | O_NONBLOCK);

	pidm = EngFncs->fork_and_execvp(volume, argv, NULL, fds, fds);
	if (pidm != -1) {

		waitpid(pidm, &status, 0);

		if (WIFEXITED(status)) {
			do {
				bytes_read = read(fds[0], buffer, MAX_USER_MESSAGE_LEN - 1);
				if (bytes_read > 0) {
					LOG_DETAILS("%s output: \n%s", argv[0], buffer);
				}
			} while (bytes_read > 0);

			rc = WEXITSTATUS(status);
			if (rc == 0) {
				LOG_DETAILS("%s completed with exit code %d \n", argv[0], rc);
			} else {
				LOG_WARNING("%s completed with exit code %d \n", argv[0], rc);
			}

		} else {
			/*
			 * The process didn't exit. It must have been
			 * interrupted by a signal.
			 */
			rc = EINTR;
		}

	} else {
		LOG_SERIOUS("Failed to fork and exec %s.  Error code is %d: %s\n",
			    argv[0], rc, EngFncs->strerror(rc));
		rc = errno;
	}

	if (rc == 0) {
		fill_private_data(volume);
	}

	EngFncs->engine_free(buffer);
	
	close(fds[0]);
	close(fds[1]);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Expand the volume to new_size.  If the volume is not expanded exactly to
 * new_size, set new_size to the new_size of the volume.
 */
static int ntfs_expand(logical_volume_t * volume,
		       sector_count_t * new_size) {

	int rc;

	LOG_ENTRY();

	/* Expand must be done offline. */
	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		LOG_DETAILS("Volume %s is mounted.\n", volume->name);
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	rc = resize_ntfs(volume, new_size);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Shrink the volume to new_size.  If the volume is not shrunk exactly to
 * new_size, set new_size to the new_size of the volume.
 */
static int ntfs_shrink(logical_volume_t * volume,
		       sector_count_t     requested_size,
		       sector_count_t   * new_size) {

	int rc;

	LOG_ENTRY();

	/* Shrink must be done offline. */
	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		LOG_DETAILS("Volume %s is mounted.\n", volume->name);
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	*new_size = requested_size;
	rc = resize_ntfs(volume, new_size);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Forget about this volume.  Don't remove the file system.  Just clean up any
 * data structures you may have associated with it.
 */
static int ntfs_discard(logical_volume_t * volume) {
	
	LOG_ENTRY();

	if (volume->private_data) {
		free_private_data(volume);
	}

	LOG_EXIT_INT(0);
	return 0;
}


/*
 * unmkfs has been scheduled.  Do any setup work such as releasing another
 * volume that was used for an external log.
 */
static int ntfs_unmkfs_setup(logical_volume_t * volume) {

	private_data_t * pd = (private_data_t *) volume->private_data;

	LOG_ENTRY();

	if (pd->flags & PDFLAG_NTFSCLONE_TARGET) {
		/*
		 * unmkfs on a clone target is a cancel of the clone
		 * operation.
		 */
		private_data_t * source_pd = (private_data_t *) pd->clone_source->private_data;

		source_pd->clone_target = NULL;
		source_pd->flags &= ~PDFLAG_NTFSCLONE_SOURCE;

		free_private_data(volume);

		EngFncs->unassign_fsim_from_volume(volume);

		MESSAGE(_("The pending ntfsclone operation has been canceled.\n"));
	}

	LOG_EXIT_INT(0);
	return 0;
}


/*
 * Remove the file system from the volume.
 */
static int ntfs_unmkfs(logical_volume_t * volume) {

	int rc = 0;
	private_data_t * pd = (private_data_t *) volume->private_data;

	LOG_ENTRY();

	/*
	 * We should not be called to unmkfs a clone target.  But just in case,
	 * cancel the clone.
	 */
	if (pd->flags & PDFLAG_NTFSCLONE_TARGET) {
		private_data_t * source_pd = (private_data_t *) pd->clone_source->private_data;

		source_pd->clone_target = NULL;
		source_pd->flags &= ~PDFLAG_NTFSCLONE_SOURCE;

		free_private_data(volume);

	} else {
		rc = clear_ntfs_boot_sectors(volume);

		if (rc == 0) {
			free_private_data(volume);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Do any operations you wanted to do at commit time, based on the phase of the
 * commit.
 */
int ntfs_commit_changes(logical_volume_t * volume, commit_phase_t phase) {

	int rc = 0;
	private_data_t * pd;

	LOG_ENTRY();

	switch (phase) {
		case POST_ACTIVATE:
			pd = (private_data_t *) volume->private_data;

			if (pd->flags & PDFLAG_RUN_NTFSFIX) {
				rc = run_ntfsfix(volume);

				/* Don't try to run it again, even if it failed. */
				pd->flags &= ~PDFLAG_RUN_NTFSFIX;
			}

			if (pd->flags & PDFLAG_NTFSCLONE_SOURCE) {
				rc = run_ntfsclone(volume,
						   pd->clone_target,
						   pd->flags & PDFLAG_FORCE_NTFSCLONE);

				if (rc == 0) {
					pd->flags &= ~PDFLAG_NTFSCLONE_SOURCE;

					/*
					 * Reprobe the clone target for the file
					 * system.  Yes, we know that is has
					 * NTFS on it, but we need to have the
					 * Engine know it and set up its records
					 * for the original FSIM on the volume.
					 * And a reprobe causes us to create new
					 * private data for the volume based on
					 * the actual contents of the volume.
					 */
					free_private_data(pd->clone_target);
					pd->clone_target->flags |= VOLFLAG_PROBE_FS;

					pd->clone_target = NULL;
				}
			}

			if (!(pd->flags & (PDFLAG_RUN_NTFSFIX | PDFLAG_NTFSCLONE_SOURCE))) {
				volume->flags &= ~VOLFLAG_DIRTY;
			}
			break;

		default:
			break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Return the total number of supported options for the specified task.
 */
static int ntfs_get_option_count(task_context_t * context) {
	
	int count = 0;

	LOG_ENTRY();

	switch (context->action) {
		case EVMS_Task_mkfs:
			count = MKFS_OPTIONS_COUNT;
			break;
		case EVMS_Task_ntfsclone:
			count = NTFSCLONE_OPTIONS_COUNT;
			break;
		default:
			count = -1;
			break;
	}

	LOG_EXIT_INT(count);
	return count;
}

/*
 * Initialize mkfs task acceptable objects by enumerating volumes, finding
 * those that have no FSIM claiming them and are of the proper size and
 * adding them to the acceptable objects list.
 */
static int init_mkfs_acceptable_objects(task_context_t * context) {
	
	int rc;
	list_anchor_t global_volumes;
	list_element_t vol_list_iter;
	logical_volume_t * volume;

	LOG_ENTRY();

	rc = EngFncs->get_volume_list(NULL, NULL, 0, &global_volumes);
	if (!rc) {
		LIST_FOR_EACH(global_volumes, vol_list_iter, volume) {
			if ((volume->file_system_manager == NULL) &&
			    (!EngFncs->is_mounted(volume->name, NULL)) &&
			    (volume->vol_size >= MIN_NTFS_SIZE)) {
				EngFncs->insert_thing(context->acceptable_objects,
						      volume,
						      INSERT_BEFORE,
						      NULL);
			}
		}
		EngFncs->destroy_list(global_volumes);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static value_collection_t make_cluster_size_constraints(void) {

	value_collection_t coll;

	/* No entry/exit logs. */

	coll.list = EngFncs->engine_alloc(sizeof(value_list_t) + sizeof(value_t) * 4);

	if (coll.list != NULL) {
		coll.list->count = 4;
		coll.list->value[0].ui = 512;
		coll.list->value[1].ui = 1024;
		coll.list->value[2].ui = 2048;
		coll.list->value[3].ui = 4096;
	}

	return coll;
}


static value_collection_t make_MFT_zone_mult_constraints(void) {

	value_collection_t coll;

	/* No entry/exit logs. */

	coll.list = EngFncs->engine_alloc(sizeof(value_list_t) + sizeof(value_t) * 4);

	if (coll.list != NULL) {
		coll.list->count = 4;
		coll.list->value[0].r32 = 12.5;
		coll.list->value[1].r32 = 25;
		coll.list->value[2].r32 = 32.5;
		coll.list->value[3].r32 = 50;
	}

	return coll;
}


/*
 * Initialize mkfs task acceptable objects by enumerating volumes, finding
 * those that have no FSIM claiming them and are of the proper size and
 * adding them to the acceptable objects list.
 */
static int init_ntfsclone_target_list(logical_volume_t * source,
				      value_list_t * * list) {
	
	int rc;
	list_anchor_t volume_list;
	list_element_t iter1;
	list_element_t iter2;
	logical_volume_t * vol;
	uint num_targets;
	value_list_t * targets;
	int i;

	LOG_ENTRY();

	rc = EngFncs->get_volume_list(NULL, NULL, 0, &volume_list);
	if (rc == 0) {
		LIST_FOR_EACH_SAFE(volume_list, iter1, iter2, vol) {
			if (!is_acceptable_clone_target(source, vol) == 0) {
				EngFncs->delete_element(iter1);
			}
		}

		num_targets = EngFncs->list_count(volume_list);

		targets = EngFncs->engine_alloc(sizeof(value_list_t) +
						sizeof(value_t) * num_targets);
		if (targets != NULL) {
			i = 0;

			LIST_FOR_EACH(volume_list,iter1,vol) {
				targets->value[i].s = EngFncs->engine_strdup(vol->name);
				i++;
			}
			targets->count = i;

			*list = targets;

		} else {
			LOG_CRITICAL("Unable to get memory for a value list.\n");
			rc = ENOMEM;
		}

		EngFncs->destroy_list(volume_list);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Fill in the initial list of acceptable objects.  Fill in the minimum and
 * maximum number of objects that must/can be selected.  Set up all initial
 * values in the option_descriptors in the context record for the given
 * task.  Some fields in the option_descriptor may be dependent on a
 * selected object.  Leave such fields blank for now, and fill in during the
 * set_objects call.
 */
static int ntfs_init_task(task_context_t * context) {

	int rc = 0;
	option_descriptor_t * opt;

	LOG_ENTRY();

	/* Parameter check */
	if (!context) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	switch (context->action) {
		
		case EVMS_Task_mkfs:
			rc = init_mkfs_acceptable_objects(context);
			if (rc) {
				LOG_EXIT_INT(rc);
				return rc;
			}

			context->option_descriptors->count = MKFS_OPTIONS_COUNT;

			/* Volume label option */
			opt = &context->option_descriptors->option[MKFS_LABEL_INDEX];
			opt->name = EngFncs->engine_strdup(MKFS_LABEL_NAME);
			opt->title = EngFncs->engine_strdup(_("Volume label"));
			opt->tip = EngFncs->engine_strdup(_("Set the label for the volume."));
			opt->help = NULL;
			opt->type = EVMS_Type_String;
			opt->unit = EVMS_Unit_None;
			opt->min_len = 1;
			opt->max_len = MAX_LABEL_LEN;
			opt->flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
			opt->constraint_type = EVMS_Collection_None;
			opt->value.s = EngFncs->engine_alloc(MAX_LABEL_LEN + 1);

			/* Cluster size option */
			opt = &context->option_descriptors->option[MKFS_CLUSTER_SIZE_INDEX];
			opt->name = EngFncs->engine_strdup(MKFS_CLUSTER_SIZE_NAME);
			opt->title = EngFncs->engine_strdup(_("Cluster size"));
			opt->tip = EngFncs->engine_strdup(_("Set the cluster size."));
			opt->help = NULL;
			opt->type = EVMS_Type_Unsigned_Int;
			opt->unit = EVMS_Unit_Bytes;
			opt->flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
			opt->constraint_type = EVMS_Collection_List;
			opt->constraint = make_cluster_size_constraints();
			opt->value.ui = 512;

			/* MFT zone multiplier option */
			opt = &context->option_descriptors->option[MKFS_MFT_ZONE_MULT_INDEX];
			opt->name = EngFncs->engine_strdup(MKFS_MFT_ZONE_MULT_NAME);
			opt->title = EngFncs->engine_strdup(_("MFT zone multiplier"));
			opt->tip = EngFncs->engine_strdup(_("Set the MFT zone multiplier."));
			opt->help = NULL;
			opt->type = EVMS_Type_Real32;
			opt->unit = EVMS_Unit_Percent;
			opt->flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
			opt->constraint_type = EVMS_Collection_List;
			opt->constraint = make_MFT_zone_mult_constraints();
			opt->value.r32 = 0.125;

			/* Compression option */
			opt = &context->option_descriptors->option[MKFS_COMPRESS_INDEX];
			opt->name = EngFncs->engine_strdup(MKFS_COMPRESS_NAME);
			opt->title = EngFncs->engine_strdup(_("Compression"));
			opt->tip = EngFncs->engine_strdup(_("Enable compression on the volume."));
			opt->help = NULL;
			opt->type = EVMS_Type_Boolean;
			opt->unit = EVMS_Unit_None;
			opt->flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
			opt->constraint_type = EVMS_Collection_None;
			opt->value.b = FALSE;

			/* Quick option */
			opt = &context->option_descriptors->option[MKFS_QUICK_INDEX];
			opt->name = EngFncs->engine_strdup(MKFS_QUICK_NAME);
			opt->title = EngFncs->engine_strdup(_("Quick mkfs"));
			opt->tip = EngFncs->engine_strdup(_("Skip both zeroing of the volume and bad sector checking."));
			opt->help = NULL;
			opt->type = EVMS_Type_Boolean;
			opt->unit = EVMS_Unit_None;
			opt->flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
			opt->constraint_type = EVMS_Collection_None;
			opt->value.b = TRUE;

			context->min_selected_objects = 1;
			context->max_selected_objects = 1;

			break;

		case EVMS_Task_ntfsfix:
			break;

		case EVMS_Task_ntfsclone:
			context->option_descriptors->count = NTFSCLONE_OPTIONS_COUNT;

			/* Target option */
			opt = &context->option_descriptors->option[NTFSCLONE_TARGET_INDEX];
			opt->name = EngFncs->engine_strdup(NTFSCLONE_TARGET_NAME);
			opt->title = EngFncs->engine_strdup(_("Target volume"));
			opt->tip = EngFncs->engine_strdup(_("The volume onto which the file system should be cloned"));
			opt->help = NULL;
			opt->type = EVMS_Type_String;
			opt->unit = EVMS_Unit_None;
			opt->min_len = 1;
			opt->max_len = EVMS_NAME_SIZE;
			opt->flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
			opt->constraint_type = EVMS_Collection_List;
			rc = init_ntfsclone_target_list(context->volume, &opt->constraint.list);
			if (rc != 0) {
				break;
			}

			opt->value.s = EngFncs->engine_alloc(EVMS_NAME_SIZE + 1);
			if (opt->value.s == NULL) {
				LOG_CRITICAL("Unable to get memory for a string value in an option descriptor.\n");
				rc = ENOMEM;
				break;
			}

			/* Force option */
			opt = &context->option_descriptors->option[NTFSCLONE_FORCE_INDEX];
			opt->name = EngFncs->engine_strdup(NTFSCLONE_FORCE_NAME);
			opt->title = EngFncs->engine_strdup(_("Force"));
			opt->tip = EngFncs->engine_strdup(_("Force ntfsclone to run, even if the file system is dirty."));
			opt->help = NULL;
			opt->type = EVMS_Type_Boolean;
			opt->unit = EVMS_Unit_None;
			opt->flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
			opt->constraint_type = EVMS_Collection_None;
			opt->value.b = TRUE;

			context->min_selected_objects = 0;
			context->max_selected_objects = 0;
			break;

		default:
			rc = EINVAL;
			break;
	}

	LOG_EXIT_INT(rc);
	return rc;

}


/*
 * Examine the specified value, and determine if it is valid for the task
 * and option_descriptor index. If it is acceptable, set that value in the
 * appropriate entry in the option_descriptor. The value may be adjusted
 * if necessary/allowed. If so, set the effect return value accordingly.
 */
static int ntfs_set_option(task_context_t * context,
			   u_int32_t        index,
			   value_t        * value,
			   task_effect_t  * effect) {
	int  rc= 0;
	logical_volume_t * clone_target;

	LOG_ENTRY();

	/* Parameter check */
	if (!context || !value || !effect) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	switch (context->action) {
		
		case EVMS_Task_mkfs:
			switch (index) {
				case MKFS_LABEL_INDEX:
					strncpy(context->option_descriptors->option[index].value.s, value->s, MAX_LABEL_LEN);
					if (strlen(value->s) > MAX_LABEL_LEN) {
						MESSAGE(_("Volume label is truncated to \"%s\".\n"),
							context->option_descriptors->option[index].value.s);
					}
					break;

				case MKFS_CLUSTER_SIZE_INDEX:
					context->option_descriptors->option[index].value.ui = value->ui;
					break;

				case MKFS_MFT_ZONE_MULT_INDEX:
					context->option_descriptors->option[index].value.r32 = value->r32;
					break;

				case MKFS_COMPRESS_INDEX:
				case MKFS_QUICK_INDEX:
					context->option_descriptors->option[index].value.b = value->b;
					break;

				default:
					break;
			}
			break;

		case EVMS_Task_ntfsclone:
			switch (index) {
				case NTFSCLONE_TARGET_INDEX:
					clone_target = find_volume(value->s);
					if (clone_target != NULL) {
						rc = is_acceptable_clone_target(context->volume, clone_target);
						if (rc == 0) {
							strcpy(context->option_descriptors->option[index].value.s, value->s);
						}

					} else {
						LOG_ERROR ("%s is not the name of a volume.\n",
							   value->s);
						rc = EINVAL;
					}
					break;

				case NTFSCLONE_FORCE_INDEX:
					context->option_descriptors->option[index].value.b = value->b;
					break;

				default:
					break;
			}
			break;

		default:
			LOG_ERROR("I don't know how to set an option for action code %d (%#x).\n",
				  context->action, context->action);
			rc = EINVAL;
			break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Validate the volumes in the selected_objects list in the task context.
 * Remove from the selected objects lists any volumes which are not
 * acceptable.  For unacceptable volumes, create a declined_handle_t
 * structure with the reason why it is not acceptable, and add it to the
 * declined_volumes list.  Modify the acceptable_objects list in the task
 * context as necessary based on the selected objects and the current
 * settings of the options.  Modify any option settings as necessary based
 * on the selected objects.  Return the appropriate task_effect_t settings
 * if the object list(s), minimum or maximum objects selected, or option
 * settings have changed.
 */
static int ntfs_set_volumes(task_context_t * context,
			    list_anchor_t    declined_volumes,	 /* of type declined_handle_t */
			    task_effect_t  * effect) {
	int rc;
	list_element_t iter1;
	list_element_t iter2;
	logical_volume_t * vol;

	LOG_ENTRY();

	switch (context->action) {
		case EVMS_Task_mkfs:

			/* get the selected volume */
			LIST_FOR_EACH_SAFE(context->selected_objects, iter1, iter2, vol){
				if (EngFncs->is_mounted(vol->dev_node, NULL)) {
					/* If mounted, can't mkntfs. */
					LOG_ERROR("Volume %s is mounted on %s.\n",
						  vol->name, vol->mount_point);
					rc = EBUSY;
				} else {
					if (vol->vol_size < MIN_NTFS_SIZE) {
						LOG_ERROR("Volume %s is too small.  "
							  "NTFS volumes must be at least 1MB in size.\n",
							  vol->name);
						rc = EINVAL;

					} else {
						/* This one looks fine. */
						rc = 0;
					}
				}

				if (rc != 0) {
					declined_object_t * dec_vol;

					EngFncs->delete_element(iter1);

					dec_vol = EngFncs->engine_alloc(sizeof(declined_object_t));
					if (dec_vol != NULL) {
						dec_vol->object = vol;
						dec_vol->reason = rc;

						EngFncs->insert_thing(declined_volumes,
								      dec_vol,
								      INSERT_AFTER,
								      NULL);
					} else {
						LOG_CRITICAL("Unable to get memory for a declined_object_t.\n");

						/*
						 * Put any already declined objects
						 * back on the selected_objects list.
						 */

						/*
						 * Yeah, I know I'm using the iterators
						 * from the loop we're in.  We're bailing
						 * out here, so it doesn't matter.
						 */
						LIST_FOR_EACH_SAFE(declined_volumes, iter1, iter2, dec_vol) {
							EngFncs->delete_element(iter1);
							EngFncs->insert_thing(context->selected_objects,
									      dec_vol->object,
									      INSERT_AFTER,NULL);
							EngFncs->engine_free(dec_vol);
						}

						LOG_EXIT_INT(ENOMEM);
						return ENOMEM;
					}
				}
			}
			break;

		default:
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
	}

	LOG_EXIT_INT(0);
	return 0;
}


/*
 * Return any additional information that you wish to provide about the
 * volume.  The Engine provides an external API to get the information
 * stored in the logical_volume_t.  This call is to get any other
 * information about the volume that is not specified in the
 * logical_volume_t.  Any piece of information you wish to provide must be
 * in an extended_info_t structure.  Use the Engine's engine_alloc() to
 * allocate the memory for the extended_info_t.  Also use engine_alloc() to
 * allocate any strings that may go into the extended_info_t.  Then use
 * engine_alloc() to allocate an extended_info_array_t with enough entries
 * for the number of extended_info_t structures you are returning.  Fill
 * in the array and return it in *info.
 * If you have extended_info_t descriptors that themselves may have more
 * extended information, set the EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag
 * in the extended_info_t flags field.  If the caller wants more information
 * about a particular extended_info_t item, this API will be called with a
 * pointer to the storage_object_t and with a pointer to the name of the
 * extended_info_t item.  In that case, return an extended_info_array_t with
 * further information about the item.  Each of those items may have the
 * EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag set if you desire.  It is your
 * responsibility to give the items unique names so that you know which item
 * the caller is asking additional information for.  If info_name is NULL,
 * the caller just wants top level information about the object.
 */
static int ntfs_get_volume_info(logical_volume_t        * volume,
				char                    * info_name,
				extended_info_array_t * * info) {
	int rc = 0;
	private_data_t * pd = volume->private_data;
	extended_info_array_t * Info;
	int i;

	LOG_ENTRY();

	if (pd == NULL) {
		LOG_SERIOUS("Oops!  Volume %s does not have private data.\n", volume->name);
		LOG_EXIT_INT(ENOENT);
		return ENOENT;
	}

	if (info_name != NULL) {
		LOG_ERROR("Volume %s has no extra information named \"%s\".\n",
			  volume->name, info_name);
		rc = EINVAL;
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	Info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + (4 * sizeof(extended_info_t)));

	if (Info == NULL) {
		LOG_CRITICAL("Unable to allocate memory for the extended_info_array_t buffer.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	i = 0;

	if (pd->ntfs_vol_name != NULL) {
		Info->info[i].name = EngFncs->engine_strdup("NTFS Volume Name");
		Info->info[i].title = EngFncs->engine_strdup(_("NTFS Volume Name"));
		Info->info[i].desc = EngFncs->engine_strdup(_("The NTFS name of the volume"));
		Info->info[i].type              = EVMS_Type_String;
		Info->info[i].unit              = EVMS_Unit_None;
		Info->info[i].value.s = EngFncs->engine_strdup(pd->ntfs_vol_name);
		Info->info[i].collection_type   = EVMS_Collection_None;
		i++;
	}

	if (pd->ntfs_version != NULL) {
		Info->info[i].name = EngFncs->engine_strdup("NTFS Version");
		Info->info[i].title = EngFncs->engine_strdup(_("NTFS Version"));
		Info->info[i].desc = EngFncs->engine_strdup(_("The NTFS version that created this file system"));
		Info->info[i].type              = EVMS_Type_String;
		Info->info[i].unit              = EVMS_Unit_None;
		Info->info[i].value.s = EngFncs->engine_strdup(pd->ntfs_version);
		Info->info[i].collection_type   = EVMS_Collection_None;
		i++;
	}

	if (pd->ntfs_cluster_size != 0) {
		Info->info[i].name = EngFncs->engine_strdup("Cluster Size");
		Info->info[i].title = EngFncs->engine_strdup(_("Cluster Size"));
		Info->info[i].desc = EngFncs->engine_strdup(_("Size of a cluster."));
		Info->info[i].type              = EVMS_Type_Unsigned_Int32;
		Info->info[i].unit              = EVMS_Unit_None;
		Info->info[i].value.ui32        = pd->ntfs_cluster_size;
		Info->info[i].collection_type   = EVMS_Collection_None;
		i++;
	}

	if (pd->ntfs_nr_clusters != 0) {
		Info->info[i].name = EngFncs->engine_strdup("Number of Clusters");
		Info->info[i].title = EngFncs->engine_strdup(_("Number of Clusters"));
		Info->info[i].desc = EngFncs->engine_strdup(_("The total number of clusters in the file system"));
		Info->info[i].type              = EVMS_Type_Int64;
		Info->info[i].unit              = EVMS_Unit_None;
		Info->info[i].value.i64         = pd->ntfs_nr_clusters;
		Info->info[i].collection_type   = EVMS_Collection_None;
		i++;
	}

	Info->count = i;
	*info = Info;

	rc = 0;

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Return Plug-in specific information.
 */
static int ntfs_get_plugin_info(char * descriptor_name, extended_info_array_t * * info) {
	
	int rc = EINVAL;
	extended_info_array_t * Info;
	char version_string[64];
	char required_engine_api_version_string[64];
	char required_fsim_api_version_string[64];

	LOG_ENTRY();

	if (info) {

		if (descriptor_name == NULL) {
			*info = NULL;	  // init to no info returned

			Info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + (7 * sizeof(extended_info_t)));
			if (Info) {

				Info->count = 7;

				sprintf(version_string, "%d.%d.%d",
					MAJOR_VERSION,
					MINOR_VERSION,
					PATCH_LEVEL);

				sprintf(required_engine_api_version_string, "%d.%d.%d",
					my_plugin_record->required_engine_api_version.major,
					my_plugin_record->required_engine_api_version.minor,
					my_plugin_record->required_engine_api_version.patchlevel);

				sprintf(required_fsim_api_version_string, "%d.%d.%d",
					my_plugin_record->required_plugin_api_version.fsim.major,
					my_plugin_record->required_plugin_api_version.fsim.minor,
					my_plugin_record->required_plugin_api_version.fsim.patchlevel);

				Info->info[0].name = EngFncs->engine_strdup("Short Name");
				Info->info[0].title = EngFncs->engine_strdup(_("Short Name"));
				Info->info[0].desc = EngFncs->engine_strdup(_("A short name given to this plug-in"));
				Info->info[0].type               = EVMS_Type_String;
				Info->info[0].unit               = EVMS_Unit_None;
				Info->info[0].value.s = EngFncs->engine_strdup(my_plugin_record->short_name);
				Info->info[0].collection_type    = EVMS_Collection_None;
				memset(&Info->info[0].group, 0, sizeof(group_info_t));

				Info->info[1].name = EngFncs->engine_strdup("Long Name");
				Info->info[1].title = EngFncs->engine_strdup(_("Long Name"));
				Info->info[1].desc = EngFncs->engine_strdup(_("A longer, more descriptive name for this plug-in"));
				Info->info[1].type               = EVMS_Type_String;
				Info->info[1].unit               = EVMS_Unit_None;
				Info->info[1].value.s = EngFncs->engine_strdup(my_plugin_record->long_name);
				Info->info[1].collection_type    = EVMS_Collection_None;
				memset(&Info->info[1].group, 0, sizeof(group_info_t));

				Info->info[2].name = EngFncs->engine_strdup("Type");
				Info->info[2].title = EngFncs->engine_strdup(_("Plug-in Type"));
				Info->info[2].desc = EngFncs->engine_strdup(_("There are various types of plug-ins, each responsible for some kind of storage object or logical volume."));
				Info->info[2].type               = EVMS_Type_String;
				Info->info[2].unit               = EVMS_Unit_None;
				Info->info[2].value.s = EngFncs->engine_strdup(_("File System Interface Module"));
				Info->info[2].collection_type    = EVMS_Collection_None;
				memset(&Info->info[2].group, 0, sizeof(group_info_t));

				Info->info[3].name = EngFncs->engine_strdup("Version");
				Info->info[3].title = EngFncs->engine_strdup(_("Plug-in Version"));
				Info->info[3].desc = EngFncs->engine_strdup(_("This is the version number of the plug-in."));
				Info->info[3].type               = EVMS_Type_String;
				Info->info[3].unit               = EVMS_Unit_None;
				Info->info[3].value.s = EngFncs->engine_strdup(version_string);
				Info->info[3].collection_type    = EVMS_Collection_None;
				memset(&Info->info[3].group, 0, sizeof(group_info_t));

				Info->info[4].name = EngFncs->engine_strdup("Required Engine Services Version");
				Info->info[4].title = EngFncs->engine_strdup(_("Required Engine Services Version"));
				Info->info[4].desc = EngFncs->engine_strdup(_("This is the version of the Engine services that this plug-in requires.  "
									      "It will not run on older versions of the Engine services."));
				Info->info[4].type               = EVMS_Type_String;
				Info->info[4].unit               = EVMS_Unit_None;
				Info->info[4].value.s = EngFncs->engine_strdup(required_engine_api_version_string);
				Info->info[4].collection_type    = EVMS_Collection_None;
				memset(&Info->info[4].group, 0, sizeof(group_info_t));

				Info->info[5].name = EngFncs->engine_strdup("Required Engine FSIM API Version");
				Info->info[5].title = EngFncs->engine_strdup(_("Required Engine FSIM API Version"));
				Info->info[5].desc = EngFncs->engine_strdup(_("This is the version of the Engine FSIM API that this plug-in requires.  "
									      "It will not run on older versions of the Engine FSIM API."));
				Info->info[5].type               = EVMS_Type_String;
				Info->info[5].unit               = EVMS_Unit_None;
				Info->info[5].value.s = EngFncs->engine_strdup(required_fsim_api_version_string);
				Info->info[5].collection_type    = EVMS_Collection_None;
				memset(&Info->info[5].group, 0, sizeof(group_info_t));

				Info->info[6].name = EngFncs->engine_strdup("NTFS Utilities Version");
				Info->info[6].title = EngFncs->engine_strdup(_("NTFS Utilities Version"));
				Info->info[6].desc = EngFncs->engine_strdup(_("This is the version of the NTFS Utilities that are installed on this machine."));
				Info->info[6].type               = EVMS_Type_String;
				Info->info[6].unit               = EVMS_Unit_None;
				if (have_mkntfs) {
					Info->info[6].value.s = EngFncs->engine_strdup(utils_version);
				} else {
					Info->info[6].value.s = EngFncs->engine_strdup("Not installed");
				}
				Info->info[6].collection_type    = EVMS_Collection_None;
				memset(&Info->info[6].group, 0, sizeof(group_info_t));

				*info = Info;

				rc = 0;
			} else {
				rc = ENOMEM;
			}

		} else {
			/* There is no more information on any of the descriptors. */
			rc = EINVAL;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Return an array of plug-in functions that you support for this volume.
 */
int ntfs_get_plugin_functions(logical_volume_t        * volume,
			      function_info_array_t * * actions) {

	private_data_t * pd;
	function_info_array_t * fia;
	int i;

	LOG_ENTRY();

	/* "volume" is NULL when the Engine is asking for functions that
	 * apply to the plug-in rather than to a particular volume.
	 */
	if (volume == NULL) {
		LOG_DEBUG("There are no functions targeted at this plug-in.\n");
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (volume->file_system_manager != my_plugin_record) {
		/* It's not my volume. */
		LOG_DEBUG("Volume %s does not have NTFS on it.\n", volume->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (!(volume->flags & (VOLFLAG_ACTIVE | VOLFLAG_NEEDS_ACTIVATE))) {
		LOG_DEBUG("Volume %s is not active.\n",
			  volume->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (volume->flags & VOLFLAG_NEEDS_DEACTIVATE) {
		LOG_DEBUG("Volume %s will be deactivated.\n", volume->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (volume->flags & (VOLFLAG_NEW | VOLFLAG_MKFS)) {
		/* mkntfs has yet to be run on this volume. */
		LOG_DEBUG("Volume %s does not yet have NTFS on it.  mkfs is scheduled to be run.\n",
			  volume->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	pd = (private_data_t *) volume->private_data;

	fia = EngFncs->engine_alloc(sizeof(function_info_array_t) + sizeof(function_info_t) * 2);
	if (fia == NULL) {
		LOG_CRITICAL("Unable to get memory for a function_info_array_t.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	i = 0;

	if (have_ntfsfix) {
		if (!(pd->flags & PDFLAG_RUN_NTFSFIX)) {
			fia->info[i].function = EVMS_Task_ntfsfix;
			fia->info[i].name = EngFncs->engine_strdup("ntfsfix");
			fia->info[i].title = EngFncs->engine_strdup("ntfsfix");
			fia->info[i].verb = EngFncs->engine_strdup("ntfsfix");
			fia->info[i].help = EngFncs->engine_strdup(_("Run the ntfsfix utility to prepare the file system for booting under Windows."));

			if (EngFncs->is_mounted(volume->name, NULL)) {
				LOG_DEBUG("Volume %s is mounted.  It must be unmounted in order to run ntfsfix.\n",
					  volume->name);
				fia->info[i].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
			}

			i++;

		} else {
			LOG_DETAILS("ntfsfix is already scheduled to be run on this volume.\n");
		}

	} else {
		LOG_DETAILS("The ntfsfix utility in not installed on this machine.\n");
	}

	if (have_ntfsclone) {

		/*
		 * Can't clone if this volume is already scheduled to be
		 * part of a clone.
		 */
		if (!(pd->flags & (PDFLAG_NTFSCLONE_SOURCE | PDFLAG_NTFSCLONE_SOURCE))) {

			/*
			 * Must have at least one volume available for the
			 * target.
			 */
			list_anchor_t vol_list = NULL;
			boolean found_target = FALSE;

			EngFncs->get_volume_list(NULL, volume->disk_group, 0, &vol_list);
			if (vol_list != NULL) {
				list_element_t iter;
				logical_volume_t * vol;

				LIST_FOR_EACH(vol_list, iter, vol) {
					if (is_acceptable_clone_target(volume, vol) == 0) {
						found_target = TRUE;
						break;
					}
				}
			}

			if (found_target) {
				fia->info[i].function = EVMS_Task_ntfsclone;
				fia->info[i].name = EngFncs->engine_strdup("ntfsclone");
				fia->info[i].title = EngFncs->engine_strdup("ntfsclone");
				fia->info[i].verb = EngFncs->engine_strdup("ntfsclone");
				fia->info[i].help = EngFncs->engine_strdup(_("Run the ntfsclone utility to prepare the file system for booting under Windows."));

				if (EngFncs->is_mounted(volume->name, NULL)) {
					LOG_DEBUG("Volume %s is mounted.  It must be unmounted in order to run ntfsclone.\n",
						  volume->name);
					fia->info[i].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
				}

				i++;

			} else {
				LOG_DETAILS("No target volumes available.\n");
			}

		} else {
			LOG_DETAILS("ntfsclone is already scheduled to be run on this volume.\n");
		}

	} else {
		LOG_DETAILS("The ntfsclone utility in not installed on this machine.\n");
	}

	fia->count = i;
	*actions = fia;

	LOG_EXIT_INT(0);
	return 0;
}


/*
 * Execute the plug-in function on the volume.
 */
int ntfs_plugin_function(logical_volume_t * volume,
			 task_action_t      action,
			 list_anchor_t      objects,
			 option_array_t   * options) {

	int rc = 0;
	private_data_t * pd = (private_data_t *) volume->private_data;
	int i;
	logical_volume_t * target = NULL;
	private_data_t * target_pd;

	LOG_ENTRY();

	if (volume == NULL) {
		LOG_ERROR("No volume specified.\n");
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	switch (action) {
		case EVMS_Task_ntfsfix:
			pd->flags |= PDFLAG_RUN_NTFSFIX;
			
			/* Call me at commit time. */
			volume->flags |= VOLFLAG_DIRTY;
			break;

		case EVMS_Task_ntfsclone:
			for (i = 0; i < options->count; i++) {
				key_value_pair_t * option = &options->option[i];

				if (!option->is_number_based) {
					if (strcmp(option->name, NTFSCLONE_TARGET_NAME) == 0) {
						option->number = NTFSCLONE_TARGET_INDEX;

					} else if (strcmp(option->name, NTFSCLONE_FORCE_NAME) == 0) {
						option->number = NTFSCLONE_FORCE_INDEX;
					}
				}
				
				switch (option->number) {
					case NTFSCLONE_TARGET_INDEX:
						target = find_volume(option->value.s);
						break;

					case NTFSCLONE_FORCE_INDEX:
						if (option->value.b) {
							pd->flags |= PDFLAG_FORCE_NTFSCLONE;
						}
						break;

				}
			}

			if (target != NULL) {				
				/* Claim the target volume.*/
				rc = EngFncs->assign_fsim_to_volume(my_plugin_record, target);
				if (rc == 0) {
					target->private_data = EngFncs->engine_alloc(sizeof(private_data_t));
					if (target->private_data != NULL) {
						target_pd = (private_data_t *) target->private_data;

						target_pd->flags |= PDFLAG_NTFSCLONE_TARGET;
						target_pd->clone_source = volume;

						pd->flags |= PDFLAG_NTFSCLONE_SOURCE;
						pd->clone_target = target;

						fill_private_data(target);

						/* Call me at commit time. */
						volume->flags |= VOLFLAG_DIRTY;

					} else {
						LOG_CRITICAL("Unable to get memory for private data.\n");
						EngFncs->unassign_fsim_from_volume(target);
						rc = ENOMEM;
					}

				} else {
					LOG_WARNING("Failed to assign %s FSIM to volume %s.\n",
						    my_plugin_record->short_name, target->name);
				}

			} else {
				LOG_ERROR("No target volume given for the clone.\n");
				pd->flags &= ~PDFLAG_FORCE_NTFSCLONE;
				rc = EINVAL;
			}
			break;

		default:
			LOG_ERROR("Plug-in function %d (%#x) is not supported.\n",
				  action, action);
			rc = EINVAL;
			break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                                PLUG-IN FUNCTION TABLE                                +
+                                                                                      +
+--------------------------------------------------------------------------------------*/
static fsim_functions_t  fsim_ops = {

	setup_evms_plugin:    ntfs_setup,
	cleanup_evms_plugin:  ntfs_cleanup,
	probe:                ntfs_probe,
	get_fs_size:          ntfs_get_fs_size,
	get_fs_limits:        ntfs_get_fs_limits,
	mkfs_setup:           ntfs_mkfs_setup,
	mkfs:                 ntfs_mkfs,
	can_mkfs:             ntfs_can_mkfs,
	can_unmkfs:           ntfs_can_unmkfs,
	can_fsck:             ntfs_can_fsck,
	can_expand_by:        ntfs_can_expand_by,
	can_shrink_by:        ntfs_can_shrink_by,
	discard:              ntfs_discard,
	unmkfs_setup:         ntfs_unmkfs_setup,
	unmkfs:               ntfs_unmkfs,
	expand:               ntfs_expand,
	shrink:               ntfs_shrink,
	commit_changes:       ntfs_commit_changes,
	get_option_count:     ntfs_get_option_count,
	init_task:            ntfs_init_task,
	set_option:           ntfs_set_option,
	set_volumes:          ntfs_set_volumes,
	get_volume_info:      ntfs_get_volume_info,
	get_plugin_info:      ntfs_get_plugin_info,
	get_plugin_functions: ntfs_get_plugin_functions,
	plugin_function:      ntfs_plugin_function
};


/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                         PLUG-IN RECORD                                               +
+                                                                                      +
+-------------------------------------------------------------------------------------*/

static plugin_record_t  ntfs_plugrec = {
	id:                               EVMS_NTFS_PLUGIN_ID,
	version:                          {MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL},
	required_engine_api_version:      {15, 0, 0},
	required_plugin_api_version:      {fsim: {11, 0, 0}},
	short_name:                       EVMS_NTFS_PLUGIN_SHORT_NAME,
	long_name:                        EVMS_NTFS_PLUGIN_LONG_NAME,
	oem_name:                         EVMS_IBM_OEM_NAME,
	functions:                        {fsim: &fsim_ops},
	container_functions:              NULL

};

// Vector of plug-in record pointers that we export for the EVMS Engine.
plugin_record_t *evms_plugin_records[] = {
	&ntfs_plugrec,
	NULL
};

