/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libgpt.so
 *
 *   File: defsegmgr.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <uuid/uuid.h>

#include <plugin.h>

#include "gpt.h"
#include "gptsegmgr.h"
#include "helpers.h"
#include "checks.h"
#include "discovery.h"
#include "commit.h"
#include "options.h"
#include "dm.h"
#include "move.h"

guid_t EFI_SYSTEM_PARTITION = {
        0xC12A7328,
        0xF81F,
        0x11D2,
        0xBA,
        0x4B,
        {0x00, 0xA0, 0xC9, 0x3E, 0xC9, 0x3B}
};

guid_t BASIC_DATA_PARTITION = {
        0xEBD0A0A2,
        0xB9E5,
        0x4433,
        0x87,
        0xC0,
        {0x68, 0xB6, 0xB7, 0x26, 0x99, 0xC7}
};

guid_t LEGACY_MBR_PARTITION = {
        0x024DEE41,
        0x33E7,
        0x11D3,
        0x9D,
        0x69,
        {0x00, 0x08, 0xC7, 0x81, 0xF3, 0x9F}
};

guid_t GPT_SWAP_PARTITION =
{
        0x0657FD6D,
        0xA4AB,
        0x43C4,
        0x84,
        0xE5,
        {0x09, 0x33, 0xC8, 0x4B, 0x4F, 0x4F}
};

guid_t MS_RESERVED_PARTITION =
{
        0xE3C9E316,
        0x0B5C,
        0x4DB8,
        0x81,
        0x7D,
        {0xF9, 0x2D, 0xF0, 0x02, 0x15, 0xAE}
};

guid_t MS_LDM_DATA_PARTITION =
{
        0xAF9B60A0,
        0x1431,
        0x4F62,
        0xBC,
        0x68,
        {0x33, 0x11, 0x71, 0x4A, 0x69, 0xAD}
};

guid_t MS_LDM_METADATA_PARTITION =
{
        0x5808C8AAL,
        0x7E8F,
        0x42E0,
        0x85,
        0xD2,
        {0xE1, 0xE9, 0x04, 0x34, 0xCF, 0xB3}
};

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                         PRIVATE DATA AREAS AND SUBROUTINES                           +
+                                                                                      +
+-------------------------------------------------------------------------------------*/

static plugin_record_t          gpt_plugin_record;

plugin_record_t                *gpt_plugin_record_ptr = &gpt_plugin_record;

struct engine_functions_s      *EngFncs=NULL;

boolean                        Discovery_Completed = FALSE;



/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                            Start Of EVMS Plugin Functions                            +
+                        (exported to engine via function table)                       +
+                                                                                      +
+-------------------------------------------------------------------------------------*/
static int GPT_SetupEVMSPlugin( engine_functions_t * engine_functions)
{
        int rc = 0;
        EngFncs = engine_functions;
        LOG_ENTRY();
        LOG_EXIT_INT(rc);
        return rc;
}


static void GPT_Cleanup(void)
{
        list_anchor_t seglist=EngFncs->allocate_list();
        int rc;
        DISKSEG *seg;
        list_element_t iter;


        LOG_ENTRY();

        if (seglist != NULL) {

                // cleanup all seg private data areas
                rc = EngFncs->get_object_list( SEGMENT,
                                               0,
                                               gpt_plugin_record_ptr,
                                               NULL,
                                               0,
                                               &seglist );

                LIST_FOR_EACH(seglist, iter, seg ) {
                        if (seg->private_data) free(seg->private_data);
                }

                EngFncs->destroy_list(seglist);
        }

        // discard all disk private data
        delete_all_gpt_disk_private_data();

        LOG_EXIT_VOID();
}




/*
 *  I will allow the object to be made into a volume (or reverted) if ...
 *
 *  (1) I actually own the object
 *  (2) Which means it is a segment and has necessarey ctl blocks
 *
 */
static int GPT_can_set_volume(storage_object_t * seg, boolean flag )
{
        int rc = EINVAL;

        LOG_ENTRY();

        if ( i_can_modify( seg ) == TRUE ) {

                rc = 0;

        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  I can delete this segment if ...
 *
 *  (1) I own the segment
 *  (2) It is either a data segment or else a metadata segment and
 *      there are no data segments.
 */
static int GPT_CanDestroy( storage_object_t * seg )
{
        int rc = EINVAL;

        LOG_ENTRY();

        if ( ( seg ) &&
             ( seg->object_type == SEGMENT ) &&
             ( seg->data_type   == DATA_TYPE ) &&
             ( disk_move_pending(seg) == FALSE) &&
             ( i_can_modify( seg ) == TRUE )) {

                rc = 0;
        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  I can expand the object if ...
 *
 *  (1) it is a data segment
 *  (2) I own the segment
 *  (3) the logical disk and segment info is Ok
 *  (4) a freespace segment immediately follows it
 *  (5) there is at least a cylinder of space available
 *
 */
static int GPT_CanExpand( storage_object_t       *seg,        // data segment
                          sector_count_t         expand_limit,      // ?
                          list_anchor_t          expansion_points ) // list to place expand object on
{
        DISKSEG              *freespace;
        LOGICALDISK          *ld;
        int                   rc = EINVAL;
        expand_object_info_t *expand_object;
        SEG_PRIVATE_DATA     *pdata;
        sector_count_t        cylinder_size=0;

        LOG_ENTRY();

        if ( ( expansion_points ) &&
             ( seg ) &&
             ( seg->object_type == SEGMENT ) &&
             ( seg->data_type   == DATA_TYPE ) &&
             ( disk_move_pending(seg) == FALSE) &&
             ( i_can_modify( seg ) == TRUE ) ) {

                freespace = get_freespace_following_gpt_disk_segment( seg );
                ld        = get_logical_disk(seg);
                pdata     = (SEG_PRIVATE_DATA *)seg->private_data;

                if ( freespace && ld ) {

                        cylinder_size = get_cylinder_size(ld);

                        // we can only expand in cylinder size amounts
                        if ( freespace->size >= cylinder_size &&
                             expand_limit >= cylinder_size ) {

                                expand_object = (expand_object_info_t *) EngFncs->engine_alloc( sizeof(expand_object_info_t) );
                                if (expand_object) {
                                        list_element_t e;

                                        expand_object->object          = seg;
                                        expand_object->max_expand_size = min(freespace->size, expand_limit);

                                        e = EngFncs->insert_thing( expansion_points,
                                                                   expand_object,
                                                                   INSERT_AFTER,
                                                                   NULL);
                                        if (e!=NULL) {
                                                rc = 0;
                                        }
                                        else {
                                                rc = EPERM;
                                                EngFncs->engine_free( expand_object );
                                        }

                                }
                                else {
                                        LOG_ERROR("\nerror, alloc of expand object failed\n");
                                        rc = ENOMEM;
                                }

                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * I can expand the object by size sectors if:
 *
 *  (1) i own the object
 *  (2) the segment can be expanded
 *  (3) the freespace has at least a cylinder of space
 *
 * If I cannot expand because the freespace is too small
 * then I'll reduce the expand sector count to my maximum.
 *
 */
static int GPT_CanExpandBy(storage_object_t * seg, sector_count_t *size)
{
        int                rc = EINVAL;
        LOGICALDISK       *ld;
        DISKSEG           *freespace;
        sector_count_t     cylinder_size;
        sector_count_t     max_expand_sectors=0;
        lba_t              freespace_end_lba=0;
        SEG_PRIVATE_DATA  *pdata;


        LOG_ENTRY();

        if ( ( seg ) &&
             ( seg->object_type == SEGMENT ) &&
             ( seg->data_type   == DATA_TYPE ) &&
             ( disk_move_pending(seg) == FALSE) &&
             ( i_can_modify( seg ) == TRUE ) ) {

                freespace = get_freespace_following_gpt_disk_segment( seg );
                ld        = get_logical_disk(seg);
                pdata     = (SEG_PRIVATE_DATA *)seg->private_data;

                if ( freespace && ld ) {

                        cylinder_size = get_cylinder_size(ld);

                        // partitions end on a cylinder boundary. if freespace doesnt end on
                        // a cylinder bdy then round it down to a cyl bdy before testing if
                        // freespace can handle the ExpandBy.
                        if (ends_on_cylinder_boundary(ld,freespace->start+freespace->size-1)) {
                                freespace_end_lba = freespace->start+freespace->size-1;
                        }
                        else {
                                freespace_end_lba = rounddown_to_cylinder_boundary(ld, freespace->start+freespace->size-1) - 1;
                        }

                        // calculate the max useable size of the freespace area, i.e. the
                        // max area that ends on a cylinder boundary.
                        if (freespace_end_lba > freespace->start) {
                                max_expand_sectors = freespace_end_lba - freespace->start + 1;
                        }
                        else {
                                max_expand_sectors = 0;
                        }

                        // we expand in cylinder size chunks ... if max useable freespace
                        // size is less than 1 cylinder then we cant do any expand at all.
                        if (max_expand_sectors >= cylinder_size) {

                                if ( max_expand_sectors >= *size ) {

                                        if ( max_expand_sectors == *size) {
                                                rc = 0;
                                        }
                                        else {
                                                freespace_end_lba = roundup_to_cylinder_boundary(ld, freespace->start + *size - 1 );
                                                *size             = freespace_end_lba - freespace->start + 1;
                                        }

                                }
                                else {
                                        *size = max_expand_sectors;
                                        rc = EINVAL;
                                }

                        }

                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * I can shrink a seg if ...
 *
 *  (1) i own the object
 *  (2) it is a data segment
 *  (3) if I chop off a cylinder, the seg will still have
 *      a minimum of 1 cylinder of space
 *
 *  If not exact set new_size to closest higher value possible.
 */
static int GPT_CanShrink( storage_object_t * seg,            // object to shrink
                          sector_count_t     shrink_limit,   // a delta size
                          list_anchor_t            shrink_points ) // of type shrink_object_info_t
{
        int                   rc = EINVAL;
        sector_count_t        cylinder_size;
        LOGICALDISK          *ld;
        shrink_object_info_t *shrink_object=NULL;
        SEG_PRIVATE_DATA     *pdata;


        LOG_ENTRY();

        if ( ( seg ) &&
             ( seg->object_type == SEGMENT ) &&
             ( seg->data_type == DATA_TYPE ) &&
             ( disk_move_pending(seg) == FALSE) &&
             ( i_can_modify( seg ) == TRUE ) ) {


                pdata = (SEG_PRIVATE_DATA *)seg->private_data;
                ld    = get_logical_disk(seg);

                if ( ld ) {

                        cylinder_size = get_cylinder_size(ld);

                        // see if caller is trying to shrink smaller than min partition size
                        if ( seg->size > cylinder_size &&
                             shrink_limit >= cylinder_size ) {

                                if (shrink_points) {

                                        shrink_object = (shrink_object_info_t *) EngFncs->engine_alloc( sizeof(shrink_object_info_t) );
                                        if (shrink_object) {

                                                list_element_t e;

                                                // we are suppose to return the max amount we can shrink. since the
                                                // minimum partition size is 1 cylinder ... then anything more than
                                                // a cylinder is the max we can shrink.
                                                shrink_object->object          = seg;
                                                shrink_object->max_shrink_size = min(seg->size - cylinder_size, shrink_limit);

                                                e = EngFncs->insert_thing( shrink_points,
                                                                           shrink_object,
                                                                           INSERT_AFTER,
                                                                           NULL);
                                                if (e!=NULL) {
                                                        rc = 0;
                                                }
                                                else {
                                                        rc = EPERM;
                                                        EngFncs->engine_free( shrink_object );
                                                }

                                        }
                                        else {
                                                LOG_ERROR("\nerror, failed to alloc shrink_object\n");
                                                rc = ENOMEM;
                                        }
                                }

                        }

                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}




/*
 *  I can allow the storage object to shrink by the specified amount if ...
 *
 *  (1) the shrink point is a segment and I own it
 *  (2) the segment is large enough to allow a shrink
 *  (3) i can shrink it and end the segment on a cylinder boundary
 *
 *  If the segment doesnt end on a cylinder boundary I'll return
 *  an error and report the amount that we could shrink by.
 *
 */
static int GPT_CanShrinkBy( storage_object_t * seg,
                            sector_count_t   * size )
{
        int               rc = EINVAL;
        sector_count_t    cylinder_size;
        LOGICALDISK      *ld=NULL;
        sector_count_t    delta=0;
        SEG_PRIVATE_DATA *pdata;

        LOG_ENTRY();

        if ( ( seg ) &&
             ( size ) &&
             (*size > 0 ) &&
             ( seg->object_type == SEGMENT ) &&
             ( seg->data_type == DATA_TYPE ) &&
             ( disk_move_pending(seg) == FALSE) &&
             ( i_can_modify( seg ) == TRUE ) ) {

                pdata = (SEG_PRIVATE_DATA *)seg->private_data;
                ld    = get_logical_disk(seg);

                if ( ld ) {

                        cylinder_size = get_cylinder_size(ld);

                        // seg cant shrink if it is a cylinder or less in size
                        if (seg->size > cylinder_size) {

                                if ( *size < seg->size ) {
                                        if (*size < cylinder_size) {
                                                delta = cylinder_size;
                                        }
                                        else {
                                                delta = (*size / cylinder_size)*cylinder_size;
                                        }

                                }
                                else {
                                        delta = seg->size - cylinder_size;
                                }

                                if ( delta == *size  ) {
                                        rc = 0;
                                }
                                else {
                                        *size = delta;
                                        rc = EINVAL;
                                }

                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Called to run discovery code on list of evms objects that
 *  the engine has found so far.  We essentially need to walk the
 *  list of objects, probeing each object to see if it has a
 *  GPT header. If so, consume the object by removing it from the
 *  list, and place all new segment objects on the output_object
 *  list. Any object we dont like in the input_object list must
 *  be copied to the output_object list.
 */
static int GPT_Discover( list_anchor_t input_objects, list_anchor_t output_objects, boolean final_call)
{
        int  rc=0;
        uint  count = 0;
        storage_object_t *object;
        list_element_t iter;

        LOG_ENTRY();

        LIST_FOR_EACH( input_objects, iter, object) {
                rc = gpt_segment_discovery( object, output_objects, &count );
                if (rc) {
                        count=0;
                }
                else {
                        rc = count;
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}




/*
 *  Called with CREATE api for a DATA type object.  This means we assign
 *  ourselves to the disk by laying down our partition scheme, creating
 *  an MBR metadata segment and a FREESPACE segment.
 */
static int Assign_SegmentManager_ToDisk( LOGICALDISK *ld, boolean create_system_partition )
{
        int                 rc=0;
        DISKSEG            *md1=NULL;
        DISKSEG            *md2=NULL;
        DISKSEG            *esp=NULL;
        DISKSEG            *pmbr=NULL;
        SEG_PRIVATE_DATA *pdata;
        gpt_header         *gh1= (gpt_header *) calloc(1, EVMS_VSECTOR_SIZE);
        gpt_header         *gh2= (gpt_header *) calloc(1, EVMS_VSECTOR_SIZE);
        DISK_PRIVATE_DATA  *disk_pdata=NULL;
        char               *uuid_string=NULL;

        lba_t               gh_lba1;
        lba_t               pt_lba1;
        lba_t               gh_lba2;
        lba_t               pt_lba2;
        lba_t               esp_lba;
        lba_t               esp_end_lba;
        lba_t               start_useable;
        lba_t               end_useable;

        sector_count_t      pt_size=0;
        sector_count_t      esp_size=0;
        sector_count_t      esp_max_size=0;

        int                 pt_entry_size=0;
        int                 pt_count=0;

        guid_t              disk_id;
        char devname[EVMS_VOLUME_NAME_SIZE+1];

        LOG_ENTRY();

        // we should not have a private data area for this disk yet because we were just
        // told to install on it.  we must have been passed a logical disk we are already
        // using and this is very wrong.
        if ( get_gpt_disk_private_data(ld) ) {
                LOG_ERROR("attempting to reinstall on disk (%s) but we already seem to know about the disk.\n", ld->name);
                rc = EINVAL;
                LOG_EXIT_INT(rc);
                return rc;
        }

        // test memory allocations
        if (gh1 == NULL || gh2 == NULL) {
                LOG_ERROR("error, attempted to malloc 2 sectors to be used as gpt headers and failed\n");
                rc = ENOMEM;
                LOG_EXIT_INT(rc);
                return rc;
        }

        // create disk private data area
        create_gpt_disk_private_data( ld );
        disk_pdata = get_gpt_disk_private_data(ld);
        if (disk_pdata == NULL) {
                LOG_ERROR("error, attempted to create disk private data and failed\n");
                free(gh1);
                free(gh2);
                rc = ENOMEM;
                LOG_EXIT_INT(rc);
                return rc;
        }

        //
        //  CREATE METADATA INFO
        //

        // Locate the primary gpt header immediately after the PMBR
        gh_lba1 = 1;

        // Locate the primary partition table immediately after the GPT header
        pt_lba1 = 2;
        pt_size = ld->geometry.sectors_per_track - 2;
        pt_entry_size = sizeof(gpt_partition);
        pt_count = (pt_size*EVMS_VSECTOR_SIZE) / pt_entry_size;

        // Locate the start of useable area immediately after the partition table
        start_useable = pt_lba1 + pt_size;

        // Locate the efi system partition at the start of the useable area
        esp_lba     = start_useable;
        esp_end_lba = roundup_to_cylinder_boundary(ld, esp_lba );
        esp_size    = esp_end_lba - esp_lba + 1;

        esp_max_size = ld->size * .01;                    // calc max size ... 1 percent of disk
        if ( esp_max_size > GPT_ESP_MAX_SECTOR_COUNT ) {  // with max of 100 MB
                esp_max_size = GPT_ESP_MAX_SECTOR_COUNT;
        }

        esp_end_lba = rounddown_to_cylinder_boundary(ld, (esp_lba+esp_size+esp_max_size-1) ) - 1;
        esp_size    = esp_end_lba - esp_lba + 1;

        // Locate alternate GPT header immediately following the partition table
        gh_lba2     = ld->size - 1;

        // locate alternate partition table (now track alligned) in last track of the drive
        pt_lba2     = gh_lba2 - pt_size;

        // Locate end of useable area ... leaving room for second copy of metadata
        end_useable = pt_lba2 - 1;

        // generate a unique disk id
        uuid_generate_random( (char *) &disk_id );



        // REGISTER DISK - convert DISK uuid to ascii string and register it.
        uuid_string = guid_to_string( &disk_id );
        if (uuid_string) {
                rc = EngFncs->register_name( uuid_string );
                free(uuid_string);
        }
        else {
                rc = ENOMEM;
        }

        if (rc) {
                free(gh1);
                free(gh2);
                LOG_ERROR("error, unable to convert DISK uuid identifier to ascii string and register it.\n");
                LOG_EXIT_INT(rc);
                return rc;
        }


        //
        // CREATE GPT HEADERS
        //
        gh1->signature         = GPT_DISKMAGIC;
        gh1->version           = GPT_VERSION_1_2;
        gh1->size              = sizeof(gpt_header);
        gh1->my_lba            = gh_lba1;
        gh1->alternate_lba     = gh_lba2;
        gh1->start_useable     = start_useable;
        gh1->end_useable       = end_useable;
        gh1->ptable_lba        = pt_lba1;
        gh1->ptable_count      = pt_count;
        gh1->ptable_entry_size = pt_entry_size;
        memcpy(&gh1->disk_id, &disk_id, sizeof(guid_t));

        gh2->signature         = GPT_DISKMAGIC;
        gh2->version           = GPT_VERSION_1_2;
        gh2->size              = sizeof(gpt_header);
        gh2->my_lba            = gh_lba2;
        gh2->alternate_lba     = gh_lba1;
        gh2->start_useable     = start_useable;
        gh2->end_useable       = end_useable;
        gh2->ptable_lba        = pt_lba2;
        gh2->ptable_count      = pt_count;
        gh2->ptable_entry_size = pt_entry_size;
        memcpy(&gh2->disk_id, &disk_id, sizeof(guid_t));


        //
        // CREATE SEGMENT STORAGE OBJECTS
        //
        // Needed: 2 metadata segments (primary and alternate copies)
        //         1 protective mbr segment
        //         1 efi system partition
        //

        md1 = create_gpt_metadata_segment( ld,
                                           gh_lba1,
                                           pt_size+1,
                                           "metadata1" );
        if (md1) {
                disk_pdata->md1 = md1;
                ((SEG_PRIVATE_DATA *)md1->private_data)->gh = gh1;
                md1->flags |= SOFLAG_DIRTY;
        }

        md2 = create_gpt_metadata_segment( ld,
                                           pt_lba2,
                                           pt_size + 1,
                                           "metadata2" );

        if (md2) {
                disk_pdata->md2 = md2;
                ((SEG_PRIVATE_DATA *)md2->private_data)->gh = gh2;
                md2->flags |= SOFLAG_DIRTY;
        }

        get_device_name(ld, devname);

        if (create_system_partition == TRUE) {

                esp = allocate_gpt_disk_segment( ld );
                if (esp) {
                        pdata = (SEG_PRIVATE_DATA *)esp->private_data;

                        esp->size        = esp_size;
                        esp->start       = esp_lba;
                        esp->data_type   = DATA_TYPE;

                        pdata->minor     = 1;
                        pdata->cflags   |= SEG_CFLAG_TOP_SEGMENT;
                        pdata->type      = efi_system_partition;
                        pdata->gh        = gh1;

                        memcpy(&pdata->guid_type, &EFI_SYSTEM_PARTITION, sizeof(guid_t) );

                        uuid_generate_random( (char *) &pdata->guid_id );

                        if ( isa_disk_object(ld) == FALSE ) {
                                sprintf(esp->name, "%s.%d", devname, 1);
                        }
                        else {
                                if (devname[strlen(devname)-1] == '/') {
                                        strcat(devname, "part");
                                }

                                sprintf(esp->name, "%s%d", devname, 1);
                        }

                        esp->flags |= SOFLAG_DIRTY;
                }

        }
        else {
                esp = NULL;
        }

        pmbr = allocate_gpt_disk_segment( ld );
        if (pmbr) {

                pmbr->size        = 1;
                pmbr->start       = 0;
                pmbr->data_type   = META_DATA_TYPE;

                if ( isa_disk_object(ld) == FALSE ) {
                        sprintf(pmbr->name, "%s.pmbr", devname );
                }
                else {
                        if (devname[strlen(devname)-1] == '/') {
                                sprintf(pmbr->name, "%spmbr", devname);
                        }
                        else {
                                sprintf(pmbr->name, "%s_pmbr", devname);
                        }
                }

                pmbr->flags |= SOFLAG_DIRTY;
        }



        //
        // CREATE DISK PARENT OBJECT list
        //

        // toss any items that may have been left in the parent list
        EngFncs->delete_all_elements(ld->parent_objects);

        // add metadata segments ...
        if ( (insert_gpt_segment_into_list( ld->parent_objects, pmbr) == NULL )||
             (insert_gpt_segment_into_list( ld->parent_objects, md1)  == NULL )||
             (insert_gpt_segment_into_list( ld->parent_objects, md2)  == NULL )) {

                EngFncs->delete_all_elements(ld->parent_objects);
                delete_gpt_disk_private_data( ld );
                if (md1) free_gpt_disk_segment(md1);
                if (md2) free_gpt_disk_segment(md2);
                if (pmbr) free_gpt_disk_segment(pmbr);
                free(gh1);
                free(gh2);
                LOG_EXIT_INT(ENOMEM);
                return ENOMEM;

        }

        // add system partition segment ...
        if ( ( create_system_partition == TRUE ) &&
             ( insert_gpt_segment_into_list( ld->parent_objects, esp)  == NULL )) {

                EngFncs->delete_all_elements(ld->parent_objects);
                delete_gpt_disk_private_data( ld );
                if (md1) free_gpt_disk_segment(md1);
                if (md2) free_gpt_disk_segment(md2);
                if (esp) free_gpt_disk_segment(esp);
                if (pmbr) free_gpt_disk_segment(pmbr);
                free(gh1);
                free(gh2);
                LOG_EXIT_INT(ENOMEM);
                return ENOMEM;

        }

        // expose freespca on disk
        find_freespace_on_gpt_disk( ld );

        // mark disk has changes for commit code
        disk_pdata->flags |= DISK_HAS_CHANGES_PENDING;

        rc = 0;
        LOG_EXIT_INT(rc);
        return rc;

}


static int get_assign_options(  option_array_t * options,  boolean *create_system_partition )
{
        int i;
        int rc = EINVAL;


        LOG_ENTRY();

        for (i = 0; i < options->count; i++) {

                if (options->option[i].is_number_based) {

                        switch (options->option[i].number) {

                        case SEG_ASSIGN_OPTION_SYSPARTITION_INDEX:

                                *create_system_partition = options->option[i].value.b;
                                rc =0;
                                break;

                        default:
                                rc = EINVAL;
                                break;

                        }

                }
                else {

                        if (strcmp(options->option[i].name,SEG_ASSIGN_OPTION_SYSPARTITION_NAME) == 0) {

                                *create_system_partition = options->option[i].value.b;
                                rc =0;
                                break;

                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to assign the GPT partitioning scheme to the specified
 *  DISK storage object.
 *
 *  05/14/2002 - no assign options yet.
 */
static int GPT_Assign( storage_object_t * input_object,
                       option_array_t * options)
{
        int rc=EINVAL;
        boolean create_system_partition;

        LOG_ENTRY();

        if (input_object != NULL && options != NULL) {

                LOG_DEBUG("adding gpt seg manger to object %s\n", input_object->name );

                rc = get_assign_options( options,  &create_system_partition );
                if (rc == 0) {

                        rc = Assign_SegmentManager_ToDisk( input_object, create_system_partition );

                }

        }
        else {
                LOG_ERROR("one or more input parms are NULL ptrs\n");
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to test if we can unassign from the specified
 *  object.
 */
static int GPT_CanUnassign( LOGICALDISK *ld )
{
        int rc=EINVAL;
        DISK_PRIVATE_DATA *disk_pdata=NULL;


        LOG_ENTRY();

        disk_pdata = get_gpt_disk_private_data( ld );

        if ( disk_pdata ) {  // is this a logical disk and do we
                             // have private data for it ?
                rc = 0;          // return ... success

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to remove the GPT partitioning scheme from the
 *  specified object.
 */
static int GPT_UnAssign( LOGICALDISK *ld )
{
        int rc=EINVAL;
        DISK_PRIVATE_DATA *disk_pdata=NULL;
        DISKSEG           *seg=NULL;
        gpt_header        *gh=NULL;
        char              *uuid_string=NULL;
        list_element_t     iter,iter2;


        LOG_ENTRY();

        disk_pdata = get_gpt_disk_private_data(ld);
        if ( disk_pdata ) {

                // unregister the disk uuid identifier
                gh = ((SEG_PRIVATE_DATA *)disk_pdata->md1->private_data)->gh;
                if (gh) {

                        uuid_string = guid_to_string( &gh->disk_id );
                        if (uuid_string) {
                                EngFncs->unregister_name( uuid_string );
                                free(uuid_string);
                        }

                }

                LIST_FOR_EACH_SAFE( ld->parent_objects, iter, iter2, seg ) {
                        free_gpt_disk_segment(seg);
                        EngFncs->delete_element(iter);
                }

                // blast PMBR
                KILL_SECTORS( ld, 0, 1);

                // blast 1st copy of metadata
                KILL_SECTORS(ld, 1, 1);

                // blast 2nd copy of metadata
                KILL_SECTORS(ld, ld->size-1, 1);

                // toss our disk private data area
                delete_gpt_disk_private_data( ld );

                rc = 0;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*  Called to create the specified segment.
 *
 *  - allocate memory for segment,
 *  - add segment to seg list of logical disk.
 *  - fill in seg info
 */
static int CreateDiskSegment( storage_object_t      * freespace,            /* a free space disk segment      */
                              storage_object_t    * * new_segment,          /* resulting data segment         */
                              sector_count_t          size,                 /* size in sectors                */
                              sector_count_t          sector_offset,        /* sector offset in freespace seg */
                              partition_type_t        ptype  )
{
        int                   rc=0;
        storage_object_t     *seg=NULL;
        SEG_PRIVATE_DATA *pdata=NULL;
        LOGICALDISK          *ld=NULL;
        DISK_PRIVATE_DATA    *disk_pdata=NULL;
        sector_count_t        sizeof_freespace = freespace->size;
        lba_t                 startof_freespace = freespace->start;
        sector_count_t        seg_size;
        lba_t                 seg_start_lba;
        lba_t                 seg_end_lba;
        sector_count_t        cylsize;
        int                   next_minor;
        char devname[EVMS_VOLUME_NAME_SIZE+1];


        LOG_ENTRY();

        ld = get_logical_disk(freespace);
        if (ld == NULL) {
                LOG_ERROR("error, unable to get gpt logical disk segment from freespace parm (%s)\n", freespace->name );
                rc = EINVAL;
                LOG_EXIT_INT(rc);
                return rc;
        }
        disk_pdata = get_gpt_disk_private_data(ld);
        if (disk_pdata == NULL) {
                LOG_ERROR("error, unable to get gpt logical disk private data from disk (%s)\n", ld->name );
                rc = EINVAL;
                LOG_EXIT_INT(rc);
                return rc;
        }
        cylsize = get_cylinder_size(ld);
        if (cylsize==0) {
                LOG_ERROR("error, unable to get the cylinder size of disk %s.\n", ld->name );
                rc = EINVAL;
                LOG_EXIT_INT(rc);
                return rc;
        }

        /*
         *  Dump debug info to engine log
         */
        LOG_DEBUG("New Seg Parms ...\n");
        LOG_DEBUG("         Size: %"PRIu64"\n", size);
        LOG_DEBUG("       Offset: %"PRIu64"\n", sector_offset );
        LOG_DEBUG("FreeSpace ...\n");
        LOG_DEBUG("    Start LBA: %"PRIu64"\n", freespace->start);
        LOG_DEBUG("         Size: %"PRIu64"\n", freespace->size);
        LOG_DEBUG("     Cyl Size: %"PRIu64"\n", get_cylinder_size(ld) );
        if (starts_on_cylinder_boundary( ld, freespace->start )==TRUE) {
                LOG_DEBUG("  Cyl Bdy: Yes ... starts on cylinder boundary\n" );
        }
        else {
                LOG_DEBUG("  Cyl Bdy: No ... does not start on cylinder boundary\n");
        }


        /*
         *  Start segment on a cylinder boundary if required
         */
        if ( starts_on_cylinder_boundary(ld, freespace->start+sector_offset) == FALSE ) {

                if ( sector_offset < cylsize ) {
                        seg_start_lba   = roundup_to_cylinder_boundary(ld, freespace->start + sector_offset)+1;
                }
                else {
                        seg_start_lba   = rounddown_to_cylinder_boundary(ld, freespace->start + sector_offset);
                }

        }
        else {
                seg_start_lba = freespace->start + sector_offset;
        }


        /*
         *  End segment on a cylinder boundary
         */
        seg_end_lba = seg_start_lba + size - 1;

        if ( ends_on_cylinder_boundary(ld, seg_end_lba) == FALSE ) {

                // if less than a cylinder ... round up to next cyl boundary
                if ( ((seg_end_lba - seg_start_lba + 1) / cylsize) == 0) {
                        seg_end_lba = roundup_to_cylinder_boundary(ld, seg_end_lba);
                }
                else {
                        seg_end_lba = rounddown_to_cylinder_boundary(ld, seg_end_lba) - 1;
                }

                // if we goofed somehow and rounded down to start of seg then fix it!
                if ( seg_start_lba >= seg_end_lba ) {
                        seg_end_lba     = roundup_to_cylinder_boundary(ld, seg_start_lba+1);
                }

        }

        /*
         *  Now get its actual size
         */
        seg_size = seg_end_lba - seg_start_lba + 1;

        /*
         * Test starting LBA
         */
        if (  seg_start_lba < freespace->start ) {

                LOG_ERROR("error, with cylinder allignment, the new seg would start before the free_space segment\n");
                *new_segment = NULL;
                rc = EINVAL;
                LOG_EXIT_INT(rc);
                return rc;

        }

        /*
         *  Test ending LBA
         */
        if (  seg_end_lba > (freespace->start + freespace->size -1) ) {

                LOG_ERROR("error, with cylinder allignment the new segment would run past end of free_space segment\n");
                *new_segment = NULL;
                rc = EINVAL;
                LOG_EXIT_INT(rc);
                return rc;

        }


        /*
         *  Calculate new dimensions for the existing freespace segment
         */
        if ( seg_start_lba == freespace->start ) {
                freespace->size  -= seg_size;
                freespace->start += seg_size;
        }
        else {
                freespace->size   = seg_start_lba - freespace->start;
        }


        /*
         *  Dump debug info to engine log
         */
        LOG_DEBUG("Create Parms:\n");
        LOG_DEBUG("FreeSpace: start_lba= %"PRIu64"  sector_offset= %"PRIu64"  size= %"PRIu64"  name= %s\n", freespace->start, sector_offset, freespace->size,freespace->name );
        LOG_DEBUG("   NewSeg: start_lba= %"PRIu64"  end_lba= %"PRIu64"  size= %"PRIu64"\n", seg_start_lba, seg_end_lba, seg_size);


        seg = allocate_gpt_disk_segment( ld );
        if (seg) {

                next_minor = get_next_gpt_minor(ld);
                if (next_minor != -1) {

                        pdata = (SEG_PRIVATE_DATA *)seg->private_data;

                        seg->size        = seg_size;
                        seg->start       = seg_start_lba;
                        seg->data_type   = DATA_TYPE;

                        pdata->minor     = next_minor;
                        pdata->type      = ptype;

                        if (ptype == efi_system_partition) {
                                pdata->cflags   |= SEG_CFLAG_TOP_SEGMENT;
                                memcpy(&pdata->guid_type, &EFI_SYSTEM_PARTITION, sizeof(guid_t) );
                        }
                        else if (ptype == basic_data_partition) {
                                memcpy(&pdata->guid_type, &BASIC_DATA_PARTITION, sizeof(guid_t) );
                        }
                        else if (ptype == legacy_mbr_partition) {
                                memcpy(&pdata->guid_type, &LEGACY_MBR_PARTITION, sizeof(guid_t) );
                        }
                        else if (ptype == swap_partition) {
                                memcpy(&pdata->guid_type, &GPT_SWAP_PARTITION, sizeof(guid_t) );
                        }

                        uuid_generate_random( (char *) &pdata->guid_id );

                        get_device_name(ld, devname);

                        if ( isa_disk_object(ld) == FALSE ) {
				char last_char = devname[strlen(devname)-1];

				if ((last_char >= '1') && (last_char <= '9')) {
					// Must be an embedded GPT partition.
					strcat(devname, ".");
				}
                        }
                        else {
                                if (devname[strlen(devname)-1] == '/') {
                                        strcat(devname, "part");
                                }
                        }
			sprintf(seg->name, "%s%d", devname, next_minor);

                        if ( insert_gpt_segment_into_list( ld->parent_objects, seg ) == NULL ) {

                                LOG_ERROR("error, some kind of list insert error");
                                rc = ENOMEM;

                        }
                        else {

                                if ( freespace->size == 0 ) {
                                        LOG_DEBUG("removing freespace segment from disk list because we used it all up\n");
                                        remove_gpt_segment_from_list(ld->parent_objects, freespace );
                                        free_gpt_disk_segment(freespace);
                                }

                                find_freespace_on_gpt_disk( ld );  // re-examine the disk for any free space and expose it by
                                                                   // hanging free space segments.

                                // mark disk has changes for commit code
                                disk_pdata->flags |= DISK_HAS_CHANGES_PENDING;

                                rc = 0;
                        }

                }
                else {
                        LOG_ERROR("error, no room left in partition table on disk %s.\n",ld->name);
                        rc = EINVAL;
                }

        }

        if (rc) {
                freespace->size  = sizeof_freespace;  // restore freespace segment
                freespace->start = startof_freespace;

                if (seg) free_gpt_disk_segment(seg);  // free memory
                seg = NULL;
        }

        *new_segment = seg;            // return the new data segment to the caller

        LOG_EXIT_INT(rc);
        return rc;
}



static int GetCreateOptions(  storage_object_t  * seg,
                              option_array_t    * options,
                              sector_count_t    * size,
                              lsn_t             * sector_offset,
                              partition_type_t  * ptype )
{
        int i;
        int rc = EINVAL;
        char partition_type[MAX_TYPENAME_SIZE+1];

        LOG_ENTRY();

        // init the options before beginning
        *ptype = basic_data_partition;
        *sector_offset = 0;
        *size = 0;

        // pickup option values
        for (i = 0; i < options->count; i++) {

                if (options->option[i].is_number_based) {

                        switch (options->option[i].number) {

                        case SEG_CREATE_OPTION_SIZE_INDEX:
                                *size = options->option[i].value.ui64;
                                break;

                        case SEG_CREATE_OPTION_OFFSET_INDEX:
                                *sector_offset = options->option[i].value.ui64;
                                break;

                        default:
                                break;

                        }

                }
                else {

                        if (strcmp(options->option[i].name, SEG_CREATE_OPTION_SIZE_NAME) == 0) {
                                *size = options->option[i].value.ui64;
                        }
                        else if (strcmp(options->option[i].name, SEG_CREATE_OPTION_OFFSET_NAME) == 0) {
                                *sector_offset = options->option[i].value.ui64;
                        }

                }
        }


        LOG_DEBUG("Create Options ...\n");
        LOG_DEBUG("              size: %"PRIu64"\n", *size);
        LOG_DEBUG("     sector offset: %"PRIu64"\n", *sector_offset);
        LOG_DEBUG("              type: %s\n", partition_type);

        // test results
        if ( (*size > 0) &&
             (*sector_offset >= 0) &&
             (*ptype != unknown_partition) ) {

                rc = 0;

        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Create storage_object_t(s) from the list of objects using the given
 * options.  Return the newly allocated storage_object_t(s) in new_objects
 * list.
 */
static int GPT_CreateSegment( list_anchor_t          input_objects,
                              option_array_t * options,
                              list_anchor_t          new_objects)
{
        int                 rc;
        uint                object_count=0;
        storage_object_t   *free_space_seg;
        storage_object_t   *newseg;

        //  Options ...
        sector_count_t      size;
        sector_count_t      sector_offset;
        partition_type_t    ptype;



        LOG_ENTRY();

        // we should only be called with a single input object
        object_count = EngFncs->list_count( input_objects );

        if ( object_count == 1 ) {

                free_space_seg = EngFncs->first_thing(input_objects, NULL);
                if ( free_space_seg != NULL ) {

                        if ( ( i_can_modify(free_space_seg) == TRUE ) &&
                             ( free_space_seg->data_type == FREE_SPACE_TYPE )) {

                                // get seg create options
                                rc = GetCreateOptions( free_space_seg,
                                                       options,
                                                       &size,
                                                       &sector_offset,
                                                       &ptype );

                                if (rc) {
                                        LOG_ERROR("invalid options\n");
                                        LOG_EXIT_INT(EINVAL);
                                        return EINVAL;
                                }

                                rc = CreateDiskSegment( free_space_seg,
                                                        &newseg,
                                                        size,
                                                        sector_offset,
                                                        ptype );

                                if (rc ==0) {
                                        list_element_t e;

                                        e = EngFncs->insert_thing( new_objects,
                                                                   newseg,
                                                                   INSERT_AFTER,
                                                                   NULL);

                                        if (e) {
                                                rc = 0;
                                        }
                                        else {
                                                rc = EPERM;
                                        }
                                }


                        }
                        else {
                                LOG_ERROR("object, to be consumed by create, has the wrong data_type\n");
                                rc = EINVAL;
                        }

                }
                else {
                        LOG_ERROR("error returned from list find_thing call\n");
                        rc = EINVAL;
                }
        }
        else {
                LOG_ERROR("expected 1 object in the input list but found %d\n", object_count);
                rc = EINVAL;
        }


        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Called to free a data segment.  If the corresponding disk partition
 *  is a logical drive then we also need to free the EBR segment
 *  as well.
 */
static int GPT_DestroySegment( DISKSEG * seg, list_anchor_t child_objects )
{
        int           rc = EINVAL;
        DISKSEG      *pmbr=NULL;
        LOGICALDISK  *ld=NULL;
        DISK_PRIVATE_DATA *disk_pdata=NULL;

        LOG_ENTRY();

        LOG_DEBUG("seg: %s\n", seg->name );

        if ( ( seg ) &&
             ( seg->object_type == SEGMENT ) &&
             ( seg->data_type   == DATA_TYPE ) &&
             ( i_can_modify( seg ) == TRUE)) {

                ld         = get_logical_disk(seg);
                disk_pdata = get_gpt_disk_private_data(ld);

                if (ld && disk_pdata) {

                        pmbr = EngFncs->first_thing( ld->parent_objects, NULL );
                        if (pmbr) {

                                // remove segment from list and unregister its name
                                rc = remove_gpt_segment_from_list( ld->parent_objects, seg );
                                if (rc == 0) {

                                        free_gpt_disk_segment(seg);

                                        find_freespace_on_gpt_disk( ld );

                                        disk_pdata->flags |= DISK_HAS_CHANGES_PENDING;

                                        pmbr->flags |= SOFLAG_DIRTY;
                                }

                        }

                }

        }

        LOG_EXIT_INT(rc);
        return  rc;
}

/*
 * Forget about these objects.  Don't delete them.  Just clean up any
 * data structures you may have associated with them.  The Engine will
 * call to deactivate the objects durring commit.
 */
static int GPT_Discard(list_anchor_t objects)
{
	int rc=0;
	LOGICALDISK *ld;
	DISKSEG *seg;
	list_element_t iter;
        DISK_PRIVATE_DATA *disk_pdata=NULL;
        gpt_header        *gh=NULL;
        char              *uuid_string=NULL;

	LOG_ENTRY();

	LIST_FOR_EACH( objects, iter, seg ) {
		if ( ( seg->object_type == SEGMENT ) &&
		     ( i_can_modify( seg ) == TRUE)) {

			ld = get_logical_disk(seg);
			if (ld) {
				remove_gpt_segment_from_list( ld->parent_objects, seg );

				disk_pdata = get_gpt_disk_private_data(ld);
				if ( disk_pdata ) {

					if (seg == disk_pdata->md1) {
						// unregister the disk uuid identifier
						gh = ((SEG_PRIVATE_DATA *)seg->private_data)->gh;
						if (gh) {

							uuid_string = guid_to_string( &gh->disk_id );
							if (uuid_string) {
								EngFncs->unregister_name( uuid_string );
								free(uuid_string);
							}

						}
					}
				}

				if (EngFncs->list_empty(ld->parent_objects) == TRUE) {
					// toss our disk private data area
					delete_gpt_disk_private_data( ld );
				}

			}

			// Free segment and private data
			free_gpt_disk_segment(seg);

		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}



static void get_expand_options( option_array_t * options,  sector_count_t  * size)
{
        int i;

        LOG_ENTRY();

        for (i = 0; i < options->count; i++) {

                if (options->option[i].is_number_based) {

                        if (options->option[i].number == SEG_EXPAND_OPTION_SIZE_INDEX) {
                                *size = options->option[i].value.ui64;
                        }

                }
                else {

                        if (strcmp(options->option[i].name, SEG_EXPAND_OPTION_SIZE_NAME) == 0) {
                                *size = options->option[i].value.ui64;
                        }

                }
        }


        LOG_EXIT_VOID();
}


/*
 *  Called to expand a data segment.  The segment will be expanded
 *  into the freespace segment that follows the data segment.
 */
static int GPT_Expand( storage_object_t *seg,
                       storage_object_t *expand_object,
                       list_anchor_t           objects,
                       option_array_t   *options )
{
        int               rc = EINVAL;
        sector_count_t    expand_sectors=0;
        DISKSEG          *freespace;
        lba_t             end_lba;
        sector_count_t    old_seg_size;
        LOGICALDISK      *ld=NULL;
        SEG_PRIVATE_DATA *pdata=NULL;
        sector_count_t    cylinder_size=0;
        sector_count_t    max_expand_sectors=0;
        lba_t             freespace_end_lba;
        DISK_PRIVATE_DATA *disk_pdata=NULL;

        LOG_ENTRY();

        // initial checks to see if we can do the expand
        if ( ( seg ) &&
             ( seg == expand_object ) &&
             ( seg->object_type == SEGMENT ) &&
             ( seg->data_type   == DATA_TYPE ) &&
             ( i_can_modify( seg ) == TRUE )) {

                pdata     = (SEG_PRIVATE_DATA *)seg->private_data;
                ld        = get_logical_disk(seg);
                disk_pdata = get_gpt_disk_private_data(ld);
                freespace = get_freespace_following_gpt_disk_segment( seg );

                get_expand_options( options,  &expand_sectors);

                if ( freespace && ld && disk_pdata && expand_sectors  ) {

                        LOG_DEBUG("     Data Seg  Name: %s\n", seg->name);
                        LOG_DEBUG("              Start: %"PRIu64"\n", seg->start);
                        LOG_DEBUG("               Size: %"PRIu64"\n", seg->size);
                        LOG_DEBUG("     Freespace Name: %s\n", freespace->name);
                        LOG_DEBUG("              Start: %"PRIu64"\n", freespace->start);
                        LOG_DEBUG("               Size: %"PRIu64"\n", freespace->size);

                        cylinder_size = get_cylinder_size(ld);

                        // just in case ...
                        if ( freespace->size < cylinder_size ) {
                                LOG_ERROR("error, trying to expand into free space that is less than 1 cylinder\n");
                                LOG_EXIT_INT(rc);
                                return rc;
                        }

                        // partitions end on a cylinder boundary. if freespace doesnt end on
                        // a cylinder bdy then round it down to a cyl bdy before testing if
                        // freespace can handle the ExpandBy.
                        if (ends_on_cylinder_boundary(ld,freespace->start+freespace->size-1)) {
                                freespace_end_lba = freespace->start+freespace->size-1;
                        }
                        else {
                                freespace_end_lba = rounddown_to_cylinder_boundary(ld, freespace->start+freespace->size-1) - 1;
                        }

                        // calculate the max useable size of the freespace area, i.e. the
                        // max area that ends on a cylinder boundary.
                        if (freespace_end_lba > freespace->start) {
                                max_expand_sectors = freespace_end_lba - freespace->start + 1;
                        }
                        else {
                                LOG_ERROR("error, cant cylinder allign end of segment in available freespace segment\n");
                                LOG_EXIT_INT(rc);
                                return rc;
                        }

                        // you just never know ...
                        if ( expand_sectors > max_expand_sectors ) {
                                expand_sectors = max_expand_sectors;
                        }

                        // do cylinder alignment
                        end_lba = seg->start + seg->size + expand_sectors - 1;

                        if (ends_on_cylinder_boundary(ld, end_lba)==FALSE) {
                                end_lba = roundup_to_cylinder_boundary( ld, end_lba );
                        }

                        // now adjust downwards if too big for freespace ...
                        if ( end_lba > (freespace->start + freespace->size - 1) ) {
                                end_lba = rounddown_to_cylinder_boundary(ld, end_lba-1 ) - 1;
                        }

                        // test if we can expand the data seg into the freespace area
                        if ( ( end_lba > freespace->start ) &&
                             ( end_lba <= freespace->start+freespace->size - 1) ) {

                                // calc actual expand sector count
                                expand_sectors = end_lba - freespace->start + 1;

				// ask the engine if this expand amount is ok.
				rc = EngFncs->can_expand_by(seg, &expand_sectors);
				if (rc) {
					LOG_ERROR("Shrink of segment %s rejected by "
						  "the engine.\n", seg->name);
					LOG_EXIT_INT(rc);
					return rc;
				}

                                // expand the data segment
                                old_seg_size = seg->size;
                                seg->size   += expand_sectors;

                                // shrink the freespace segment
                                freespace->start  += expand_sectors;
                                freespace->size   -= expand_sectors;

                                // mark seg dirty for commit and needs DM activate
				seg->flags |= SOFLAG_DIRTY;
				if (seg->flags & SOFLAG_ACTIVE) {
					seg->flags |= SOFLAG_NEEDS_ACTIVATE;
				}

                                LOG_DEBUG("     Data Seg  Name: %s\n", seg->name);
                                LOG_DEBUG("           New Start: %"PRIu64"\n", seg->start);
                                LOG_DEBUG("           New Size: %"PRIu64"\n", seg->size);
                                LOG_DEBUG("     Freespace Name: %s\n", freespace->name);
                                LOG_DEBUG("           New Start: %"PRIu64"\n", freespace->start);
                                LOG_DEBUG("           New Size: %"PRIu64"\n", freespace->size);

                                // success.
                                rc = 0;

                                // if we used up all the free space then discard the freespace seg
                                if ( freespace->size == 0 ) {

                                        rc = remove_gpt_segment_from_list( ld->parent_objects, freespace );
                                        if (rc==0) {
                                                free_gpt_disk_segment(freespace);
                                        }
                                        else {   // error ... backout the expand
                                                LOG_ERROR("error, unable to remove the freespace segment from the disk list\n");
                                                seg->size = old_seg_size;
                                                freespace->start  -= expand_sectors;
                                                freespace->size   += expand_sectors;
                                        }

                                }

                        }
                }
        }

        if (!rc) {
                disk_pdata->flags |= DISK_HAS_CHANGES_PENDING;
        }


        LOG_EXIT_INT(rc);
        return rc;
}


static void get_shrink_options( option_array_t * options, sector_count_t * size)
{
        int i;

        LOG_ENTRY();

        for (i = 0; i < options->count; i++) {

                if (options->option[i].is_number_based) {

                        if (options->option[i].number == SEG_SHRINK_OPTION_SIZE_INDEX) {
                                *size = options->option[i].value.ui64;
                        }

                }
                else {

                        if (strcmp(options->option[i].name, SEG_SHRINK_OPTION_SIZE_NAME) == 0) {
                                *size = options->option[i].value.ui64;
                        }
                }

        }

        LOG_EXIT_VOID();
}


/*
 *  Called to shrink a data segment to new_size or next smaller increment.
 *  Then, update appropriate fields in the segment and make
 *  changes to freespace segments as needed.
 *
 */
static int GPT_Shrink( storage_object_t * seg,
                       storage_object_t * shrink_object,
                       list_anchor_t            objects,
                       option_array_t   * options )
{
        int               rc = EINVAL;
        sector_count_t    shrink_sector_count=0;
        u_int64_t         end_lba;
        LOGICALDISK      *ld=NULL;
        sector_count_t    new_seg_size=0;
        SEG_PRIVATE_DATA *pdata=NULL;
        sector_count_t    cylinder_size=0;
        DISK_PRIVATE_DATA *disk_pdata=NULL;

        LOG_ENTRY();

        // initial checks to see if we can do the shrink
        if ( ( seg ) &&
             ( seg == shrink_object ) &&
             ( seg->object_type == SEGMENT ) &&
             ( seg->data_type   == DATA_TYPE ) &&
             ( i_can_modify( seg ) == TRUE ) ) {

                pdata     = (SEG_PRIVATE_DATA *)seg->private_data;
                ld        = get_logical_disk(seg);
                disk_pdata = get_gpt_disk_private_data(ld);

                get_shrink_options( options,  &shrink_sector_count);


                if ( (ld && disk_pdata) &&
                     (shrink_sector_count > 0) &&
                     (shrink_sector_count < seg->size ) ) {

                        LOG_DEBUG("     Data Seg  Name: %s\n",   seg->name);
                        LOG_DEBUG("              Start: %"PRIu64"\n", seg->start);
                        LOG_DEBUG("               Size: %"PRIu64"\n", seg->size);
                        LOG_DEBUG("Shrink Sector Count: %"PRIu64"\n", shrink_sector_count );

                        cylinder_size = get_cylinder_size(ld);

                        // we shrink in cylinder size chunks
                        if (shrink_sector_count < cylinder_size) {
                                shrink_sector_count = cylinder_size;
                        }
                        else {
                                shrink_sector_count = (shrink_sector_count/cylinder_size)*cylinder_size;
                        }

			// ask the engine if it's ok to shrink by this amount.
			rc = EngFncs->can_shrink_by(seg, &shrink_sector_count);
			if (rc) {
				LOG_ERROR("Shrink of segment %s rejected by "
					  "the engine.\n", seg->name);
				LOG_EXIT_INT(rc);
				return rc;
			}

                        // resulting seg size
                        new_seg_size = seg->size - shrink_sector_count;

                        // make sure it ends on cylinder boundary
                        end_lba = seg->start + new_seg_size - 1;
                        if (ends_on_cylinder_boundary(ld, end_lba)==FALSE) {
                                end_lba = rounddown_to_cylinder_boundary( ld, end_lba ) - 1;
                        }

                        if ( end_lba >= (seg->start + seg->size - 1) ) {
                                end_lba = rounddown_to_cylinder_boundary(ld, end_lba ) - 1;
                        }

                        // final test if we can shrink the segment
                        if (  ( end_lba > seg->start ) &&
                              ( end_lba < (seg->start+seg->size-1)) ) {

                                // actual new seg size
                                new_seg_size = end_lba - seg->start + 1;

                                // shrink the data segment
                                seg->size = new_seg_size;

                                // expose freespace extent on disk
                                find_freespace_on_gpt_disk( ld );

                                // mark seg dirty for commit and needs DM activate
				seg->flags |= SOFLAG_DIRTY;
				if (seg->flags & SOFLAG_ACTIVE) {
					seg->flags |= SOFLAG_NEEDS_ACTIVATE;
				}

                                // success.
                                rc = 0;

                        }

                }
                else {
                        LOG_ERROR("error, something wrong with shrink sector count, cant shrink segment\n");
                        rc = EINVAL;
                }

        }
        else {
                LOG_ERROR("error, something wrong with parms\n");
                rc = EINVAL;
        }

        if (!rc) {
                disk_pdata->flags |= DISK_HAS_CHANGES_PENDING;
        }


        LOG_EXIT_INT(rc);
        return rc;
}


static int GPT_AddSectorsToKillList( storage_object_t *seg, lsn_t lsn, sector_count_t count)
{
        int                         rc = EINVAL;
        struct plugin_functions_s  *funcs=NULL;
        LOGICALDISK                *ld=NULL;

        LOG_ENTRY();

        if ( i_can_modify( seg ) == TRUE ) {

                if ( lsn + count > seg->size ) {
                        rc = EINVAL;
                }
                else {

                        ld = get_logical_disk( seg );

                        if (ld) {

                                funcs = (struct plugin_functions_s *)ld->plugin->functions.plugin;

                                rc   = funcs->add_sectors_to_kill_list( ld, seg->start+lsn, count);
                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


static int GPT_CommitChanges( storage_object_t *seg, uint phase )
{
        int               rc = 0;
        int               rc2 = 0;
        LOGICALDISK  *ld=NULL;
        DISKSEG      *clean_seg=NULL;
        SEG_PRIVATE_DATA  *pdata = (SEG_PRIVATE_DATA *) seg->private_data;
        DISK_PRIVATE_DATA *disk_pdata=NULL;
        list_element_t   iter;

        LOG_ENTRY();
        LOG_DEBUG("object= %s  commit phase= %d\n", seg->name, phase );

        // get gpt child storage object
        ld = get_logical_disk(seg);

        // commit changes if we own the segments on the disk
        if ( ld ) {

                disk_pdata = get_gpt_disk_private_data(ld);

                if (phase == MOVE) {
                        if ( disk_pdata->flags & DISK_HAS_MOVE_PENDING ) {
                                LOG_DEBUG("committing move on the disk\n");
                                rc = gpt_move_segment_commit(seg, pdata->move_target, disk_pdata->copy_job );
                                if (disk_pdata->copy_job) free( disk_pdata->copy_job );
                                disk_pdata->copy_job = NULL;
                                disk_pdata->flags   &= ~DISK_HAS_MOVE_PENDING;
                        }
                        else {
                                rc = 0;
                        }
                }
                else if ( phase==FIRST_METADATA_WRITE || phase==SECOND_METADATA_WRITE ) {

                        if ( seg->flags & SOFLAG_DIRTY &&
                             disk_pdata->flags & DISK_HAS_CHANGES_PENDING &&
                             !(disk_pdata->flags & DISK_HAS_MOVE_PENDING) ) {

                                rc = commit_guid_partition_tables( ld, seg, 1, FALSE );

                                if ( rc == 0 ) {

                                        rc2 = commit_guid_partition_tables( ld, seg,
                                                                            2, FALSE );  // dont care ... discovery will fixup
                                                                                         // any problems with 2nd copy of metadata

                                        // mark all seg storage objects clean on this disk
                                        LIST_FOR_EACH( ld->parent_objects, iter, clean_seg ) {
                                                        clean_seg->flags &= ~SOFLAG_DIRTY;
                                        }

                                }
                        }
                        else {
                                rc = 0;
                        }

                }
                else {
                        rc=0;
                }
        }
        else {
                rc = EINVAL;
        }


        LOG_EXIT_INT(rc);
        return rc;
}


static int GPT_Read( storage_object_t *seg,
                     lsn_t             lsn,
                     sector_count_t    count,
                     void             *buffer )
{
        int rc = ENODEV;
        struct plugin_functions_s  *funcs=NULL;
        LOGICALDISK                *ld=NULL;

        LOG_ENTRY();

        if ( i_can_modify( seg ) == TRUE ) {

                if ( lsn + count > seg->size ) {
                        rc = EINVAL;
                }
                else {

                        ld = get_logical_disk( seg );
                        if (ld) {

                                funcs = (struct plugin_functions_s *)ld->plugin->functions.plugin;

                                rc = funcs->read( ld, lsn + seg->start, count, buffer);

                        }

                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}

static int GPT_Write( storage_object_t *seg,
                      lsn_t             lsn,
                      sector_count_t    count,
                      void             *buffer )
{
        int rc = ENODEV;
        struct plugin_functions_s  *funcs=NULL;
        LOGICALDISK                *ld=NULL;

        LOG_ENTRY();

        if ( i_can_modify( seg ) == TRUE ) {

                if ( lsn + count > seg->size ) {
                        rc = EINVAL;
                }
                else {

                        ld = get_logical_disk( seg );
                        if (ld) {

                                funcs = (struct plugin_functions_s *)ld->plugin->functions.plugin;

                                rc = funcs->write( ld, lsn + seg->start, count, buffer);

                        }

                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * This call notifies you that your object is being made into (or part of)
 * a volume or that your object is no longer part of a volume.  The "flag"
 * parameter indicates whether the volume is being created (TRUE) or
 * removed (FALSE).
 */
static void GPT_set_volume(storage_object_t * object, boolean flag)
{
        return;
}



/*
 *  Called to return information about a segment object. If the segment
 *  is a metadata object, the user can ask for extended information about
 *  the metadata and we will display the gpt header info.
 */
static int GPT_get_info( storage_object_t *object, char * name, extended_info_array_t   ** info_array )
{
        int rc = EINVAL;

        LOG_ENTRY();

        if ( object!=NULL && info_array != NULL) {

                if (object->object_type==SEGMENT) {

                        if ( name==NULL ) {
                                rc = GPT_GetInfo(object, info_array);
                        }
                        else if ( ( strcmp(name, "Type")==0 ) && (object->data_type==META_DATA_TYPE) ) {
                                rc = GPT_GetMetadataInfo(object, info_array);
                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Execute the private action on the object.
 */
static  int GPT_plugin_function( storage_object_t * object,
                                 task_action_t      action,
                                 list_anchor_t            objects,
                                 option_array_t   * options)
{
        int rc=EINVAL;
        DISKSEG *freespace=NULL;

        LOG_ENTRY();

        switch (action) {

        case EVMS_Task_GPT_Move_Segment:

                if ( EngFncs->list_count(objects) == 1) {
                        freespace = EngFncs->first_thing(objects,NULL);
                        if (freespace) {
                                rc = gpt_move_segment(object,freespace);
                        }
                }
                break;


        default:
                rc = ENOSYS;
                break;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Return an array of private actions that you support for this object.
 */
static int GPT_get_plugin_functions( storage_object_t        * object,
                                     function_info_array_t * * actions)
{
        int                    rc = EINVAL;
        function_info_array_t *func_info=NULL;

        LOG_ENTRY();


        func_info = EngFncs->engine_alloc( sizeof(function_info_array_t) + sizeof(function_info_t) );
        if (func_info) {

                func_info->count = 0;

                rc = gpt_can_move_segment( object );
                if (!rc) {

                        func_info->count = 1;

                        func_info->info[SEG_MOVE_OPTION_INDEX].function = EVMS_Task_GPT_Move_Segment;

                        func_info->info[SEG_MOVE_OPTION_INDEX].title = EngFncs->engine_strdup( "Move" );
                        func_info->info[SEG_MOVE_OPTION_INDEX].verb = EngFncs->engine_strdup( _("Move") );
                        func_info->info[SEG_MOVE_OPTION_INDEX].name = EngFncs->engine_strdup( _("Move") );
                        func_info->info[SEG_MOVE_OPTION_INDEX].help = EngFncs->engine_strdup( _("Use this function to move a data segment.") );

                }

                rc = 0;
        }
        else {
                rc = ENOMEM;
        }

        *actions = func_info;

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Save the metadata needed to build this segment. Call the Engine's
 * save_metadata() service for each contiguous chunk of metadata that
 * we need to write to the underlying disk.
 */
static int GPT_backup_metadata(storage_object_t *object)
{
	LOGICALDISK *ld;
	int rc = 0;

	LOG_ENTRY();

	if (object->data_type != FREE_SPACE_TYPE) {
                ld = get_logical_disk(object);
                if (ld) {
                        rc = commit_guid_partition_tables(ld, object, 1, TRUE);
                        if (!rc) {
                                rc = commit_guid_partition_tables(ld, object,
                                                                  2, TRUE);
                        }
                }
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                              PLUGIN FUNCTION TABLE                                   +
+                                                                                      +
+--------------------------------------------------------------------------------------*/
static struct plugin_functions_s sft={

        // the following routines are found above
        setup_evms_plugin:                   GPT_SetupEVMSPlugin,
        cleanup_evms_plugin:                 GPT_Cleanup,
        can_set_volume:                      GPT_can_set_volume,
        can_delete:                          GPT_CanDestroy,
        can_expand:                          GPT_CanExpand,
        can_expand_by:                       GPT_CanExpandBy,
        can_shrink:                          GPT_CanShrink,
        can_shrink_by:                       GPT_CanShrinkBy,
        discover:                            GPT_Discover,
        assign:                              GPT_Assign,
        can_unassign:                        GPT_CanUnassign,
        unassign:                            GPT_UnAssign,
        create:                              GPT_CreateSegment,
        delete:                              GPT_DestroySegment,
        discard:                             GPT_Discard,
        expand:                              GPT_Expand,
        shrink:                              GPT_Shrink,
        add_sectors_to_kill_list:            GPT_AddSectorsToKillList,
        commit_changes:                      GPT_CommitChanges,
        read:                                GPT_Read,
        write:                               GPT_Write,
        set_volume:                          GPT_set_volume,
        can_activate:                        GPT_can_activate,
        activate:                            GPT_activate,
        can_deactivate:                      GPT_can_deactivate,
        deactivate:                          GPT_deactivate,
        get_plugin_functions:                GPT_get_plugin_functions,
        plugin_function:                     GPT_plugin_function,
        backup_metadata:                     GPT_backup_metadata,

        // the following routines found in options.c
        get_option_count:                    GPT_GetOptionCount,
        init_task:                           GPT_InitTask,
        set_option:                          GPT_SetOption,
        set_objects:                         GPT_SetObjects,
        get_info:                            GPT_get_info,
        get_plugin_info:                     GPT_GetPluginInfo
};

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                       BUILD AND EXPORT AN EVMS PLUGIN RECORD                         +
+                                                                                      +
+--------------------------------------------------------------------------------------*/

static plugin_record_t gpt_plugin_record = {

        id:                                 EVMS_GPT_PLUGIN_ID,

        version:                            {MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL},

        required_engine_api_version:        {15,0,0},
        required_plugin_api_version:        {plugin: {13,1,0}},

        short_name:                         EVMS_GPT_PLUGIN_SHORT_NAME,
        long_name:                          EVMS_GPT_PLUGIN_LONG_NAME,
        oem_name:                           EVMS_IBM_OEM_NAME,

        functions:                          {plugin: &sft},

        container_functions:                NULL

};

// Vector of plugin record ptrs that we export for the EVMS Engine.
plugin_record_t *evms_plugin_records[] = {
        &gpt_plugin_record,
        NULL
};


