/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: dm-ioctl3.c
 *
 * Routines specific to version 3 of the Device-Mapper ioctl interface.
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>

#include "fullengine.h"
#include "engine.h"
#include "memman.h"
#include "dm.h"
#include "dm-ioctl.h"
#include "dm-ioctl3.h"
#include "remote.h"

#define ALIGNMENT		sizeof(u_int64_t)
#define IOCTL_MIN_LENGTH	16 * 1024

static inline void *align(void *ptr)
{
	register unsigned long a = ALIGNMENT - 1;
	return (void *)(((unsigned long)ptr + a) & ~a);
}

/**
 * add_ioctl_target
 *
 * @target:	The target information to add to the ioctl.
 * @begin:	The location to put the target info.
 * @end:	The end of the ioctl packet, for boundary checking.
 *
 * Add information about a single target to the end of an ioctl packet.
 **/
static void *add_ioctl_target(dm_target_t *target,
			      void *begin,
			      void *end)
{
	dm_ioctl_target_t i_target;
	void *params = begin;
	unsigned long params_length = strlen(target->params);

	LOG_PROC_ENTRY();

	params += sizeof(dm_ioctl_target_t);
	if (params + params_length + 1 >= end) {
		/* Ran off the end of the ioctl packet. */
		LOG_PROC_EXIT_PTR(NULL);
		return NULL;
	}

	/* Initialize the ioctl target. */
	i_target.start = target->start;
	i_target.length = target->length;
	i_target.status = 0;
	strncpy(i_target.target_type, dm_target_type_info[target->type].name,
		sizeof(i_target.target_type));

	/* Copy the parameter string. */
	strcpy((char *)params, target->params);
	params += params_length + 1;
	params = align(params);

	i_target.next = params - begin;
	memcpy(begin, &i_target, sizeof(dm_ioctl_target_t));

	LOG_PROC_EXIT_PTR(params);
	return params;
}

/**
 * build_target_list
 *
 * @dmi: Pointer to ioctl packet.
 *
 * Convert an ioctl packet to a list of targets.
 **/
static dm_target_t *build_target_list(dm_ioctl_t *dmi)
{
	dm_target_t *target, *target_list = NULL;
	dm_ioctl_target_t *i_target;
	dm_target_type type;
	char *data_start = (char*)dmi + dmi->data_start;
	unsigned int num_devs, num_groups;
	char *params;
	int i, rc;

	LOG_PROC_ENTRY();

	i_target = (dm_ioctl_target_t*)data_start;
	for (i = 0; i < dmi->target_count; i++) {
		params = (char*)i_target + sizeof(dm_ioctl_target_t);
		num_devs = 0;
		num_groups = 0;

		/* Determine the target type. */
		for (type = 0; type <= DM_TARGET_MAX; type++) {
			if (!strncmp(i_target->target_type,
				     dm_target_type_info[type].name,
				     DM_MAX_TYPE_NAME)) {
				break;
			}
		}
		if (type > DM_TARGET_MAX) {
			/* Invalid target identifier. */
			LOG_ERROR("Invalid target type (%d) in ioctl packet.\n",
				  type);
			goto error;
		}

		/* Get the number of devs and groups from the target type. */
		rc = dm_target_type_info[type].pretranslate_params(params,
								   &num_devs,
								   &num_groups);
		if (rc) {
			LOG_ERROR("Error getting number of devices and "
				  "groups from the target type.\n");
			goto error;
		}
		
		/* Allocate a target structure. */
		target = dm_allocate_target(type, i_target->start,
					    i_target->length,
					    num_devs, num_groups);
		if (!target) {
			LOG_ERROR("Error allocating target for type \"%s\"\n",
				  dm_target_type_info[type].name);
			goto error;
		}
		target->params = params;

		rc = dm_target_type_info[type].translate_params(target);
		if (rc) {
			LOG_ERROR("Invalid parameter string for target type "
				  "\"%s\"\n", dm_target_type_info[type].name);
			LOG_ERROR("   Returned parameter string is: %s\n",
				  params);
			goto error;
		}

		dm_add_target(target, &target_list);

		i_target = (dm_ioctl_target_t*)(data_start + i_target->next);
	}

	/* Need to NULL the params fields in each target. */
	for (target = target_list; target; target = target->next) {
		target->params = NULL;
	}

	LOG_PROC_EXIT_PTR(target_list);
	return target_list;

error:
	LOG_ERROR("Error building target list. Name = %s\n", dmi->name);
	dm_deallocate_targets(target_list);
	LOG_PROC_EXIT_PTR(NULL);
	return NULL;
}

/**
 * build_ioctl_packet
 *
 * @name:	Name of device
 * @target_list:List of targets to include with this packet.
 * @new_name:	New device name. Don't specify both target_list and new_name.
 *
 * Construct a DM ioctl packet with the given information. Each DM ioctl
 * packet starts with a header (dm_ioctl_t), followed immediately by padding
 * to ensure 64-bit alignment of the remaining data. After the header and
 * the padding can be one of two things:
 * 1) An array of targets. Each target begins with a dm_ioctl_target_t
 *    structure, and is followed immediately by an ASCII string representing
 *    the target-type-specific data. After this string is enough padding to
 *    ensure 64-bit alignment of the next target.
 * 2) A new device name, as an ASCII string.
 *
 * This function returns a pointer to the beginning of the constructed
 * packet. The caller is responsible for filling in extra information
 * in the ioctl header (flags and dev) based on the command being sent.
 **/
static dm_ioctl_t *build_ioctl_packet(unsigned char *name,
				      dm_target_t *target_list,
				      unsigned char *new_name)
{
	dm_ioctl_t *dmi;
	dm_target_t *target;
	void *begin, *end;
	unsigned long packet_length = sizeof(dm_ioctl_t) + ALIGNMENT;
	int target_count = 0;

	LOG_PROC_ENTRY();

	/* Can't specify both target_list and new_name. */
	if (target_list != NULL && new_name != NULL) {
		LOG_PROC_EXIT_PTR(NULL);
		return NULL;
	}

	/* Count the number of targets. */
	for (target = target_list; target; target = target->next) {
		packet_length += sizeof(dm_ioctl_target_t);
		packet_length += strlen(target->params) + 1 + ALIGNMENT;
		target_count++;
	}

	if (new_name) {
		packet_length += strlen(new_name) + 1;
	}

	/* Make sure packet is minimum length, in case target information
	 * needs to be returned from the kernel.
	 */
	if (packet_length < IOCTL_MIN_LENGTH) {
		packet_length = IOCTL_MIN_LENGTH;
	}

	/* Allocate the ioctl packet. */
	dmi = engine_alloc(packet_length);
	if (!dmi) {
		LOG_ERROR("Error allocating memory for ioctl packet. ");
		LOG_ERROR("Name = %s\n", name);
		LOG_PROC_EXIT_PTR(NULL);
		return NULL;
	}

	/* Initialize the ioctl header. */
	dmi->version[0] = DM_VERSION_MAJOR;
	dmi->version[1] = DM_VERSION_MINOR;
	dmi->version[2] = DM_VERSION_PATCHLEVEL;
	dmi->data_size = packet_length;
	dmi->data_start = (u_int32_t)(unsigned long)align((void *)sizeof(dm_ioctl_t));
	dmi->target_count = target_count;
	if (name) {
		strncpy(dmi->name, name, sizeof(dmi->name));
	}

	/* Add extra information after the ioctl header. */
	begin = (void *)((char *)dmi + dmi->data_start);
	end = (void *)((char *)dmi + dmi->data_size);

	/* Add each target to the packet. */
	for (target = target_list; target; target = target->next) {
		begin = add_ioctl_target(target, begin, end);
		if (!begin) {
			engine_free(dmi);
			LOG_PROC_EXIT_PTR(NULL);
			return NULL;
		}
	}

	/* Add the new name to the packet. */
	if (new_name) {
		strcpy(begin, new_name);
	}

	LOG_PROC_EXIT_PTR(dmi);
	return dmi;
}

/**
 * run_command_v3
 *
 * Execute the ioctl command for version-3 of DM.
 **/
int run_command_v3(void *dmi_in, unsigned long command)
{
	dm_ioctl_t *dmi = dmi_in;
	int rc = 0;

	LOG_PROC_ENTRY();
	LOG_DEBUG("Issuing DM ioctl %ld for device %s.\n",
		  _IOC_NR(command), dmi->name);

	if (dm_control_fd != 0) {
		rc = ioctl(dm_control_fd, command, dmi);
		if (rc) {
			rc = errno;
			LOG_ERROR("Error returned from ioctl call: "
				  "%d: %s.\n", rc, strerror(rc));
		}
	} else {
		LOG_WARNING("Device-Mapper control file not open.\n");
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_get_version_v3
 *
 * Query Device-Mapper for the current version of the ioctl interface. This
 * uses the version-3 of the DM ioctl interface.
 **/
int dm_get_version_v3(int *major, int *minor, int *patch)
{
	dm_ioctl_t *dmi;
	int rc;

	LOG_PROC_ENTRY();

	dmi = build_ioctl_packet(NULL, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	/* Run the command. */
	rc = run_command_v3(dmi, DM_VERSION);
	if (rc) {
		goto out;
	}

	*major = dmi->version[0];
	*minor = dmi->version[1];
	*patch = dmi->version[2];
	rc = 0;

out:
	engine_free(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_suspend_v3
 *
 * @name:	Name of device to be suspended.
 * @node:	Cluster-node that owns this device.
 * @suspend:	TRUE for suspend. FALSE for resume.
 *
 * Suspend or resume the Device-Mapper device representing an active EVMS
 * object or volume. This works with version-3 of the DM ioctl interface.
 **/
int dm_suspend_v3(char *name, int suspend)
{
	dm_ioctl_t *dmi;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(name, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	/* Suspend and resume are the same ioctl, with different flags. */
	if (suspend) {
		dmi->flags |= DM_SUSPEND_FLAG;
	}

	/* Run the command. */
	rc = run_command_v3(dmi, DM_DEV_SUSPEND);

out:
	engine_free(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_activate_v3
 *
 * @name:	Name of device to be (re)activated.
 * @node:	Cluster-node that owns this device.
 * @target_list:List of targets that will make up the device's mapping.
 * @reactivate:	TRUE to reactivate an already-active device.
 * @read_only:	TRUE to make the device read-only.
 * @dev_major:	Return the device major number.
 * @dev_minor:	Return the device minor number.
 *
 * Activate (or re-activate) an EVMS object or volume as a Device-Mapper device.
 * This works with version-3 of the DM ioctl interface.
 **/
int dm_activate_v3(char *name,
		   dm_target_t *target_list,
		   int reactivate,
		   int read_only,
		   u_int32_t *dev_major,
		   u_int32_t *dev_minor)
{
	dm_ioctl_t *dmi;
	unsigned long command;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(name, target_list, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	if (read_only) {
		dmi->flags |= DM_READONLY_FLAG;
	}

	/* Run the command. */
	if (reactivate) {
		command = DM_DEV_RELOAD;
		dmi->flags |= DM_SUSPEND_FLAG;
	} else {
		command = DM_DEV_CREATE;
	}

	rc = run_command_v3(dmi, command);
	if (rc) {
		goto out;
	}

	/* On an activate, update the object's information */
	if (!reactivate) {
		*dev_major = major(dmi->dev);
		*dev_minor = minor(dmi->dev);
	}
	
out:
	engine_free(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_deactivate_v3
 *
 * @name: Name of device to deactivate.
 * @node: Cluster-node that owns this device.
 *
 * Deactivate the Device-Mapper device representing an EVMS storage object
 * or volume.
 **/
int dm_deactivate_v3(char *name)
{
	dm_ioctl_t *dmi;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(name, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	/* Run the command. */
	rc = run_command_v3(dmi, DM_DEV_REMOVE);

out:
	engine_free(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_rename_v3
 *
 * @old_name:	The device's old name.
 * @new_name:	The device's new name.
 * @node:	Cluster-node that owns this device.
 *
 * Tell Device-Mapper to change the name for an active object or volume.
 **/
int dm_rename_v3(char *old_name, char *new_name)
{
	dm_ioctl_t *dmi;
	int rc;

	LOG_PROC_ENTRY();

	dmi = build_ioctl_packet(old_name, NULL, new_name);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	/* Run the command. */
	rc = run_command_v3(dmi, DM_DEV_RENAME);

out:
	engine_free(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_update_status_v3
 *
 * @name:	Name of device to get status for.
 * @node:	Cluster-node that owns this device.
 * @active:	Return TRUE if device is active in the kernel.
 * @read_only:	Return TRUE if device is read-only in the kernel.
 * @dev_major:	Return the device's major number.
 * @dev_minor:	Return the device's minor number.
 *
 * Query device-mapper for the status of an EVMS object or volume. If the
 * device is found in DM, return the state as active. If it is active, also
 * return the read-only state and the device major:minor number.
 **/
int dm_update_status_v3(char *name,
			int *active,
			int *read_only,
			u_int32_t *dev_major,
			u_int32_t *dev_minor)
{
	dm_ioctl_t *dmi;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(name, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	/* Run the command. */
	rc = run_command_v3(dmi, DM_DEV_STATUS);
	if (rc) {
		goto out;
	}

	/* Update device's information. */
	if (dmi->flags & DM_EXISTS_FLAG) {
		*active = TRUE;
		*dev_major = major(dmi->dev);
		*dev_minor = minor(dmi->dev);
		*read_only = (dmi->flags & DM_READONLY_FLAG) ? TRUE : FALSE;
	} else {
		*active = FALSE;
		*dev_major = 0;
		*dev_minor = 0;
	}

out:
	engine_free(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_get_targets_v3
 *
 * @name:	Name of device to get mapping for.
 * @node:	Cluster-node that owns this device.
 * @target_list:Return the list of targets.
 *
 * Query device-mapper for the set of targets that comprise an EVMS object or
 * volume. This function sets target_list to point at a list of targets. When
 * the caller is done using this list, it must be freed with a call to
 * dm_deallocate_targets().
 **/
int dm_get_targets_v3(char *name, dm_target_t **target_list)
{
	dm_ioctl_t *dmi;
	dm_target_t *targets = NULL;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(name, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	dmi->flags |= DM_STATUS_TABLE_FLAG;

	/* Run the command. */
	rc = run_command_v3(dmi, DM_TARGET_STATUS);
	if (rc) {
		goto out;
	}

	targets = build_target_list(dmi);
	if (!targets) {
		rc = EINVAL;
	}

out:
	*target_list = targets;
	engine_free(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * dm_get_info_v3
 *
 * @name: Name of device to get info for.
 * @node: Cluster-node that owns this device.
 * @info: Return pointer to info string.
 *
 * Query device-mapper for the latest status info for an EVMS object or volume.
 * This function sets info to point at a newly-allocated string. When the caller
 * is done using that string, it must be freed with a call to the engine_free
 * service. Status info is target-dependent. The caller should know how to
 * interpret the returned string.
 *
 * Currently, this function assumes the desired device has only one target.
 * Thus, only one string will be returned. May change this later to return
 * an array of strings, one for each target in the device. However, since
 * currently only snapshot provides any meaningful info for this call, there
 * is no need for multi-target info.
 **/
int dm_get_info_v3(char *name, char **info)
{
	dm_ioctl_t *dmi;
	char *info_str;
	int rc;

	LOG_PROC_ENTRY();

	/* Build the ioctl packet. */
	dmi = build_ioctl_packet(name, NULL, NULL);
	if (!dmi) {
		rc = ENOMEM;
		goto out;
	}

	/* Run the command. */
	rc = run_command_v3(dmi, DM_TARGET_STATUS);
	if (rc) {
		goto out;
	}

	info_str = (char *)dmi + dmi->data_start + sizeof(dm_ioctl_target_t);
	*info = engine_alloc(strlen(info_str)+1);
	if (!*info) {
		rc = ENOMEM;
		goto out;
	}

	strcpy(*info, info_str);

out:
	engine_free(dmi);
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

