/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * plugins/xfs/fsimxfs.h
 */

#include <uuid/uuid.h>
#include <asm/page.h>

extern plugin_record_t xfs_plugin_record;
extern plugin_record_t * my_plugin_record;
extern engine_functions_t * EngFncs;

/* File system ID ... used by the SetPluginID macro. */
#define FS_TYPE_XFS 11

/* fsck option array indices */
#define FSCK_READONLY_INDEX		0
#define FSCK_VERBOSE_INDEX		1
#define FSCK_XFS_OPTIONS_COUNT		2
#define FSCK_XFS_OPTIONS_COUNT_TOTAL	7

/* mkfs option array indices */
#define MKFS_SETVOL_INDEX		0
#define MKFS_JOURNAL_VOL_INDEX		1
#define MKFS_SETLOGSIZE_INDEX		2
#define MKFS_FORCE_INDEX		3
#define MKFS_XFS_OPTIONS_COUNT		4
#define MKFS_XFS_OPTIONS_COUNT_TOTAL	11

/* fsck exit codes */
#define FSCK_OK				0
#define FSCK_CORRECTED			1
#define FSCK_REBOOT			2
#define FSCK_ERRORS_UNCORRECTED		4
#define FSCK_OP_ERROR			8
#define FSCK_USAGE_ERROR		16

/* xfsutils_support flag values */
#define XFS_VALID_UTILS			0x0001
#define XFS_V2_LOG			0x0002
#define XFS_EXTERNAL_LOG		0x0003
#define LV_NAME_SIZE			12

/* FSIM options defines */
#define NO_SELECTION			"None"

typedef __uint32_t	xfs_agblock_t;	/* blockno in alloc. group */
typedef	__uint32_t	xfs_extlen_t;	/* extent length in blocks */
typedef	__uint32_t	xfs_agnumber_t;	/* allocation group number */
typedef __int32_t	xfs_extnum_t;	/* # of extents in a file */
typedef __int16_t	xfs_aextnum_t;	/* # extents in an attribute fork */
typedef	__int64_t	xfs_fsize_t;	/* bytes in a file */
typedef __uint64_t	xfs_ufsize_t;	/* unsigned bytes in a file */
typedef	__int32_t	xfs_suminfo_t;	/* type of bitmap summary info */
typedef	__int32_t	xfs_rtword_t;	/* word type for bitmap manipulations */
typedef	__int64_t	xfs_lsn_t;	/* log sequence number */
typedef	__int32_t	xfs_tid_t;	/* transaction identifier */
typedef	__uint32_t	xfs_dablk_t;	/* dir/attr block number (in file) */
typedef	__uint32_t	xfs_dahash_t;	/* dir/attr hash value */
typedef __uint16_t	xfs_prid_t;	/* prid_t truncated to 16bits in XFS */

/* These types are 64 bits on disk but are either 32 or 64 bits in memory.
 * Disk based types:
 */
typedef __uint64_t	xfs_dfsbno_t;	/* blockno in filesystem (agno|agbno) */
typedef __uint64_t	xfs_drfsbno_t;	/* blockno in filesystem (raw) */
typedef	__uint64_t	xfs_drtbno_t;	/* extent (block) in realtime area */
typedef	__uint64_t	xfs_dfiloff_t;	/* block number in a file */
typedef	__uint64_t	xfs_dfilblks_t;	/* number of blocks in a file */

/* Fom platform_def.h - whaty they were there, who knows. */
typedef loff_t		xfs_off_t;
typedef __uint64_t	xfs_ino_t;
typedef __uint32_t	xfs_dev_t;
typedef __int64_t	xfs_daddr_t;
typedef char*		xfs_caddr_t;
typedef unsigned char	uchar_t;

#include "xfs_sb.h"

#define	BBSHIFT			9
#define	BBSIZE			(1 << BBSHIFT)
#define	BBMASK			(BBSIZE - 1)
#define XLOG_RECORD_BSIZE	(16*1024)	/* eventually 32k */
#define XLOG_MAX_RECORD_BSIZE	(32*1024)

typedef struct xlog_rec_header {
	uint		h_magicno;	/* log record (LR) identifier		:  4 */
	uint		h_cycle;	/* write cycle of log			:  4 */
	int		h_version;	/* LR version				:  4 */
	int		h_len;		/* len in bytes; should be 64-bit aligned: 4 */
	xfs_lsn_t	h_lsn;		/* lsn of this LR			:  8 */
	xfs_lsn_t	h_tail_lsn;	/* lsn of 1st LR w/ buffers not committed: 8 */
	uint		h_chksum;	/* may not be used; non-zero if used	:  4 */
	int		h_prev_block;	/* block number to previous LR		:  4 */
	int		h_num_logops;	/* number of log operations in this LR	:  4 */
	uint		h_cycle_data[XLOG_MAX_RECORD_BSIZE / BBSIZE];
        /* new fields */
        int		h_fmt;		/* format of log record                 :  4 */
        uuid_t		h_fs_uuid;	/* uuid of FS                           : 16 */
} xlog_rec_header_t;

#define XLOG_HEADER_MAGIC_NUM	0xFEEDbabe	/* Illegal cycle number */

typedef struct xfs_volume {
	xfs_sb_t * sb;			/* sb if this is fs vol */
	xlog_rec_header_t * log_sb;	/* log rec if this is external log */
	logical_volume_t * log_vol;	/* external log vol if fs vol and there is one */
	logical_volume_t * fs_vol;	/* fs vol that owns this external log vol */
} xfs_volume_t;

/* generic defines */
#define GET                     0
#define PUT                     1

/* Macros for referencing fields in disk structures.
 * XFS writes all disk structures in big endian format.  These macros can
 * be used to access the fields of structures on disk regardless of the
 * endianness of the CPU architecture.
 */
#if __BYTE_ORDER == __LITTLE_ENDIAN
#define XCPU_TO_DISK16(x)	(bswap_16(x))
#define XCPU_TO_DISK32(x)	(bswap_32(x))
#define XCPU_TO_DISK64(x)	(bswap_64(x))

#define XDISK_TO_CPU16(x)	(bswap_16(x))
#define XDISK_TO_CPU32(x)	(bswap_32(x))
#define XDISK_TO_CPU64(x)	(bswap_64(x))

#elif __BYTE_ORDER == __BIG_ENDIAN
#define XCPU_TO_DISK16(x)	(x)
#define XCPU_TO_DISK32(x)	(x)
#define XCPU_TO_DISK64(x)	(x)

#define XDISK_TO_CPU16(x)	(x)
#define XDISK_TO_CPU32(x)	(x)
#define XDISK_TO_CPU64(x)	(x)

#else
#error "__BYTE_ORDER must be defined as __LITTLE_ENDIAN or __BIG_ENDIAN"
#endif


/* Minimum number of bytes supported for a XFS partition */
#define MINXFS			(16 * 1024 * 1024)  // 16 MB
#define XFS_MIN_DATA_SECTORS	3200  // 400 blocks * 8 sectors/block
#define XFS_DEFAULT_LOG_SIZE	4     // in MB = 1000 4k blocks
#define XFS_MIN_LOG_SIZE	2     // in MB = 512 4k blocks
#define XFS_MAX_LOG_SIZE	128   // in MB = 64k 4k blocks

/* SIZE_OF_SUPER defines the total amount of space reserved on disk for the
 * superblock.  This is not the same as the superblock structure, since all of
 * this space is not currently being used.
 */
#define SIZE_OF_SUPER		PAGE_SIZE

/* Fixed byte offset of primary superblock */
#define XFS_SUPER1_OFF		0x00

extern int xfs_test_version(void);
extern int xfs_get_superblock(logical_volume_t * volume, xfs_sb_t * sb);
extern int xfs_get_log_superblock(logical_volume_t * volume, xlog_rec_header_t * log_sb);
extern int xfs_create(logical_volume_t * volume, option_array_t * options);
extern int xfs_remove(logical_volume_t * volume);
extern int xfs_check(logical_volume_t * volume, option_array_t * options);

