/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: md_dlist.c
 *
 * Description: This file contains all functions related to manipulating the
 *              dlists that are used by the MD region manager.
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <plugin.h>

#include "md.h"

#define my_plugin_record my_plugin

/* Function: md_append_region_to_object
 *
 *	Associate the specified region and object as parent/child. Add the
 *	region to the object's "parent" dlist, and add the object to the
 *	region's "child" dlist. Also need to check for duplicates, so the
 *	same region/object don't get associated more than once.
 */
 int md_append_region_to_object(storage_object_t	* region,
				storage_object_t	* object )
{
	list_element_t li;

	LOG_ENTRY();

	li = EngFncs->insert_thing(object->parent_objects,
				   region,
				   INSERT_AFTER | EXCLUSIVE_INSERT,
				   NULL);

	if (!li) {
		/*
		 * There are 2 possible failures here:
		 * no memory or exclusive insert failed
		 */
		LOG_SERIOUS("Error adding region %s as a parent to object %s\n",
			region->name, object->name);
		LOG_EXIT_INT(EPERM);
		return EPERM;
	}

	li = EngFncs->insert_thing(region->child_objects,
				   object,
				   INSERT_AFTER | EXCLUSIVE_INSERT,
				   NULL);

	if (!li) {
		/*
		 * There are 2 possible failures here:
		 * no memory or exclusive insert failed
		 */
		LOG_SERIOUS("Error adding object %s as a child to region %s\n",
			object->name, region->name);
		EngFncs->remove_thing(object->parent_objects, region);
		LOG_EXIT_INT(EPERM);
		return EPERM;
	} else {
		object->volume = region->volume;
		region->geometry.bytes_per_sector = max(region->geometry.bytes_per_sector, object->geometry.bytes_per_sector);
	}

	LOG_EXIT_INT(0);
	return 0;
}


/* Function: md_remove_region_from_object
 *
 *	Remove the parent/child association between the specified region
 *	and object. Remove the region from the object's "parent" list,
 *	and remove the object from the region's "child" list.
 */
void md_remove_region_from_object(	storage_object_t	* region,
					storage_object_t	* object )
{
	LOG_ENTRY();

	EngFncs->remove_thing(object->parent_objects, region);
	EngFncs->remove_thing(region->child_objects, object);
	object->volume = NULL; // null out the volume pointer.

	LOG_EXIT_VOID();
}


/* Function: md_clear_child_list
 *
 *	Remove all objects from this region's child list.
 */
void md_clear_child_list( storage_object_t * region , list_anchor_t return_list)
{
	storage_object_t	* object = NULL;
	list_element_t iter1, iter2;

	LOG_ENTRY();

	LIST_FOR_EACH_SAFE(region->child_objects, iter1, iter2, object) {
		if ( object ) {
			md_remove_region_from_object(region, object);
			if (return_list) {
				if (md_add_object_to_list(object, return_list)) {
					LOG_SERIOUS("Could not add all objects to the return list for%s\n", region->name);
				}
			}
		}
	}

	LOG_EXIT_VOID();
}



/* Function: md_add_object_to_list
 *
 *	Add the specified object to the specified list.
 */
int md_add_object_to_list(	storage_object_t	* object,
				list_anchor_t objects )
{
	int rc=0;
	list_element_t li = NULL;
	
	LOG_ENTRY();

	li = EngFncs->insert_thing(objects, object, INSERT_AFTER, NULL);

	if (!li) {
		LOG_SERIOUS("Error adding object %s to output list\n", object->name);
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: md_transfer_lists
 *
 *	This function simply removes each item from the input list and adds it
 *	to the output list.
 */
int md_transfer_list(	list_anchor_t	input,
			list_anchor_t	output )
{
	int rc;

	LOG_ENTRY();

	rc = EngFncs->merge_lists(output, input, NULL);

	LOG_EXIT_INT(rc);
	return rc;
}


void md_add_volume_to_list(md_volume_t * volume)
{
	LOG_ENTRY();
	volume->next = volume_list_head;
	volume_list_head = volume;
	LOG_EXIT_VOID();
}

void md_remove_volume_from_list(md_volume_t * volume)
{
	md_volume_t * tmp;
	LOG_ENTRY();
	if (volume == volume_list_head) {
		volume_list_head = volume_list_head->next;
	}else{
		for (tmp = volume_list_head; tmp != NULL; tmp = tmp->next) {
			if (volume == tmp->next) {
				tmp->next = tmp->next->next;
			}
		}
	}
	LOG_EXIT_VOID();
}


void empty_setup_funcs_queue(md_volume_t *vol)
{
	md_setup_func_t *setup;
	list_element_t iter1, iter2;
	
	LIST_FOR_EACH_SAFE(vol->setup_funcs,iter1,iter2,setup) {
		if (setup->setup_func) {
			setup->proceed = FALSE;
			setup->setup_func(vol, setup);
		}
		EngFncs->engine_free(setup);
		EngFncs->remove_element(iter1);
	}
}

int schedule_setup_func(md_volume_t *vol,
			evms_md_disk_info_t *disk_info,
			setup_func_t *setup_func)
{
	int rc=0;
	md_setup_func_t *setup;
	list_element_t li;
	
	if (!vol) {
		md_log_internal_bug(__FILE__, __FUNCTION__, __LINE__);
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	
	}

	if (!vol->region) {
		md_log_internal_bug(__FILE__, __FUNCTION__, __LINE__);
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (!setup_func) {
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	setup = EngFncs->engine_alloc(sizeof(md_setup_func_t));
	
	if (!setup) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	setup->disk_info = disk_info;
	setup->setup_func = setup_func;
	
	li = EngFncs->insert_thing(vol->setup_funcs, setup, INSERT_AFTER, NULL);
	if (!li) {
		EngFncs->engine_free(setup);
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int process_setup_funcs(storage_object_t *region)
{
	int rc=0;
	list_element_t iter1, iter2;
	md_setup_func_t *setup;
	md_volume_t * vol;

	LOG_ENTRY();
	
	if (!region) {
		LOG_EXIT_INT(EFAULT);
		return rc;
	}

	vol = (md_volume_t *)region->private_data;
	if (!vol) {
		LOG_EXIT_INT(EFAULT);
		return rc;
	}
	
	LIST_FOR_EACH_SAFE(vol->setup_funcs, iter1, iter2, setup) {
		setup->proceed = TRUE;
		rc = setup->setup_func(vol, setup);
		EngFncs->engine_free(setup);
		EngFncs->remove_element(iter1);
		if (rc)
			break;
	}

	empty_setup_funcs_queue(vol);

	LOG_EXIT_INT(rc);
	return rc;
}


void empty_ioctl_queue(md_volume_t *vol)
{
	md_ioctl_pkg_t *pkg;
	list_element_t iter1, iter2;
	
	LOG_ENTRY();
	if (vol->ioctl_pkgs == NULL) {
		LOG_EXIT_VOID();
		return;
	}

	LIST_FOR_EACH_SAFE(vol->ioctl_pkgs, iter1, iter2, pkg) {

		if (pkg->callback_func) {
			pkg->callback_func(vol, pkg);
		}
		EngFncs->engine_free(pkg);
		EngFncs->remove_element(iter1);
	}
	LOG_EXIT_VOID();
}

void free_ioctl_pkgs(md_volume_t *vol)
{
	md_ioctl_pkg_t *pkg;
	list_element_t iter1, iter2;
	
	LOG_ENTRY();

	if (vol->ioctl_cleanup == NULL) {
		LOG_EXIT_VOID();
		return;
	}

	LIST_FOR_EACH_SAFE(vol->ioctl_cleanup, iter1, iter2, pkg) {
		if (pkg->callback_func) {
			pkg->callback_func(vol, pkg);
		}
		EngFncs->engine_free(pkg);
		EngFncs->remove_element(iter1);
	}

	/* Just in case there are still some IOCTLs */
	empty_ioctl_queue(vol);

	LOG_EXIT_VOID();
}

int schedule_md_ioctl_pkg(md_volume_t *vol,
			  int cmd,
			  evms_md_ioctl_parm_t *parm,
			  md_ioctl_callback_t *callback)
{
	int rc=0;
	md_ioctl_pkg_t *pkg = NULL;
	list_element_t li;


	LOG_ENTRY();

	if (!vol->region) {
		md_log_internal_bug(__FILE__, __FUNCTION__, __LINE__);
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	pkg = EngFncs->engine_alloc(sizeof(md_ioctl_pkg_t));
	
	if (!pkg) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	pkg->cmd = cmd;
	pkg->callback_func = callback;
	switch (pkg->cmd) {
	case EVMS_MD_ADD:
	case EVMS_MD_REMOVE:
	case EVMS_MD_ACTIVATE:
	case EVMS_MD_DEACTIVATE:
		pkg->parm.disk_info = parm->disk_info;
		break;
	case EVMS_MD_ACTIVATE_REGION:
		pkg->parm.sb = parm->sb;
		break;
	case EVMS_MD_DEACTIVATE_REGION:
	case EVMS_MD_INVOKE_CALLBACK:
		break;
	default:
		rc = EINVAL;
	}
	if (rc) {
		goto error_free_mem;
	}
	
	li = EngFncs->insert_thing(vol->ioctl_pkgs, pkg, INSERT_AFTER, NULL);
	if (!li) {
		rc = ENOMEM;
		goto error_free_mem;
	}

	LOG_EXIT_INT(rc);
	return rc;

error_free_mem:
	if (pkg) {
		EngFncs->engine_free(pkg);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

boolean remove_scheduled_md_ioctl_pkg(md_volume_t *vol,
				      int cmd,
				      evms_md_ioctl_parm_t *parm)
{
	boolean removed = FALSE;
	int rc;
	md_ioctl_pkg_t *pkg;
	list_element_t iter1, iter2;

	LOG_ENTRY();

	LIST_FOR_EACH_SAFE(vol->ioctl_pkgs, iter1, iter2, pkg) {

		/* We can only remove ADD, REMOVE, ACTIVATE and DEACTIVATE ioctls */
		if (pkg->cmd == cmd) {
			switch (pkg->cmd) {
			case EVMS_MD_ADD:
			case EVMS_MD_REMOVE:
			case EVMS_MD_ACTIVATE:
			case EVMS_MD_DEACTIVATE:
				if ( (pkg->parm.disk_info->number == parm->disk_info->number) &&
				     (pkg->parm.disk_info->object == parm->disk_info->object) )
					removed = TRUE;
				break;
			case EVMS_MD_ACTIVATE_REGION:
			case EVMS_MD_DEACTIVATE_REGION:
			case EVMS_MD_INVOKE_CALLBACK:
			default:
				rc = EINVAL;
				break;
			}
		}

		if (removed == TRUE) {
			/*
			 * Ask the owner of this ioctl to cancel its operation
			 * invoke callback function if it exists
			 */
			if (pkg->callback_func) {
				int saved_cmd = pkg->cmd;
				pkg->cmd = EVMS_MD_CANCEL_OPERATION;
				rc = pkg->callback_func(vol, pkg);
				if (rc) {
					/*
					 * Found the ioctl to remove,
					 * but the owner refuses to cancel operation.
					 * Break out of do-while loop
					 */
					pkg->cmd = saved_cmd;
					removed = FALSE;
					break;
				}
			} else {
				/*
				 * Found the ioctl to remove,
				 * but there is no callback function,
				 * can't remove IOCTL.
				 * Break out of do-while loop
				 */
				removed = FALSE;
				break;
			}
		}

		if (removed == TRUE) {
			EngFncs->remove_element(iter1);
			/* Done. Break out of do-while loop */
			break;
		}
	}

	LOG_EXIT_INT(removed);
	return removed;
}



int process_md_ioctl_pkgs(storage_object_t *region)
{
	int rc=0;
	list_element_t iter1, iter2;
	md_ioctl_pkg_t *pkg;
	md_volume_t * vol;
	evms_md_disk_info_t *disk_info;
	mdu_disk_info_t tmp_info;


	LOG_ENTRY();
	if (!region || !(vol = (md_volume_t *)region->private_data)) {
		rc = EFAULT;
		LOG_EXIT_INT(rc);
		return rc;
	}

	if (EngFncs->list_count(vol->ioctl_pkgs) == 0) {
		/* nothing to do */
		LOG_EXIT_INT(0);
		return 0;
	}
	
	LIST_FOR_EACH_SAFE(vol->ioctl_pkgs, iter1, iter2, pkg) {
		switch (pkg->cmd) {
		case EVMS_MD_ADD:
			disk_info = pkg->parm.disk_info;

			tmp_info.number = disk_info->number;
			rc = md_ioctl_get_disk_info(region, &tmp_info);
			if (rc) {
				break;
			}
			if (tmp_info.major) {
				if ( (tmp_info.major == disk_info->object->dev_major) &&
				     (tmp_info.minor == disk_info->object->dev_major) &&
				     (tmp_info.state & (1 << MD_DISK_ACTIVE)) ) {
					LOG_WARNING("ADD: Device (%d:%d) already exists in region [%s]\n",
						    disk_info->object->dev_major,
						    disk_info->object->dev_minor,
						    region->name);
					break;
				}
			}
			rc = md_ioctl_hot_add_disk(
				region,
				makedev(disk_info->object->dev_major, disk_info->object->dev_minor));
			break;
		case EVMS_MD_REMOVE:
			disk_info = pkg->parm.disk_info;

			tmp_info.number = disk_info->number;
			rc = md_ioctl_get_disk_info(region, &tmp_info);
			if (rc) {
				break;
			}
			/*
			 * See if the disk to be removed is still at same slot (number)
			 */
			if ((tmp_info.major != disk_info->major) ||
			    (tmp_info.minor != disk_info->minor) ) {
				/*
				 * The disk may has been moved to a different slot.
				 * Find it.
				 */
				disk_info->number = find_disk_in_active_region(region,
									disk_info->major,
									disk_info->minor);
				if (disk_info->number == MAX_MD_DEVICES)
					rc = ENODEV;
			}

			if (!rc) {
				rc = md_ioctl_hot_remove_disk(
					region,
					makedev(disk_info->major, disk_info->minor));
			} else {
				LOG_WARNING("REMOVE: cound not find %s (%d:%d) in region [%s]\n",
					    disk_info->object->name,
					    disk_info->major,
					    disk_info->minor,
					    region->name);
				rc = ENODEV;
				break;
			}
			break;
		case EVMS_MD_ACTIVATE:
			break;
			
		case EVMS_MD_DEACTIVATE:
			disk_info = pkg->parm.disk_info;

			tmp_info.number = disk_info->number;
			rc = md_ioctl_get_disk_info(region, &tmp_info);
			if (rc) {
				break;
			}

			/*
			 * See if the disk to be marked faulty is still at same slot (number)
			 */
			if ((tmp_info.major != disk_info->major) ||
			    (tmp_info.minor != disk_info->minor) ) {
				/*
				 * The disk may has been moved to a different slot.
				 * Find it.
				 */
				disk_info->number = find_disk_in_active_region(region,
									disk_info->major,
									disk_info->minor);
				if (disk_info->number == MAX_MD_DEVICES)
					rc = ENODEV;
			}

			if (!rc) {
				rc = md_ioctl_set_disk_faulty(
					region,
					makedev(disk_info->major, disk_info->minor));
			} else {
				LOG_WARNING("DEACTIVATE: Could not find %s [%d:%d] in region %s.\n",
					    disk_info->object->name,
					    disk_info->major,
					    disk_info->minor,
					    region->name);
				break;
			}
			break;

		case EVMS_MD_ACTIVATE_REGION:
			if (!md_is_region_active(region)) {
				mdp_super_t *tmp_sb;

				tmp_sb = vol->super_block;
				vol->super_block = (mdp_super_t *)pkg->parm.sb;

				rc = md_activate_region(region);

				vol->super_block = tmp_sb;
			}
			break;
		case EVMS_MD_DEACTIVATE_REGION:
			if (md_is_region_active(region)) {
				rc = md_activate_region(region);
			}
			break;

		case EVMS_MD_INVOKE_CALLBACK:
			break;
		default:
			md_log_internal_bug(__FILE__, __FUNCTION__, __LINE__);
			rc = EINVAL;
		}

		pkg->rc = rc;
		/*
		 * If there is an error, invoke callback function.
		 * Otherwise, the callback function will be called later in free_ioctl_pkgs();
		 */
		if (rc && pkg->callback_func) {
			rc = pkg->callback_func(vol, pkg);
		}
		rc = pkg->rc;

		EngFncs->remove_element(iter1);

		/* place this package to the ioctl_cleanup list */
		if (!EngFncs->insert_thing(vol->ioctl_cleanup, pkg, INSERT_AFTER, NULL)) {
			/* insert failed */
			LOG_WARNING("Can't insert to ioctl cleanup list.\n");
			rc = pkg->callback_func(vol, pkg);
			EngFncs->engine_free(pkg);
		}
	}
	
	/* Give the recovery process a chance to start, sleep for a second. */
	sleep(1);
	
	LOG_EXIT_INT(rc);
	return rc;
}


