/*
 *
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libmac.so
 *
 *   File: dm.c
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "mac_plugin.h"



/*
 *  Function:  get_DM_info
 *
 *  Called when a DM activate fails.  The job here is to see if the 
 *  segment is active in the kernel.  If so ... compare the target 
 *  mapping against the engine segment object.  If the segments starting
 *  LBA and SIZE match ... then we dont need to activate the segment
 *  because it is already active and has a correct mapping ... return 0.
 */
static int mac_get_DM_info( DISKSEG *seg )
{       
	dm_target_t *targets=NULL;
	dm_device_t *dev = NULL;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("seg= %s\n", seg->name);

	rc = EngFncs->dm_update_status(seg);
	if (!rc) {
		if (seg->flags & SOFLAG_ACTIVE) {
			LOG_DEBUG("segment IS active in the kernel\n");
			rc = EngFncs->dm_get_targets(seg, &targets);
			if ( !rc && targets != NULL) {
				dev = (dm_device_t *) targets->data.linear;
				if ( seg->start == dev->start &&
				     seg->size  == targets->length) {
					LOG_DEBUG("kernel object matches ... marking segment active\n");
					rc = 0;
				}
				else {
					LOG_ERROR("error, got a DM object using our segment name but the metadata differs. dont know what to do!\n");
					rc = ENODEV; // dont know how to handle this.
				}
			}
			else {
				rc = ENODEV;
			}
			if (targets) EngFncs->dm_deallocate_targets(targets);
		}
		else {
			LOG_DEBUG("segment is NOT active in the kernel\n");
			rc=ENODEV;
		}
	}

	if (rc) {
		seg->flags |= SOFLAG_NEEDS_ACTIVATE;
	}

	LOG_EXIT_INT(rc);
	return rc;
}                         


/*
 *  Function: mac_activate
 *
 *  Called in order to create a mapping of a disk partition
 *  in the kernel.
 */
int mac_activate( DISKSEG *seg )
{
	int rc=EINVAL;
	LOGICALDISK *ld=get_logical_disk(seg);
	dm_target_t target;
	dm_device_t linear;


	LOG_ENTRY();

	REQUIRE(ld != NULL);
	REQUIRE(seg!=NULL);
	REQUIRE(seg->data_type == DATA_TYPE);
	REQUIRE(isa_mac_segment(seg)==TRUE);

	target.start = 0;
	target.length = seg->size;
	target.type = DM_TARGET_LINEAR;
	target.data.linear = &linear;
	target.params = NULL;
	target.next = NULL;
	linear.major = ld->dev_major;
	linear.minor = ld->dev_minor;
	linear.start = seg->start;
	rc = EngFncs->dm_activate(seg, &target);

	if (rc) {
		rc = mac_get_DM_info(seg);  // FAILED ... test if already active
	}

	if (!rc) {
		seg->flags &= ~SOFLAG_NEEDS_ACTIVATE;
	}


	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function: mac_deactivate
 *
 *  Called in order to delete an existing mapping of a disk
 *  partition in the kernel.
 */
int mac_deactivate( DISKSEG *seg )
{
	int rc;

	LOG_ENTRY();

	rc = EngFncs->dm_deactivate(seg);

	LOG_EXIT_INT(rc);
	return rc;
}

