/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LVM Region Manager
 * File: evms2/engine/plugins/lvm/lvm_io.c
 *
 * Description: This file contains all functions pertaining to disk I/O within
 *              the LVM region manager plugin. This file should be the only
 *              location of calls to READ and WRITE (except the lvm_read and
 *              lvm_write API calls).
 */ 

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvmregmgr.h"

/**
 * lvm_convert_pv
 * @pv: pv_disk_t structure to be converted
 *
 * Convert a PV structure from memory-order to disk-order (or vice-versa).
 **/
static inline void lvm_endian_convert_pv(pv_disk_t * pv)
{
	LOG_ENTRY();

	pv->version			= DISK_TO_CPU16(pv->version);
	pv->pv_on_disk.base		= DISK_TO_CPU32(pv->pv_on_disk.base);
	pv->pv_on_disk.size		= DISK_TO_CPU32(pv->pv_on_disk.size);
	pv->vg_on_disk.base		= DISK_TO_CPU32(pv->vg_on_disk.base);
	pv->vg_on_disk.size		= DISK_TO_CPU32(pv->vg_on_disk.size);
	pv->pv_uuidlist_on_disk.base	= DISK_TO_CPU32(pv->pv_uuidlist_on_disk.base);
	pv->pv_uuidlist_on_disk.size	= DISK_TO_CPU32(pv->pv_uuidlist_on_disk.size);
	pv->lv_on_disk.base		= DISK_TO_CPU32(pv->lv_on_disk.base);
	pv->lv_on_disk.size		= DISK_TO_CPU32(pv->lv_on_disk.size);
	pv->pe_on_disk.base		= DISK_TO_CPU32(pv->pe_on_disk.base);
	pv->pe_on_disk.size		= DISK_TO_CPU32(pv->pe_on_disk.size);
	pv->pv_major			= DISK_TO_CPU32(pv->pv_major);
	pv->pv_number			= DISK_TO_CPU32(pv->pv_number);
	pv->pv_status			= DISK_TO_CPU32(pv->pv_status);
	pv->pv_allocatable		= DISK_TO_CPU32(pv->pv_allocatable);
	pv->pv_size			= DISK_TO_CPU32(pv->pv_size);
	pv->lv_cur			= DISK_TO_CPU32(pv->lv_cur);
	pv->pe_size			= DISK_TO_CPU32(pv->pe_size);
	pv->pe_total			= DISK_TO_CPU32(pv->pe_total);
	pv->pe_allocated		= DISK_TO_CPU32(pv->pe_allocated);
	pv->pe_start			= DISK_TO_CPU32(pv->pe_start);

	LOG_EXIT_VOID();
}

/**
 * lvm_read_pv
 * @segment:	Storage object to read the metadata from
 * @pv:		Location to put a pointer to the PV metadata
 *
 * Read the PV metadata from the specified segment. Check for an LVM PV
 * signature. Allocate space for and return a pointer to the metadata
 * that was read. If any error occurs, *pv will be set to NULL and no
 * memory will be allocated.
 **/
int lvm_read_pv(storage_object_t * segment,
		pv_disk_t ** pv)
{
	pv_disk_t * pv_buffer;
	int rc;

	LOG_ENTRY();
	LOG_EXTRA("Reading PV metadata from object %s\n", segment->name);

	*pv = NULL;

	/* Buffer for reading the PV metadata. */
	pv_buffer = EngFncs->engine_alloc(LVM_PV_DISK_SIZE);
	if (!pv_buffer) {
		LOG_CRITICAL("Memory error creating buffer to read PV metadata from object %s\n",
			     segment->name);
		rc = ENOMEM;
		goto out;
	}
	
	/* Read the first sector. */
	rc = READ(segment, bytes_to_sectors(LVM_PV_DISK_BASE),
		  bytes_to_sectors(LVM_PV_DISK_SIZE), pv_buffer);
	if (rc) {
		LOG_SERIOUS("Error reading PV metadata from object %s\n",
			    segment->name);
		goto out;
	}

	/* Endian-neutral conversion of PV metadata. */
	lvm_endian_convert_pv(pv_buffer);

	/* Check for an LVM signature and make sure the sizes match.
	 * Versions 1 and 2 are both valid now.
	 */
	if (!(pv_buffer->id[0] == 'H' &&
	      pv_buffer->id[1] == 'M' &&
	      (pv_buffer->version == 1 || pv_buffer->version == 2) &&
	      pv_buffer->pv_size == segment->size)) {
		LOG_EXTRA("Object %s is not an LVM PV\n", segment->name);
		rc = EINVAL;
		goto out;
	}

	/* This is a valid PV. Allocate a new pv_disk_t. */
	*pv = EngFncs->engine_alloc(sizeof(pv_disk_t));
	if (!*pv) {
		LOG_CRITICAL("Memory error creating new PV for object %s\n",
			     segment->name);
		rc = ENOMEM;
		goto out;
	}

	/* Copy the metadata. */
	memcpy(*pv, pv_buffer, sizeof(pv_disk_t));

out:
	EngFncs->engine_free(pv_buffer);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_write_pv
 *
 * Write the PV metadata sector to the specified PV.
 **/
int lvm_write_pv(lvm_physical_volume_t * pv_entry)
{
	pv_disk_t * pv_buffer;
	int rc;

	LOG_ENTRY();

	/* Buffer for writing the PV metadata. */
	pv_buffer = EngFncs->engine_alloc(LVM_PV_DISK_SIZE);
	if (!pv_buffer) {
		LOG_CRITICAL("Memory error creating buffer to write PV metadata to object %s\n",
			     pv_entry->segment->name);
		rc = ENOMEM;
		goto out;
	}

	memcpy(pv_buffer, pv_entry->pv, sizeof(pv_disk_t));

	/* Endian conversion of PV metadata. */
	lvm_endian_convert_pv(pv_buffer);

	/* Write the metadata. */
	rc = WRITE(pv_entry->segment, bytes_to_sectors(LVM_PV_DISK_BASE), 
		   bytes_to_sectors(LVM_PV_DISK_SIZE), pv_buffer);
	if (rc) {
		LOG_SERIOUS("Error writing PV metadata to object %s\n",
			    pv_entry->segment->name);
	}

	EngFncs->engine_free(pv_buffer);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_erase_pv
 *
 * Clear the PV metadata from the specified PV.
 **/
int lvm_erase_pv(storage_object_t * object)
{
	int rc;

	LOG_ENTRY();

	rc = KILL_SECTORS(object,
			  bytes_to_sectors(LVM_PV_DISK_BASE),
			  bytes_to_sectors(LVM_PV_DISK_SIZE));

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_endian_convert_vg
 * @vg: vg_disk_t structure to be converted
 *
 * Convert a VG structure from memory-order to disk-order (or vice-versa).
 **/
static inline void lvm_endian_convert_vg(vg_disk_t * vg)
{
	LOG_ENTRY();

	vg->vg_number	= DISK_TO_CPU32(vg->vg_number);
	vg->vg_access	= DISK_TO_CPU32(vg->vg_access);
	vg->vg_status	= DISK_TO_CPU32(vg->vg_status);
	vg->lv_max	= DISK_TO_CPU32(vg->lv_max);
	vg->lv_cur	= DISK_TO_CPU32(vg->lv_cur);
	vg->lv_open	= DISK_TO_CPU32(vg->lv_open);
	vg->pv_max	= DISK_TO_CPU32(vg->pv_max);
	vg->pv_cur	= DISK_TO_CPU32(vg->pv_cur);
	vg->pv_act	= DISK_TO_CPU32(vg->pv_act);
	vg->dummy	= DISK_TO_CPU32(vg->dummy);
	vg->vgda	= DISK_TO_CPU32(vg->vgda);
	vg->pe_size	= DISK_TO_CPU32(vg->pe_size);
	vg->pe_total	= DISK_TO_CPU32(vg->pe_total);
	vg->pe_allocated= DISK_TO_CPU32(vg->pe_allocated);
	vg->pvg_total	= DISK_TO_CPU32(vg->pvg_total);

	LOG_EXIT_VOID();
}

/**
 * lvm_read_vg
 * @segment:	Storage object to read the metadata from.
 * @pv:		PV metadata for this segment.
 * @vg:		Location to store a pointer to the VG metadata.
 *
 * Read the VG metadata from the specified segment/PV. Allocate space for
 * and return the metadata that was read.
 **/
int lvm_read_vg(storage_object_t * segment,
		pv_disk_t * pv,
		vg_disk_t ** vg)
{
	vg_disk_t * vg_buffer;
	int vg_sectors;
	int rc;

	LOG_ENTRY();
	LOG_EXTRA("Reading VG metadata from object %s\n", segment->name);

	*vg = NULL;

	/* Allocate a buffer to read the VG metadata. */
	vg_sectors = bytes_to_sectors(pv->vg_on_disk.size);
	vg_buffer = EngFncs->engine_alloc(sectors_to_bytes(vg_sectors));
	if (!vg_buffer) {
		LOG_CRITICAL("Memory error creating buffer to read VG metadata from object %s.\n",
			     segment->name);
		rc = ENOMEM;
		goto out;
	}

	/* Read the VG metadata. */
	rc = READ(segment, bytes_to_sectors(pv->vg_on_disk.base),
		  vg_sectors, vg_buffer);
	if (rc) {
		LOG_SERIOUS("Error reading VG metadata from object %s\n",
			    segment->name);
		goto out;
	}

	/* Endian-neutral conversion of VG metadata. */
	lvm_endian_convert_vg(vg_buffer);

	/* Allocate a new vg_disk_t to return. */
	*vg = EngFncs->engine_alloc(sizeof(vg_disk_t));
	if (!*vg) {
		LOG_CRITICAL("Memory error creating new VG structure for object %s\n",
			     segment->name);
		rc = ENOMEM;
		goto out;
	}

	/* Copy metadata. */
	memcpy(*vg, vg_buffer, sizeof(vg_disk_t));

out:
	EngFncs->engine_free(vg_buffer);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_write_vg
 *
 * Write the VG metadata to the specified PV.
 **/
int lvm_write_vg(lvm_physical_volume_t * pv_entry)
{
	vg_disk_t * vg_buffer;
	lvm_volume_group_t * group = pv_entry->group;
	storage_object_t * segment = pv_entry->segment;
	pv_disk_t * pv = pv_entry->pv;
	unsigned long vg_sectors;
	int rc;

	LOG_ENTRY();

	/* Allocate buffer for writing the VG metadata. */
	vg_sectors = bytes_to_sectors(pv->vg_on_disk.size);
	vg_buffer = EngFncs->engine_alloc(sectors_to_bytes(vg_sectors));
	if (!vg_buffer) {
		LOG_CRITICAL("Memory error creating buffer to write VG metadata to object %s.\n",
			     segment->name);
		rc = ENOMEM;
		goto out;
	}

	memcpy(vg_buffer, group->vg, sizeof(vg_disk_t));

	/* Endian conversion of PV metadata. */
	lvm_endian_convert_vg(vg_buffer);

	/* Write the metadata. */
	rc = WRITE(segment, bytes_to_sectors(pv->vg_on_disk.base),
		   vg_sectors, vg_buffer);
	if (rc) {
		LOG_SERIOUS("Error writing VG metadata to object %s\n",
			    segment->name);
	}
	
	EngFncs->engine_free(vg_buffer);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_read_uuid_list
 * @segment:	Object to read the UUIDs from
 * @pv:		PV metadata for this segment
 * @group:	Group to read the UUIDs into
 *
 * Read the list of PV UUIDs from the specified segment. The specified
 * group must already have a memory buffer to store this data. The first
 * entry in the UUID list (entry 0), corresponds to the first PV in the
 * group (pv_number 1). In order that we can index the UUID list by PV
 * number, skip the first slot in the group's UUID list when reading the
 * list from disk.
 **/
int lvm_read_uuid_list(storage_object_t	* segment,
		       pv_disk_t * pv,
		       lvm_volume_group_t * group)
{
	char * uuid_buffer = NULL;
	int uuid_sectors, i, rc = 0;

	LOG_ENTRY();

	/* Only read in the UUID list if it hasn't been already. */
	if (!(group->flags & LVM_VG_FLAG_UUID_LIST_PRESENT)) {
		LOG_DETAILS("Reading PV UUIDs for container %s\n",
			    group->container->name);

		/* Allocate I/O buffer. */
		uuid_sectors = bytes_to_sectors(pv->pv_uuidlist_on_disk.size);
		uuid_buffer = EngFncs->engine_alloc(sectors_to_bytes(uuid_sectors));
		if (!uuid_buffer) {
			LOG_CRITICAL("Memory error creating buffer to read UUID list from object %s\n",
				     segment->name);
			rc = ENOMEM;
			goto out;
		}

		/* Read the array from the PV. */
		rc = READ(segment,
			  bytes_to_sectors(pv->pv_uuidlist_on_disk.base),
			  uuid_sectors, uuid_buffer);
		if (rc) {
			LOG_SERIOUS("Error reading PV UUID list from object %s\n",
				    segment->name);
			goto out;
		}

		/* Copy each valid UUID to the group. UUIDs are char-strings,
		 * so no endian conversion is necessary.
		 */
		for (i = 0; i < group->vg->pv_max; i++) {
			if (uuid_buffer[i*NAME_LEN]) {
				if (!group->uuid_list[i+1]) {
					group->uuid_list[i+1] = EngFncs->engine_alloc(UUID_LEN);
					if (!group->uuid_list[i+1]) {
						LOG_CRITICAL("Memory error creating string for UUID entry %d in container %s\n",
							     i+1, group->container->name);
						rc = ENOMEM;
						goto out;
					}
				}
				memcpy(group->uuid_list[i+1],
				       &(uuid_buffer[i*NAME_LEN]), UUID_LEN);
			}
		}

		group->flags |= LVM_VG_FLAG_UUID_LIST_PRESENT;
	}
	else {
		LOG_EXTRA("Already read PV UUIDs for container %s\n",
			  group->container->name);
	}

out:
	EngFncs->engine_free(uuid_buffer);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_write_uuid_list
 *
 * Write the list of PV UUIDs to the specified PV. The group stores the
 * UUID list offset by one, so it can use the PV number to index the
 * table internally. So be sure to leave out the first entry in the
 * table when writing it to the PV.
 **/
int lvm_write_uuid_list(lvm_physical_volume_t * pv_entry)
{
	lvm_volume_group_t * group = pv_entry->group;
	storage_object_t * segment = pv_entry->segment;
	pv_disk_t * pv = pv_entry->pv;
	char * uuid_buffer;
	int uuid_sectors, i, rc;

	LOG_ENTRY();

	/* Allocate I/O buffer. */
	uuid_sectors = bytes_to_sectors(pv->pv_uuidlist_on_disk.size);
	uuid_buffer = EngFncs->engine_alloc(sectors_to_bytes(uuid_sectors));
	if (!uuid_buffer) {
		LOG_CRITICAL("Memory error creating buffer to write UUID list to object %s\n",
			     segment->name);
		rc = ENOMEM;
		goto out;
	}

	/* Copy all valid UUIDs from the group to the buffer. UUIDs are
	 * char-strings, so no endian conversion is necessary.
	 */
	for (i = 0; i < group->vg->pv_max; i++) {
		if (group->uuid_list[i+1]) {
			memcpy(&(uuid_buffer[i*NAME_LEN]),
			       group->uuid_list[i+1], UUID_LEN);
		}
	}

	/* Write the array to the PV. */
	rc = WRITE(segment, bytes_to_sectors(pv->pv_uuidlist_on_disk.base),
		   uuid_sectors, uuid_buffer);
	if (rc) {
		LOG_SERIOUS("Error writing UUID list to object %s\n",
			    segment->name);
	}

	EngFncs->engine_free(uuid_buffer);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_endian_convert_lv
 *
 * Convert a LV structure from memory-order to disk-order (or vice-versa).
 **/
static inline void lvm_endian_convert_lv(lv_disk_t * lv)
{
	lv->lv_access		= DISK_TO_CPU32(lv->lv_access);
	lv->lv_status		= DISK_TO_CPU32(lv->lv_status);
	lv->lv_open		= DISK_TO_CPU32(lv->lv_open);
	lv->lv_dev		= DISK_TO_CPU32(lv->lv_dev);
	lv->lv_number		= DISK_TO_CPU32(lv->lv_number);
	lv->lv_mirror_copies	= DISK_TO_CPU32(lv->lv_mirror_copies);
	lv->lv_recovery		= DISK_TO_CPU32(lv->lv_recovery);
	lv->lv_schedule		= DISK_TO_CPU32(lv->lv_schedule);
	lv->lv_size		= DISK_TO_CPU32(lv->lv_size);
	lv->lv_snapshot_minor	= DISK_TO_CPU32(lv->lv_snapshot_minor);
	lv->lv_chunk_size	= DISK_TO_CPU16(lv->lv_chunk_size);
	lv->dummy		= DISK_TO_CPU16(lv->dummy);
	lv->lv_allocated_le	= DISK_TO_CPU32(lv->lv_allocated_le);
	lv->lv_stripes		= DISK_TO_CPU32(lv->lv_stripes);
	lv->lv_stripesize	= DISK_TO_CPU32(lv->lv_stripesize);
	lv->lv_badblock		= DISK_TO_CPU32(lv->lv_badblock);
	lv->lv_allocation	= DISK_TO_CPU32(lv->lv_allocation);
	lv->lv_io_timeout	= DISK_TO_CPU32(lv->lv_io_timeout);
	lv->lv_read_ahead	= DISK_TO_CPU32(lv->lv_read_ahead);
}

/**
 * lvm_endian_convert_lvs
 *
 * Endian-convert all LV structures in a volume group.
 **/
void lvm_endian_convert_lvs(lvm_volume_group_t * group)
{
	int i;
	LOG_ENTRY();
	for (i = 0; i < MAX_LV; i++) {
		lvm_endian_convert_lv(&(group->lv_array[i]));
	}
	LOG_EXIT_VOID();
}

/**
 * lvm_read_lv_array
 * @group: Group to read the LV array for.
 *
 * Read the LV metadata from the first PV in the group. The group needs
 * to already have a memory buffer to store this data. If the read on the
 * first PV fails, continue reading from the remaining PVs until one works.
 **/
int lvm_read_lv_array(lvm_volume_group_t * group)
{
	storage_object_t * segment;
	pv_disk_t * pv;
	int i, rc = 0;

	LOG_ENTRY();

	/* Only read the LV array if it hasn't been already. */
	if (!(group->flags & LVM_VG_FLAG_LV_LIST_PRESENT)) {
		LOG_DETAILS("Reading LV metadata for container %s\n",
			    group->container->name);

		for (i = 1; i <= MAX_PV; i++) {
			/* Find the next PV in the group. */
			if (!group->pv_list[i]) {
				continue;
			}

			segment = group->pv_list[i]->segment;
			pv = group->pv_list[i]->pv;
			rc = READ(segment,
				  pv->lv_on_disk.base >> EVMS_VSECTOR_SIZE_SHIFT,
				  bytes_to_sectors(min(EVMS_LVM_MAX_LV_METADATA_SIZE,
						       pv->lv_on_disk.size)),
				  group->lv_array_disk);
			if (rc) {
				LOG_SERIOUS("Error reading LV metadata from object %s\n",
					    segment->name);
				continue;
			}
			group->lv_array = (lv_disk_t*)(((unsigned long)group->lv_array_disk) +
						       (pv->lv_on_disk.base & (EVMS_VSECTOR_SIZE - 1)));

			/* Endian conversion of LV metadata. */
			lvm_endian_convert_lvs(group);

			group->flags |= LVM_VG_FLAG_LV_LIST_PRESENT;
			break;
		}

		if (rc) {
			LOG_SERIOUS("Failed to read LV metadata from all objects in container %s\n",
				    group->container->name);
		}
	} else {
		LOG_EXTRA("Already read LV metadata for container %s\n",
			  group->container->name);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_write_lv_array
 *
 * Write the LV metadata to the specified PV.
 **/
int lvm_write_lv_array(lvm_physical_volume_t * pv_entry)
{
	lvm_volume_group_t * group = pv_entry->group;
	storage_object_t * segment = pv_entry->segment;
	pv_disk_t * pv = pv_entry->pv;
	int rc;

	LOG_ENTRY();

	/* Endian-conversion is done outside
	 * this function as an optimization.
	 */

	rc = WRITE(segment, pv->lv_on_disk.base >> EVMS_VSECTOR_SIZE_SHIFT,
		   bytes_to_sectors(min(EVMS_LVM_MAX_LV_METADATA_SIZE,
					pv->lv_on_disk.size)),
		   group->lv_array_disk);
	if (rc) {
		LOG_SERIOUS("Error writing LV array to object %s\n",
			    segment->name);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

#define PE_MAP_SECTORS(pe_total)  bytes_to_sectors((pe_total) * sizeof(pe_disk_t))

static inline void lvm_endian_convert_pe_map(pe_disk_t * pe_map,
					     u_int32_t pe_total)
{
	int i;
	LOG_ENTRY();
	for (i = 0; i < pe_total; i++) {
		pe_map[i].lv_num = DISK_TO_CPU16(pe_map[i].lv_num);
		pe_map[i].le_num = DISK_TO_CPU16(pe_map[i].le_num);
	}
	LOG_EXIT_VOID();
}

/**
 * lvm_read_pe_map
 *
 * Read the PE maps from the specified PV. This PV needs to already
 * have a buffer to hold the PE maps.
 **/
int lvm_read_pe_map(lvm_physical_volume_t * pv_entry)
{
	storage_object_t * segment = pv_entry->segment;
	pv_disk_t * pv = pv_entry->pv;
	pe_disk_t * pe_map;
	u_int64_t pe_map_sectors = PE_MAP_SECTORS(pv->pe_total);
	unsigned int i;
	int rc;

	LOG_ENTRY();

	/* Allocate a buffer to read the PE map from disk. */
	pe_map = EngFncs->engine_alloc(sectors_to_bytes(pe_map_sectors));
	if (!pe_map) {
		rc = ENOMEM;
		goto out;
	}

	/* Read the map from disk, and convert to CPU order. */
	rc = READ(segment, bytes_to_sectors(pv->pe_on_disk.base),
		  pe_map_sectors, pe_map);
	if (rc) {
		LOG_SERIOUS("Error reading PE map from object %s\n",
			    segment->name);
		goto out;
	}

	lvm_endian_convert_pe_map(pe_map, pv->pe_total);

	/* Copy the PE info into the PV's map. */
	for (i = 0; i < pv->pe_total; i++) {
		pv_entry->pe_map[i].pe.lv_num = pe_map[i].lv_num;
		pv_entry->pe_map[i].pe.le_num = pe_map[i].le_num;
	}

out:
	EngFncs->engine_free(pe_map);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_write_pe_map
 *
 * Write the PE map to the specified PV.
 **/
int lvm_write_pe_map(lvm_physical_volume_t * pv_entry)
{
	storage_object_t * segment = pv_entry->segment;
	pv_disk_t * pv = pv_entry->pv;
	pe_disk_t * pe_map;
	u_int64_t pe_map_sectors = PE_MAP_SECTORS(pv->pe_total);
	unsigned int i;
	int rc;

	LOG_ENTRY();

	/* Allocate a buffer to write the PE map to disk. */
	pe_map = EngFncs->engine_alloc(sectors_to_bytes(pe_map_sectors));
	if (!pe_map) {
		rc = ENOMEM;
		goto out;
	}

	/* Copy the PE info from the PV's map. */
	for (i = 0; i < pv->pe_total; i++) {
		pe_map[i].lv_num = pv_entry->pe_map[i].pe.lv_num;
		pe_map[i].le_num = pv_entry->pe_map[i].pe.le_num;
	}

	/* Convert the map to disk order and write. */
	lvm_endian_convert_pe_map(pe_map, pv->pe_total);

	rc = WRITE(segment, bytes_to_sectors(pv->pe_on_disk.base),
		   pe_map_sectors, pe_map);
	if (rc) {
		LOG_SERIOUS("Error writing PE map to object %s\n",
			    segment->name);
	}

	EngFncs->engine_free(pe_map);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_erase_group_metadata
 *
 * This function erases all of the group-specific metadata from the
 * specified PV. This MUST be called BEFORE the PV is removed from
 * the group, or the metadata pointers will have been erased.
 */
void lvm_erase_group_metadata(lvm_physical_volume_t * pv_entry)
{
	storage_object_t * segment = pv_entry->segment;
	pv_disk_t * pv = pv_entry->pv;

	LOG_ENTRY();

	/* Erase the PE map, the LV structures, the UUID list,
	 * and the VG structure.
	 */
	KILL_SECTORS(segment, bytes_to_sectors(pv->pe_on_disk.base),
		     PE_MAP_SECTORS(pv->pe_total));
	KILL_SECTORS(segment, bytes_to_sectors(pv->lv_on_disk.base),
		     bytes_to_sectors(pv->lv_on_disk.size));
	KILL_SECTORS(segment, bytes_to_sectors(pv->pv_uuidlist_on_disk.base),
		     bytes_to_sectors(pv->pv_uuidlist_on_disk.size));
	KILL_SECTORS(segment, bytes_to_sectors(pv->vg_on_disk.base),
		     bytes_to_sectors(pv->vg_on_disk.size));

	LOG_EXIT_VOID();
}

/**
 * lvm_remap_sector
 *
 * This function is used by Read and Write to remap volume-relative LBA to
 * PV-relative LBA.
 **/
void lvm_remap_sector(lvm_logical_volume_t * volume,
		      lsn_t org_sector,
		      sector_count_t org_size,
		      lsn_t * new_sector,
		      sector_count_t * new_size,
		      lvm_physical_volume_t ** pv_entry)
{
	lv_disk_t * lv = volume->lv;
	vg_disk_t * vg = volume->group->vg;
	u_int32_t le;
	u_int32_t column, columns;
	u_int32_t sectors_per_column, sector_in_column, stripe_in_column;
	u_int32_t le_in_column, stripe_in_le;
	u_int32_t offset_in_stripe, offset_in_le;

	LOG_ENTRY();

	*new_size = org_size;

	if (lv->lv_stripes > 1) {
		/* Volume is striped. If the request crosses a stripe
		 * boundary, reset the new_size appropriately.
		 */
       		sectors_per_column	= lv->lv_stripes * vg->pe_size;
       		column			= org_sector / sectors_per_column;
       		sector_in_column	= org_sector % sectors_per_column;
       		stripe_in_column	= sector_in_column / lv->lv_stripesize;
       		le_in_column		= stripe_in_column % lv->lv_stripes;
       		columns			= lv->lv_allocated_le / lv->lv_stripes;
       		le			= column + (columns * le_in_column);

       		offset_in_stripe	= org_sector % lv->lv_stripesize;
       		stripe_in_le		= stripe_in_column / lv->lv_stripes;
       		offset_in_le		= offset_in_stripe + stripe_in_le *
					  lv->lv_stripesize;

		if (offset_in_stripe + org_size > lv->lv_stripesize) {
			*new_size = lv->lv_stripesize - offset_in_stripe;
		}
	} else {
		/* Non-striped volume. Find LE and offset. If the request.
		 * crosses a PE boundary, reset the new_size appropriately.
		 */
       		le = org_sector / vg->pe_size;
       		offset_in_le = org_sector % vg->pe_size;

		if (offset_in_le + org_size > vg->pe_size) {
			*new_size = vg->pe_size - offset_in_le;
		}
	}

	if (volume->le_map[le].pe) {
		*pv_entry = volume->le_map[le].pe->pv;
		*new_sector = volume->le_map[le].pe->sector + offset_in_le;
	} else {
		*pv_entry = NULL;
		*new_sector = 0;
	}

	LOG_EXIT_VOID();
}

