/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: shrink.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include "fullengine.h"
#include "shrink.h"
#include "lists.h"
#include "engine.h"
#include "handlemgr.h"
#include "common.h"
#include "commit.h"
#include "discover.h"
#include "object.h"
#include "memman.h"
#include "internalAPI.h"
#include "message.h"
#include "remote.h"
#include "volume.h"


int get_object_shrink_points(storage_object_t * obj, u_int64_t * max_delta_size, list_anchor_t shrink_points) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = obj->plugin->functions.plugin->can_shrink(obj, max_delta_size, shrink_points);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int get_volume_shrink_points(logical_volume_t * volume, list_anchor_t shrink_points) {

	int rc = 0;
	u_int64_t max_delta_size = -1;

	LOG_PROC_ENTRY();

	if (is_kernel_volume_mounted(volume, ERROR)) {
		LOG_PROC_EXIT_INT(EBUSY);
		return EBUSY;
	}

	if (volume->flags & VOLFLAG_MKFS) {
		LOG_ERROR("Volume %s cannot be shrunk because it is scheduled to have a file system installed on it.\n", volume->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/*
	 * If the volume currently has an FSIM and the FSIM is not scheduled to be
	 * removed, use the FSIM's limits.
	 */
	if ((volume->original_fsim != NULL) &&
	    !(volume->flags & VOLFLAG_UNMKFS)) {

		/*
		 * Reget the current size and limits because the file system could be
		 * mounted and be active.
		 */
		rc = get_volume_sizes_and_limits(volume);

		if (rc == 0) {
			max_delta_size = volume->fs_size - volume->min_fs_size;
		}
	}

	if (rc == 0) {
		rc = get_object_shrink_points(volume->object, &max_delta_size, shrink_points);
	}

	if (rc == 0) {
		/*
		 * If the volume has an FSIM and the FSIM is not scheduled to be removed
		 * and there are shrink points, check if the FSIM can handle the shrink.
		 */
		if ((volume->original_fsim != NULL) &&
		    !(volume->flags & VOLFLAG_UNMKFS) &&
		    !list_empty(shrink_points)) {

			max_delta_size = round_down_to_hard_sector(max_delta_size, volume->object);

			rc = volume->original_fsim->functions.fsim->can_shrink_by(volume, &max_delta_size);

			/*
			 * The FSIM cannot handle the shrink.  Delete all the shrink
			 * points from the list.
			 */
			if (rc != 0) {
				/*
				 * Can't use DeleteAllItems(shrink_ponts, TRUE)
				 * because it uses free().  Shrink points are
				 * allocted with engine_alloc() which may add a
				 * debug header.  Extract each item and use
				 * engine_free().
				 */
				list_element_t iter1;
				list_element_t iter2;
				void * thing;

				LIST_FOR_EACH_SAFE(shrink_points, iter1, iter2, thing) {
					delete_element(iter1);
					engine_free(thing);
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Find out if an object can be shrunk.  The "thing" must be either a volume
 * or a top level object.  What we'll do is call the internal
 * get_shrink_points() and see if any shrink candidates show up.  If we get
 * shrink points then return 0 to say that the object can be shrunk.  Else,
 * return an error.
 */
int evms_can_shrink(object_handle_t thing) {

	int rc = 0;
	void * object;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_shrink(thing);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(thing,
			      &object,
			      &type);

	if (rc == HANDLE_MANAGER_NO_ERROR) {

		switch (type) {
			case DISK:
			case SEGMENT:
			case REGION:
			case EVMS_OBJECT:
				{
					storage_object_t * obj = object;

					/*
					 * Only top level objects can be shrunk.
					 */
					if (is_top_object(obj)) {
						STATIC_LIST_DECL(shrink_points);
						u_int64_t max_delta_size = obj->size;

						rc = get_object_shrink_points(obj, &max_delta_size, &shrink_points);

						if (rc == 0) {
							if (list_empty(&shrink_points)) {
								rc = ENOENT;
							}
						}

					} else {
						LOG_DETAILS("Object %s is not a top level object.  Only volumes and top level objects can be shrunk.\n", obj->name);
						rc = EINVAL;
					}
				}
				break;

			case VOLUME:
				{
					logical_volume_t * volume = (logical_volume_t *) object;

					if (volume->flags & VOLFLAG_COMPATIBILITY) {
						if (is_kernel_volume_mounted(volume, DETAILS)) {
							rc = EBUSY;
						}
					}
					
					if (rc == 0) {
						if (volume->flags & (VOLFLAG_ACTIVE | VOLFLAG_NEW)) {
							STATIC_LIST_DECL(shrink_points);

							rc = get_volume_shrink_points(volume, &shrink_points);

							if (rc == 0) {
								if (list_empty(&shrink_points)) {
									rc = ENOENT;
								}
							}

						} else {
							LOG_DETAILS("Volume %s cannot be shrunk because it is not active.\n", volume->name);
							rc = EINVAL;
						}
					}
				}
				break;

			default:
				rc = EINVAL;
				break;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Make an array of shrink handles (shrink_handle_array_t) for the objects in
 * a list_anchor_t.
 */
static int make_shrink_handle_array(list_anchor_t list, shrink_handle_array_t * * psha) {

	int rc = 0;
	uint count;
	uint size;
	shrink_handle_array_t * sha = NULL;

	LOG_PROC_ENTRY();

	count = list_count(list);

	if (rc == 0) {
		LOG_DEBUG("Number of objects in the list:  %d\n", count);
		if (count > 1) {
			size = sizeof(shrink_handle_array_t) + ((count -1) * sizeof(shrink_handle_t));
		} else {
			size = sizeof(shrink_handle_array_t);
		}

		sha = alloc_app_struct(size, NULL);
		if (sha != NULL) {
			list_element_t iter;
			shrink_object_info_t * object_info;

			LIST_FOR_EACH(list, iter, object_info) {

				LOG_DEBUG("Add entry for storage object %s.\n", object_info->object->name);

				rc = ensure_app_handle(object_info->object);
				if (rc == 0) {
					sha->shrink_point[sha->count].object = object_info->object->app_handle;
					sha->shrink_point[sha->count].max_shrink_size = object_info->max_shrink_size;
					sha->count++;
				}
			}
		} else {
			rc = ENOMEM;
		}
	}

	*psha = sha;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Find out which objects in the stack can shrink.  evms_get_shrink_points()
 * must be targeted at either a volume or a top level object.  Any plug-in
 * that can shrink will add a shrink_object_info_t to the shrink_points
 * list.  The list is then converted into a shrink_handle_array_t for return
 * to the user.
 */
int evms_get_shrink_points(object_handle_t thing, shrink_handle_array_t * * shrink_points) {

	int rc = 0;
	void * object;
	object_type_t type;
	STATIC_LIST_DECL(shrink_point_list);

	LOG_PROC_ENTRY();

	rc = check_engine_read_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_get_shrink_points(thing, shrink_points);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(thing,
			      &object,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (shrink_points == NULL) {
		LOG_ERROR("The pointer to the shrink points list cannot be NULL.\n");
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}
	
	switch (type) {
		case VOLUME:
			{
				logical_volume_t * volume = (logical_volume_t *) object;


				if (volume->flags & VOLFLAG_COMPATIBILITY) {
					if (is_kernel_volume_mounted(volume, ERROR)) {
						rc = EBUSY;
                                        }
                                }
					
				if (rc == 0) {
					if (volume->flags & (VOLFLAG_ACTIVE | VOLFLAG_NEW)) {
						rc = get_volume_shrink_points(volume, &shrink_point_list);

					} else {
						LOG_DETAILS("Volume %s cannot be shrunk because it is not active.\n", volume->name);
						rc = EINVAL;
					}
				}
			}
			break;

		case EVMS_OBJECT:
		case REGION:
		case SEGMENT:
		case DISK:
			{
				storage_object_t * obj = (storage_object_t *) object;
				/*
				 * Only top level objects can be shrunk.
				 */
				if (is_top_object(obj)) {
					u_int64_t max_delta_size = obj->size;

					rc = get_object_shrink_points(obj, &max_delta_size, &shrink_point_list);

				} else {
					LOG_ERROR("Object %s is not a top level object.  Only volumes and top level objects can be shrunk.\n", obj->name);
					rc = EINVAL;
				}
			}
			break;

		default:
			LOG_ERROR("An object of type %d cannot be shrunk.\n", type);
			rc = EINVAL;
			break;
	}

	if (rc == 0) {
		rc = make_shrink_handle_array(&shrink_point_list, shrink_points);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * If the object is part of a volume and the volume has no FSIM, warn the
 * user that there is no FSIM to handle the shrink.  The user has the
 * choice of continuing or aborting.
 * Returns 0 if the user wants to continue, E_CANCELED if the user wants to abort.
 */
static int warn_if_no_fsim(storage_object_t * obj) {

	int rc = 0;

	LOG_PROC_ENTRY();

	if (obj->volume != NULL) {
		if (obj->volume->original_fsim == NULL) {
			char * choices[] = {"Continue", "Cancel", NULL};
			int answer = 0;	    /* Initialize to "Continue" */

			engine_user_message(&answer, choices,
					    "WARNING: Volume \"%s\" does not have an associated File System Interface Module which could coordinate the shrinking of the file system on the volume.  "
					    "The file system will not be shrunk.  "
					    "You may need to run a separate utility to shrink the file system before this operation starts.  "
					    "Shrinking the volume may make the file system unusable.\n",
					    obj->volume->name);

			if (answer != 0) {
				rc = E_CANCELED;
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int isa_valid_shrink_input_object(storage_object_t    * obj,
					 storage_container_t * disk_group) {

	LOG_PROC_ENTRY();

	if ((obj->object_type != DISK) &&
	    (obj->object_type != SEGMENT) &&
	    (obj->object_type != REGION) &&
	    (obj->object_type != EVMS_OBJECT)) {
		LOG_ERROR("This is not a storage object.\n");
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* The object must not be corrupt. */
	if (obj->flags & SOFLAG_CORRUPT) {
		LOG_ERROR("Object %s is not a valid input object.  It is corrupt.\n", obj->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* The object must not insist on being the top object. */
	if (obj->flags & SOFLAG_MUST_BE_TOP) {
		LOG_ERROR("Object %s is not a valid input object.  It insists it must be a top level object.\n", obj->name);
		return EINVAL;
	}

	/* The object must be in the specified disk group. */
	if (obj->disk_group != disk_group) {
		LOG_ERROR("Object %s in disk group %s is not in disk group %s.\n", obj->name, (obj->disk_group != NULL) ? obj->disk_group->name : "(local)", (disk_group != NULL) ? disk_group->name : "(local)");
		return EINVAL;
	}

	LOG_DEBUG("Object %s is a valid input object.\n", obj->name);
	LOG_PROC_EXIT_INT(0);
	return 0;
}


static void post_shrink_cleanup(list_anchor_t      input_object_list,
				storage_object_t * top_object,
				storage_object_t * obj) {

	uint count;
	list_element_t iter;
	storage_object_t * child_obj;

	/*
	 * Check if the input object(s) is/are different from the object that
	 * was shrunk.
	 */
	count = list_count(input_object_list);

	if ((count > 1) ||
	    ((count == 1) &&
	     (first_thing(input_object_list, NULL) != obj))) {

		LIST_FOR_EACH(input_object_list, iter, child_obj) {

			/*
			 * Make sure the child objects are marked as not being
			 * part of a volume.
			 */
			set_volume_in_object(child_obj, NULL);

			 /*
			  * The child objects were in use.  Stop data may have
			  * been trashed.
			  */
			child_obj->flags &= ~SOFLAG_HAS_STOP_DATA;
		}
	}

	if (obj->object_type == EVMS_OBJECT) {
		/*
		 * Mark all the child feature headers dirty to ensure that they
		 * get the correct depth and get written in the right place.
		 */
		LIST_FOR_EACH(obj->child_objects, iter, child_obj) {
			mark_feature_headers_dirty(child_obj);
		}
	}

	/* If the object is part of a volume... */
	if (obj->volume != NULL) {

		/*
		 * Mark it dirty so that the feature headers get written to
		 * their new locations.
		 */
		obj->volume->flags |= VOLFLAG_DIRTY;

		if (!(obj->volume->flags & VOLFLAG_COMPATIBILITY)) {
			obj->volume->flags |= VOLFLAG_NEEDS_ACTIVATE;
		}

		/* Update the volume size. */
		obj->volume->vol_size = top_object->size;

		if (top_object->feature_header != NULL) {

			/*
			 * Object has EVMS volume feature headers.  Subtract
			 * them from the size.
			 */
			obj->volume->vol_size -= EVMS_FEATURE_HEADER_SECTORS * 2;
		}

		obj->volume->vol_size = round_down_to_hard_sector(obj->volume->vol_size, top_object);

		/* Set the smallest shrink size for the volume. */
		obj->volume->shrink_vol_size = min(obj->volume->shrink_vol_size,
						   obj->volume->vol_size);
	}
}


/*
 * Shrink the size of an object.  The shrink gets targeted at the top object of
 * the feature stack.  The plug-ins pass it on down until it hits the object
 * that is to do the real shrink.
 */
int evms_shrink(object_handle_t thing, handle_array_t * input_objects, option_array_t * options) {

	int rc = 0;
	void * object;
	object_type_t type;
	storage_object_t * obj;
	list_anchor_t input_object_list;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_shrink(thing, input_objects, options);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(thing,
			      &object,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if ((type != EVMS_OBJECT) &&
	    (type != REGION) &&
	    (type != SEGMENT) &&
	    (type != DISK)) {
		LOG_ERROR("Object of type %d cannot be shrunk.\n", type);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	obj = (storage_object_t *) object;

	if (obj->volume != NULL) {

		if (obj->volume->flags & VOLFLAG_COMPATIBILITY) {
			if (is_kernel_volume_mounted(obj->volume, ERROR)) {
				LOG_PROC_EXIT_INT(EBUSY);
				return EBUSY;
			}
		}

		if (!(obj->volume->flags & (VOLFLAG_ACTIVE | VOLFLAG_NEW))) {
			LOG_DETAILS("Object %s cannot be shrunk because volume %s is not active.\n", obj->name, obj->volume->name);
			LOG_PROC_EXIT_INT(EINVAL);
			return EINVAL;

		} else {
			if (obj->volume->flags & VOLFLAG_MKFS) {
				LOG_ERROR("Volume %s cannot be shrunk because it is scheduled to have a file system installed on it.\n", obj->volume->name);
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
			}
		}
	}

	input_object_list = allocate_list();

	if (input_object_list == NULL) {
		LOG_CRITICAL("Error allocating memory to create the input object list.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	rc = make_list(input_objects, input_object_list);

	if (rc == 0) {
		list_element_t iter;
		storage_object_t * input_obj;

		/*
		 * Make sure all of the input objects are storage objects.
		 * (The caller could have supplied handles for bogus items such
		 * as plug-ins or volumes.)
		 */
		LIST_FOR_EACH(input_object_list, iter, input_obj) {
			rc = isa_valid_shrink_input_object(input_obj, obj->disk_group);
			if (rc != 0) {
				break;
			}
		}

		if (rc == 0) {

			/* Find the top object in the feature stack. */
			storage_object_t * top_object = obj;
			uint parent_count = 0;

			parent_count = list_count(top_object->parent_objects);

			while (parent_count == 1) {
				top_object = first_thing(top_object->parent_objects, NULL);
				parent_count = list_count(top_object->parent_objects);
			}

			if (rc == 0) {
				/*
				 * Warn the user if there is no FSIM to handle
				 * the shrink.
				 */
				rc = warn_if_no_fsim(obj);

				if (rc == 0) {
					/*
					 * If the object is not part of a volume
					 * and it has stop data, remove the stop
					 * data from its current location.  It
					 * will be reapplied during commit.
					 */
					if ((top_object->volume == NULL) &&
					    (top_object->flags & SOFLAG_HAS_STOP_DATA)) {
						top_object->plugin->functions.plugin->add_sectors_to_kill_list(top_object, top_object->size - (EVMS_FEATURE_HEADER_SECTORS * 2), EVMS_FEATURE_HEADER_SECTORS * 2);
						top_object->flags &= ~SOFLAG_HAS_STOP_DATA;
					}

					/*
					 * Start the shrink at the top level
					 * object.
					 */
					rc = top_object->plugin->functions.plugin->shrink(top_object, obj, input_object_list, options);

					if (rc == 0) {
						post_shrink_cleanup(input_object_list, top_object, obj);
					}
				}

			} else {
				LOG_WARNING("Error code %d encountered when trying to find the top level object.\n", rc);
			}

		} else {
			LOG_ERROR("One or more items in the input object list is not a storage object.\n");
			rc = EINVAL;
		}

	} else {
		LOG_ERROR("Error code %d when making a list from the input_objects handle array.\n", rc);
	}

	destroy_list(input_object_list);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * engine_can_shrink_by() is a service that the Engine provides for the
 * plug-ins.  A plug-in calls engine_can_shrink_by() to find out if all of its
 * parent objects, including the volume, would approve of a shrink by the
 * specified amount.  The Engine walks up the object tree calling each
 * plug-in's can_shrink_by() function to see if it would approve of the
 * shrink.  Each plug-in can update the "size" if the resulting size of its
 * object would have a different delta size than the delta size of its child
 * object.  If the top of the object stack is reached and the object is part of
 * a volume, the Engine then checks the resulting delta size against any limits
 * that an FSIM may have imposed on the volume.
 */
int engine_can_shrink_by(storage_object_t * object, sector_count_t * delta_size) {

	int rc = 0;
	storage_object_t * curr_object = object;
	sector_count_t original_delta_size = *delta_size;

	LOG_PROC_ENTRY();

	while ((rc == 0) && !list_empty(curr_object->parent_objects)) {

		curr_object = first_thing(curr_object->parent_objects, NULL);

		if (curr_object != NULL) {
			rc = curr_object->plugin->functions.plugin->can_shrink_by(curr_object, delta_size);
		}
	}

	if (rc == 0) {
		/*
		 * If this object is part of a volume and the volume has an FSIM that is
		 * not scheduled to be removed, check to make sure the final size does
		 * not go below the FSIM's minimum size for the file system.
		 */
		if (object->volume != NULL) {

			if (object->volume->flags & VOLFLAG_COMPATIBILITY) {
				if (is_kernel_volume_mounted(object->volume, ERROR)) {
					rc = EBUSY;
				}

			}
			
			if (rc == 0) {
				if ((object->volume->original_fsim != NULL) &&
				    !(object->volume->flags & VOLFLAG_UNMKFS)) {
					rc = object->volume->original_fsim->functions.fsim->can_shrink_by(object->volume, delta_size);
				}
			}
		}
	}

	if (rc == 0) {
		if (*delta_size != original_delta_size) {
			rc = EAGAIN;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

