# Copyright (C) 2008, 2009, 2010  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Main entry point for Computer Janitor dbus daemon."""

from __future__ import absolute_import, unicode_literals

__metaclass__ = type
__all__ = [
    'main',
    ]


import os
import gobject
import logging
import argparse
import warnings
import dbus.mainloop.glib
import logging.handlers

from computerjanitorapp import __version__, setup_gettext
from computerjanitord.service import Service

_ = setup_gettext()


# 2010-02-09 barry: computerjanitor.package_cruft has a DeprecationWarning,
# but that really needs to be fixed in that package, which in turn needs to be
# ripped out of update-manager.
warnings.filterwarnings('ignore', category=DeprecationWarning,
                        module='computerjanitor.package_cruft')


# As per aptdaemon, when run as a dbus service, we must add the necessary
# paths to $PATH.
REQUIRED_PATH = '/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin'


class Options:
    """Command line options."""

    def __init__(self):
        self.parser = argparse.ArgumentParser(
            description=_("""\
            Computer janitor dbus daemon.  You must run this daemon as root.
            """))
        self.parser.add_argument(
            '--version', action='version',
            version=_('Computer Janitor dbus daemon {version}'.format(
                version=__version__)))
        self.parser.add_argument(
            '-n', '--dry-run', action='store_true',
            help=_("""\
            Only pretend to do anything permanent.  This is useful for testing
            and debugging."""))
        self.parser.add_argument(
            '-f', '--state-file', metavar='FILE',
            help=_('Store ignored state in FILE instead of the default.'))
        self.arguments = self.parser.parse_args()


class ASCIIFormatter(logging.Formatter):
    """Force the log messages to ASCII."""
    def format(self, record):
        message = logging.Formatter.format(self, record)
        return message.encode('ascii', 'replace')


def main():
    """Main entry point."""
    # Set up logging.
    if os.environ.get('COMPUTER_JANITOR_DEBUG') is not None:
        level = logging.DEBUG
    else:
        level = logging.INFO
    # When running under dbus, $PATH will not be set.  However it must be in
    # order to find dpkg(1) and friends.  Hard code it in the same way that
    # aptdaemon does.
    if os.environ.get('PATH') is None:
        os.putenv('PATH', REQUIRED_PATH)
    logging.basicConfig(level=level)
    log = logging.getLogger('computerjanitor')
    # SysLogHandler does not recognize unicode arguments.
    syslog = logging.handlers.SysLogHandler(b'/dev/log')
    # syslog will provide a timestamp.
    formatter = ASCIIFormatter('computerjanitord:%(levelname)s: %(message)s')
    syslog.setFormatter(formatter)
    syslog.setLevel(level)
    log.addHandler(syslog)
    options = Options()
    # Ensure that we're running as root.  Do this here instead of ../janitord
    # so that we can still run 'janitord --version' and 'janitord --help' as
    # non-root.
    if os.getuid() != 0:
        options.parser.error('You must run this as root')
        # No return.
    dbus.mainloop.glib.DBusGMainLoop(set_as_default=True)
    server = Service(options)
    try:
        gobject.MainLoop().run()
    except KeyboardInterrupt:
        pass


if __name__ == '__main__':
    main()
