/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2001-2002  Nokia Corporation
 *  Copyright (C) 2002-2003  Maxim Krasnyansky <maxk@qualcomm.com>
 *  Copyright (C) 2002-2004  Marcel Holtmann <marcel@holtmann.org>
 *  Copyright (C) 2002-2003  Stephen Crane <steve.crane@rococosoft.com>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation;
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 *  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF THIRD PARTY RIGHTS.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) AND AUTHOR(S) BE LIABLE FOR ANY
 *  CLAIM, OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES 
 *  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN 
 *  ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF 
 *  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *  ALL LIABILITY, INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PATENTS, 
 *  COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS, RELATING TO USE OF THIS 
 *  SOFTWARE IS DISCLAIMED.
 *
 *
 *  $Id: sdp.h,v 1.8 2004/10/13 12:03:16 holtmann Exp $
 */

#ifndef __SDP_H
#define __SDP_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>

#define SDP_UNIX_PATH "/var/run/sdp"
#define SDP_RESPONSE_TIMEOUT	20
#define SDP_REQ_BUFFER_SIZE	2048
#define SDP_RSP_BUFFER_SIZE	65535
#define SDP_PDU_CHUNK_SIZE	1024

/*
 * All definitions are based on Bluetooth Assigned Numbers
 * of the Bluetooth Specification
 */
#define SDP_PSM 		0x0001

/*
 * Protocol UUIDs
 */
#define SDP_UUID	0x0001
#define RFCOMM_UUID	0x0003
#define TCS_BIN_UUID	0x0005
#define L2CAP_UUID	0x0100
#define IP_UUID		0x0009
#define UDP_UUID	0x0002
#define TCP_UUID	0x0004
#define TCS_BIN_UUID	0x0005
#define TCS_AT_UUID	0x0006
#define OBEX_UUID	0x0008
#define FTP_UUID	0x000A
#define HTTP_UUID	0x000C
#define WSP_UUID	0x000E
#define BNEP_UUID	0x000F
#define HIDP_UUID	0x0011
#define HCRP_CTRL_UUID	0x0012
#define HCRP_DATA_UUID	0x0014
#define HCRP_NOTE_UUID	0x0016
#define AVCTP_UUID	0x0017
#define AVDTP_UUID	0x0019
#define CMTP_UUID	0x001B

/*
 * Service class identifiers of standard services and service groups
 */
#define SDP_SERVER_SVCLASS_ID		0x1000
#define BROWSE_GRP_DESC_SVCLASS_ID	0x1001
#define PUBLIC_BROWSE_GROUP		0x1002
#define SERIAL_PORT_SVCLASS_ID		0x1101
#define LAN_ACCESS_SVCLASS_ID		0x1102
#define DIALUP_NET_SVCLASS_ID		0x1103
#define IRMC_SYNC_SVCLASS_ID		0x1104
#define OBEX_OBJPUSH_SVCLASS_ID		0x1105
#define OBEX_FILETRANS_SVCLASS_ID	0x1106
#define IRMC_SYNC_CMD_SVCLASS_ID	0x1107
#define HEADSET_SVCLASS_ID		0x1108
#define CORDLESS_TELEPHONY_SVCLASS_ID	0x1109
#define AUDIO_SOURCE_SVCLASS_ID		0x110a
#define AUDIO_SINK_SVCLASS_ID		0x110b
#define INTERCOM_SVCLASS_ID		0x1110
#define FAX_SVCLASS_ID			0x1111
#define HEADSET_AGW_SVCLASS_ID		0x1112
#define PANU_SVCLASS_ID			0x1115
#define NAP_SVCLASS_ID			0x1116
#define GN_SVCLASS_ID			0x1117
#define IMAGING_SVCLASS_ID		0x111a
#define IMAGING_RESPONDER_SVCLASS_ID	0x111b
#define HANDSFREE_SVCLASS_ID		0x111e
#define HANDSFREE_AUDIO_GW_SVCLASS_ID	0x111f
#define HID_SVCLASS_ID			0x1124
#define HCR_SVCLASS_ID			0x1125
#define HCR_PRINT_SVCLASS_ID		0x1126
#define HCR_SCAN_SVCLASS_ID		0x1127
#define CIP_SVCLASS_ID			0x1128
#define SAP_SVCLASS_ID			0x112d
#define PNP_INFO_SVCLASS_ID		0x1200
#define GENERIC_NETWORKING_SVCLASS_ID	0x1201
#define GENERIC_FILETRANS_SVCLASS_ID	0x1202
#define GENERIC_AUDIO_SVCLASS_ID	0x1203
#define GENERIC_TELEPHONY_SVCLASS_ID	0x1204
#define VIDEO_SOURCE_SVCLASS_ID		0x1303
#define VIDEO_SINK_SVCLASS_ID		0x1304

/*
 * Standard profile descriptor identifiers; note these
 * may be identical to some of the service classes defined above
 */
#define SERIAL_PORT_PROFILE_ID		0x1101
#define LAN_ACCESS_PROFILE_ID		0x1102
#define DIALUP_NET_PROFILE_ID		0x1103
#define IRMC_SYNC_PROFILE_ID		0x1104
#define OBEX_OBJPUSH_PROFILE_ID		0x1105
#define OBEX_FILETRANS_PROFILE_ID	0x1106
#define IRMC_SYNC_CMD_PROFILE_ID	0x1107
#define HEADSET_PROFILE_ID		0x1108
#define CORDLESS_TELEPHONY_PROFILE_ID	0x1109
#define ADVANCED_AUDIO_PROFILE_ID	0x110d
#define INTERCOM_PROFILE_ID		0x1110
#define FAX_PROFILE_ID			0x1111
#define HEADSET_AGW_PROFILE_ID		0x1112
#define PANU_PROFILE_ID			0x1115
#define NAP_PROFILE_ID			0x1116
#define GN_PROFILE_ID			0x1117
#define IMAGING_PROFILE_ID		0x111a
#define IMAGING_RESPONDER_PROFILE_ID	0x111b
#define HANDSFREE_PROFILE_ID		0x111e
#define HID_PROFILE_ID			0x1124
#define HCR_PROFILE_ID			0x1125
#define CIP_PROFILE_ID			0x1128
#define SAP_PROFILE_ID			0x112d

/*
 * Attribute identifier codes
 */
#define SDP_SERVER_RECORD_HANDLE	0x0000

/*
 * Possible values for attribute-id are listed below. 
 * See SDP Spec, section "Service Attribute Definitions" for more details.
 */
#define SDP_ATTR_RECORD_HANDLE		0x0000
#define SDP_ATTR_SVCLASS_ID_LIST	0x0001
#define SDP_ATTR_RECORD_STATE		0x0002
#define SDP_ATTR_SERVICE_ID		0x0003
#define SDP_ATTR_PROTO_DESC_LIST	0x0004
#define SDP_ATTR_BROWSE_GRP_LIST	0x0005
#define SDP_ATTR_LANG_BASE_ATTR_ID_LIST	0x0006
#define SDP_ATTR_SVCINFO_TTL		0x0007
#define SDP_ATTR_SERVICE_AVAILABILITY	0x0008
#define SDP_ATTR_PFILE_DESC_LIST	0x0009
#define SDP_ATTR_DOC_URL		0x000A
#define SDP_ATTR_CLNT_EXEC_URL		0x000B
#define SDP_ATTR_ICON_URL		0x000C
#define SDP_ATTR_ADD_PROTO_DESC_LIST	0x000D

#define SDP_ATTR_IP_SUBNET			0x0200
#define SDP_ATTR_SERVICE_VERSION		0x0300
#define SDP_EXTERNAL_NETWORK			0x0301
#define SDP_ATTR_SUPPORTED_DATA_STORES_LIST	0x0301
#define SDP_ATTR_REMOTE_AUDIO_VOLUME_CONTROL	0x0302
#define SDP_ATTR_SUPPORTED_FORMATS_LIST		0x0303
#define SDP_ATTR_SECURITY_DESC			0x030A
#define SDP_ATTR_NET_ACCESS_TYPE		0x030B
#define SDP_ATTR_MAX_NET_ACCESSRATE		0x030C
#define SDP_ATTR_IP4_SUBNET			0x030D
#define SDP_ATTR_IP6_SUBNET			0x030E
#define SDP_SUPPORTED_FEATURES			0x0311


/*
 * These identifiers are based on the SDP spec stating that 
 * "base attribute id of the primary (universal) language must be 0x0100"
 */
#define SDP_PRIMARY_LANG_BASE 		0x0100

#define SDP_ATTR_SVCNAME_PRIMARY	0x0000 + SDP_PRIMARY_LANG_BASE
#define SDP_ATTR_SVCDESC_PRIMARY	0x0001 + SDP_PRIMARY_LANG_BASE
#define SDP_ATTR_PROVNAME_PRIMARY	0x0002 + SDP_PRIMARY_LANG_BASE

/*
 * Other languages should have their own offset; e.g.:
 * #define XXXLangBase yyyy
 * #define AttrServiceName_XXX	0x0000+XXXLangBase
 * ...
 */

/*
 * These attributes are specific to the SDP server only; i.e.,
 * can be present only in the service record of the SDP server
 */
#define SDP_ATTR_VERSION_NUM_LIST	0x0200
#define SDP_ATTR_SVCDB_STATE		0x0201
#define SDP_ATTR_GROUP_ID		0x0200

/*
 * The Data representation in SDP PDUs (pps 339, 340 of BT SDP Spec)
 * These are the exact data type+size descriptor values
 * that go into the PDU buffer.
 *
 * The datatype (leading 5bits) + size descriptor (last 3 bits)
 * is 8 bits. The size descriptor is critical to extract the
 * right number of bytes for the data value from the PDU.
 *
 * For most basic types, the datatype+size descriptor is
 * straightforward. However for constructed types and strings,
 * the size of the data is in the next "n" bytes following the
 * 8 bits (datatype+size) descriptor. Exactly what the "n" is
 * specified in the 3 bits of the data size descriptor.
 *
 * TextString and URLString can be of size 2^{8, 16, 32} bytes
 * DataSequence and DataSequenceAlternates can be of size 2^{8, 16, 32}
 * The size are computed post-facto in the API and are not known apriori
 */
#define SDP_DATA_NIL 		0x00
#define SDP_UINT8  		0x08
#define SDP_UINT16		0x09
#define SDP_UINT32		0x0A
#define SDP_UINT64		0x0B
#define SDP_UINT128		0x0C
#define SDP_INT8		0x10
#define SDP_INT16		0x11
#define SDP_INT32		0x12
#define SDP_INT64		0x13
#define SDP_INT128		0x14
#define SDP_UUID_UNSPEC		0x18
#define SDP_UUID16		0x19
#define SDP_UUID32		0x1A
#define SDP_UUID128		0x1C
#define SDP_TEXT_STR_UNSPEC	0x20
#define SDP_TEXT_STR8		0x25
#define SDP_TEXT_STR16		0x26
#define SDP_TEXT_STR32		0x27
#define SDP_BOOL		0x28
#define SDP_SEQ_UNSPEC		0x30
#define SDP_SEQ8		0x35
#define SDP_SEQ16		0x36
#define SDP_SEQ32		0x37
#define SDP_ALT_UNSPEC		0x38
#define SDP_ALT8		0x3D
#define SDP_ALT16		0x3E
#define SDP_ALT32		0x3F
#define SDP_URL_STR_UNSPEC	0x40
#define SDP_URL_STR8		0x45
#define SDP_URL_STR16		0x46
#define SDP_URL_STR32		0x47

/*
 * The PDU identifiers of SDP packets between client and server
 */
#define SDP_ERROR_RSP		0x01
#define SDP_SVC_SEARCH_REQ	0x02
#define SDP_SVC_SEARCH_RSP	0x03
#define SDP_SVC_ATTR_REQ	0x04
#define SDP_SVC_ATTR_RSP	0x05
#define SDP_SVC_SEARCH_ATTR_REQ	0x06
#define SDP_SVC_SEARCH_ATTR_RSP	0x07

/*
 * Some additions to support service registration.
 * These are outside the scope of the Bluetooth specification
 */
#define SDP_SVC_REGISTER_REQ	0x75
#define SDP_SVC_REGISTER_RSP	0x76
#define SDP_SVC_UPDATE_REQ	0x77
#define SDP_SVC_UPDATE_RSP	0x78
#define SDP_SVC_REMOVE_REQ	0x79
#define SDP_SVC_REMOVE_RSP	0x80

/*
 * Values of the flags parameter to sdp_record_register
 */
#define SDP_RECORD_PERSIST	0x01

/*
 * Values of the flags parameter to sdp_connect
 */
#define SDP_RETRY_IF_BUSY	0x01

/*
 * SDP Error codes
 */
#define SDP_INVALID_VERSION		0x0001
#define SDP_INVALID_RECORD_HANDLE	0x0002
#define SDP_INVALID_SYNTAX		0x0003
#define SDP_INVALID_PDU_SIZE		0x0004
#define SDP_INVALID_CSTATE		0x0005

/*
 * SDP PDU
 */
typedef struct {
	uint8_t  pdu_id;
	uint16_t tid;
	uint16_t plen;
} __attribute__ ((packed)) sdp_pdu_hdr_t;

/*
 * Common definitions for attributes in the SDP.
 * Should the type of any of these change, you need only make a change here. 
 */
typedef struct {
	char data[16];
} uint128_t;

typedef struct {
	uint8_t type;
	union {
		uint16_t  uuid16;
		uint32_t  uuid32;
		uint128_t uuid128;
	} value;
} uuid_t;

#define SDP_IS_UUID(x) ((x) == SDP_UUID16 || (x) == SDP_UUID32 || (x) ==SDP_UUID128)

typedef struct _sdp_list sdp_list_t;
struct _sdp_list {
	sdp_list_t *next;
	void *data;
};

/*
 * User-visible strings can be in many languages
 * in addition to the universal language.
 *
 * Language meta-data includes language code in ISO639 
 * followed by the encoding format. The third field in this 
 * structure is the attribute offset for the language. 
 * User-visible strings in the specified language can be 
 * obtained at this offset.
 */
typedef struct {
	uint16_t code_ISO639;
	uint16_t encoding;
	uint16_t base_offset;
} sdp_lang_attr_t;

/*
 * Profile descriptor is the Bluetooth profile metadata. If a 
 * service conforms to a well-known profile, then its profile 
 * identifier (UUID) is an attribute of the service. In addition,
 * if the profile has a version number it is specified here.
 */
typedef struct {
	uuid_t uuid;
	uint16_t version;
} sdp_profile_desc_t;

typedef struct {
	uint8_t major;
	uint8_t minor;
} sdp_version_t;

typedef struct {
	char *data;
	int data_size;
	int buf_size;
} sdp_buf_t;

typedef struct {
	uint32_t handle;

	/*
	 * Search pattern: a sequence of all UUIDs seen in this record
	 */
	sdp_list_t *pattern;
	sdp_list_t *attrlist;
} sdp_record_t;

typedef struct sdp_data_struct sdp_data_t;
struct sdp_data_struct {
	uint8_t dtd;
	uint16_t attrId;
	union {
		int8_t    int8;
		int16_t   int16;
		int32_t   int32;
		int64_t   int64;
		uint128_t int128;
		uint8_t   uint8;
		uint16_t  uint16;
		uint32_t  uint32;
		uint64_t  uint64;
		uint128_t uint128;
		uuid_t    uuid;
		char     *str;
		sdp_data_t *dataseq;
	} val;
	sdp_data_t *next;
	int unitSize;
};

/*
 * a session with an SDP server
 */
typedef struct {
	int      sock;
	int      state;
	int      local;
	int      flags;
	uint16_t tid;   // Current transaction ID
} sdp_session_t;

#ifdef __cplusplus
}
#endif

#endif /* __SDP_H */
